function [endmembers,coordenates] = ILSIA(data,theta,visualize)
%% [endmembers,coordenates] = ILSIA(data,theta,visualize)
% 
% Manuel Grana <manuel.grana[AT]ehu.es>
% Miguel Angel Veganzones <miguelangel.veganzones[AT]ehu.es>
% Grupo de Inteligencia Computacional (GIC), Universidad del Pais Vasco /
% Euskal Herriko Unibertsitatea (UPV/EHU)
% http://www.ehu.es/inteligenciacomputacional
% 
% The Incremental lattice Source Induction Algorithm (ILSIA) is an
% Endmember Induction Algorithm (EIA) based on Lattice Computing.
% This implementation is suited for the analysis of hyperspectral images.
% See reference:
%       Manuel Grana, Darya Chyzhyk, Maite Garcia-Sebastian, Carmen
%       Hernandez. "lattice Independent Component Analysis for functional
%       Magnetic Resonance Imaging" (Draft). Information Science (acepted).
%       2010.
%
% Parameters:
%   - data: hyperspectral image in the form of a cube, a [m n p] matrix,
%   where m and n are the spatial dimensions (rows and columns), and p is
%   the spectral dimenionality (bands).
%   - theta: Chebyshev-best approximation tolerance threshold
%   - visualize: indicates if the results must be visualized in a figure
%   (visualize > 0) or not (visualize <= 0)
%
% The algorithm retrieves the set of induced endmembers and
% the coordenates of the pixels selected as endmembers.

%% Check parameters
if (nargin < 1)
    error('Insufficient parameters');
end
if (nargin < 2)
    theta = 1;
    visualize = false;
end

[height, width, nbands] = size(data);
if (height == 0 || width == 0 || nbands == 0)
    error('Missmatch data dimensionality');
end

tic

%% Reshape the cube and transform to Z values
data_Z = reshape(data,height*width,nbands);
% Normalize
%%%%%%%%%%%%%%%%%%%%%%%%%%%% E[data]
mean_data = mean(data_Z,1);
std_data = zeros(1,nbands);
for i=1:nbands
   std_data(i) = std(data_Z(:,i));
end
for i=1:width*height
   data_Z(i,:) = (data_Z(i,:) - mean_data) ./ std_data;
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%% E[x_i]
% mean_data = mean(data_Z,2);
% for i=1:nbands
%     data_Z(:,i) = data_Z(:,i) - mean_data;
% end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Initialization
LIS = [];
endmembers = [];
coordenates = [];
%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Pick a random pixel
%pixel = floor(rand*height*width) + 1;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Pick minimum norm pixel
[value,pixel] = min(sum(power(data_Z,2),2));
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
new_LIS = true;
nendmembers = 1;
coordenates(nendmembers,1) = floor(pixel/width) + 1; 
coordenates(nendmembers,2) = mod(pixel,width); 
if coordenates(nendmembers,2) == 0 
    coordenates(nendmembers,2) = width;
end
LIS(1:nbands,nendmembers) = data_Z(pixel,:);
endmembers(1:nbands,nendmembers) = data(coordenates(nendmembers,1),coordenates(nendmembers,2),:);

%% Algorithm
% Run over each pixel
for pixel=1:height*width
    % Check for LAAM recalculation
    if (new_LIS)
        % Recalculate LAAM
        Wxx = lam(LIS'); 
    end
    new_LIS = false;
    % Pixel spectra
    f = data_Z(pixel,:);
    if sum(abs(f)) > 0
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% A
        % Check if the pixel is lattice dependent
        y = zeros(1,nbands);
        for i=1:nbands
            y(1,i) = max(Wxx(i,:) + f);
        end
        if (f == y) 
            break;
        end
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% B
        % Best-Chebyshev approximation
        if (theta > 0)
            x_sharp = zeros(1,nbands);
            Wxx_transposed = Wxx';
            for i=1:nbands
                x_sharp(1,i) = min(Wxx_transposed(i,:) + f);
                mu = max(Wxx(i,:) + x_sharp);
                mu = max(mu + f)/2;
            end
            c1 = zeros(1,nbands);
            for i=1:nbands
                c1(1,i) = max(Wxx(i,:) + mu + x_sharp);
            end
            if (chebyshev(c1,f) < theta)
                break;
            end
        end
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% C
        % Max-Min dominance
        mu1 = 0; mu2 = 0;
        for i = 1:nendmembers+1
            s1 = zeros(nbands,1);
            s2 = zeros(nbands,1);
            for k= 1:nendmembers+1
                if i~=k
                    if (i == (nendmembers+1))
                        vi = f';
                    else
                        vi = LIS(:,i);
                    end
                    if (k == (nendmembers+1))
                        vk = f';
                    else
                        vk = LIS(:,k);
                    end
                    d = vi - vk;
                    m1 = max(d);
                    m2 = min(d);
                    s1 = s1 + (d == m1);
                    s2 = s2 + (d == m2);
                end
            end
            mu1 = mu1 + (max(s1) == nendmembers);
            mu2 = mu2 + (max(s2) == nendmembers);
        end
        if (mu1 == (nendmembers+1) || mu2 == (nendmembers+1))
            % new LIS
            new_LIS = true;
            nendmembers = nendmembers + 1
            coordenates(nendmembers,1) = floor(pixel/width) + 1; 
            coordenates(nendmembers,2) = mod(pixel,width); 
            if coordenates(nendmembers,2) == 0 
                coordenates(nendmembers,2) = width;
            end
            LIS(1:nbands,nendmembers) = data_Z(pixel,:);
            endmembers(1:nbands,nendmembers) = data(coordenates(nendmembers,1),coordenates(nendmembers,2),:);
        end
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    end %% if norm(pixel) > 0
end %% for each pixel

fprintf('Endmembers induced in %g seconds\n', toc);

%% Visualize the results
if (visualize)
    figure;
    imagesc(mean(data,3)); colormap(gray);
    for i = 1:size(coordenates,1)
        hold on;
        plot(coordenates(i,2), coordenates(i,1), 'yo');
    end
end
