function [W, M, products] = lam(X,Y)
%% function [W, M] = lam(X,Y)
% 
% Manuel Grana <manuel.grana[AT]ehu.es>
% Miguel Angel Veganzones <miguelangel.veganzones[AT]ehu.es>
% Grupo de Inteligencia Computacional (GIC), Universidad del Pais Vasco /
% Euskal Herriko Unibertsitatea (UPV/EHU)
% http://www.ehu.es/inteligenciacomputacional
% 
% The Lattice Associative Memories is a kind of associative memory working
% over lattice operations. If X=Y we are talking of Lattice AutoAssociative
% Memories (LAAM), otherwise we are talking of Lattice HeteroAssociative
% Memories (LHAM).
% See reference:
%       Manuel Grana, Darya Chyzhyk, Maite Garcia-Sebastian, Carmen
%       Hernandez. "lattice Independent Component Analysis for functional
%       Magnetic Resonance Imaging" (Draft). Information Science (acepted).
%       2010.
%
% Parameters:
%   - X: matrix of input patterns. X must have dimensions [k n] where k is
%   the number of input vectors, and n is the dimensionality of each input
%   vector.
%   - Y: matrix of output patterns. Y must have dimensions [k m] where k is
%   the number of output vectors, and m is the dimensionality of each output
%   vector.
% If only X is passed as a parameter, we consider the autoasssociative case Y=X.
%
% The algorithm retrieves the dilative (W) and erosive (M) LAMs.

%% Check parameters
if (nargin < 1)
    error('Insufficient parameters');
end
if (nargin < 2)
    Y = X;
end
[kx n] = size(X);
[ky m] = size(Y);
if (kx ~= ky) || kx == 0
    error('Input and output dimensions missmatch or zero dimensionality');
end
k = kx;
if (n == 0)
    error('Zero dimensionality for X columns');
end
if (m == 0)
    error('Zero dimensionality for Y columns');
end

%% Compute vector products
products = zeros(k,m,n);
for a=1:k
    for b = 1:m
        for c = 1:n
            products(a,b,c) = Y(a,b) - X(a,c);
        end
    end
end

%% Compute LAMs
W = squeeze(min(products,[],1));
M = squeeze(max(products,[],1));
