/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.ListIterator;

import es.ehu.www.ccwintco.npp.core.SinglePointNppEstimator;
import es.ehu.www.ccwintco.npp.core.SinglePointNppResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.AbstractSiteDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.ClimateDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.DailyDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SeasonDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SiteDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.DailyDataManager.DailyData;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SeasonDataManager.SeasonData;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Implementation of the ForestBGC ecophysiological method for a single point using climate data and specific-species data from the study site.
 * Based on the Joseph C. Coughlan's SIMLAT5 version 3.6 implementation of the Forest-BGC model. Copyright 1988 J.C. Coughlan.
 * Bibliographical references:
 * <ul>
 * <li>S.W. Running and J.C. Coughlan. A general model of forest ecosystem processes for regional applications. I. Hydrologic balance, canopy gas exchange and primary production processes. Ecological Modelling, 42(2):125-154, 1988.</li>
 * <li>S.W. Running and S.T. Gower. Forest-BGC, a general model of forest ecosystem processes for regional applications. II. Dynamic carbon allocation and nitrogen budgets. Tree Physiology, 9(1-2):147-160, 1991.</li>
 * </ul>
 * 
 */
public final class ForestBGC implements SinglePointNppEstimator {

	/** Name of he simulation */
	private String name = null;
	
	/**
	 * @return the name
	 */
	public final String getName() {
		return name;
	}

	/**
	 * @param name the name to set
	 */
	public final void setName(String name) {
		this.name = name;
	}

	// C/N models
	/** Running's 1988 C/N model (DC model) */
	public static final int CNMODEL_RUNNING_88 = 0;

	/** Gower's 1991 C/N model (BGC model) */
	public static final int CNMODEL_GOWER_91 = 1;

	/** C/N Model (Running's 88 is the default) */
	private int cnModel = ForestBGC.CNMODEL_RUNNING_88;
	
	/**
	 * @return the cnModel
	 */
	public final int getCnModel() {
		return cnModel;
	}

	/**
	 * @param cnModel the cnModel to set
	 */
	public final void setCnModel(int cnModel) {
		this.cnModel = cnModel;
	}

	/** Climate data manager */
	private ClimateDataManager climateDataManager = null;
	
	/** Site data manager */
	private SiteDataManager siteDataManager = null;
	
	/** Daily site data manager */
	private DailyDataManager dailyDataManager = null;
	
	/** Seasonally site data manager */
	private SeasonDataManager seasonDataManager = null;

	/**
	 * @return the climateDataManager
	 */
	public final ClimateDataManager getClimateDataManager() {
		return climateDataManager;
	}

	/**
	 * @param climateDataManager the climateDataManager to set
	 */
	public final void setClimateDataManager(ClimateDataManager climateDataManager) {
		this.climateDataManager = climateDataManager;
	}

	/**
	 * @return the siteDataManager
	 */
	public final SiteDataManager getSiteDataManager() {
		return siteDataManager;
	}

	/**
	 * @param siteDataManager the siteDataManager to set
	 */
	public final void setSiteDataManager(SiteDataManager siteDataManager) {
		this.siteDataManager = siteDataManager;
	}

	/**
	 * @return the dailyDataManager
	 */
	public final DailyDataManager getDailyDataManager() {
		return dailyDataManager;
	}

	/**
	 * @param dailyDataManager the dailyDataManager to set
	 */
	public final void setDailyDataManager(DailyDataManager dailyDataManager) {
		this.dailyDataManager = dailyDataManager;
	}

	/**
	 * @return the seasonDatamanager
	 */
	public final SeasonDataManager getSeasonDataManager() {
		return seasonDataManager;
	}

	/**
	 * @param seasonDatamanager the seasonDatamanager to set
	 */
	public final void setSeasonDataManager(SeasonDataManager seasonDataManager) {
		this.seasonDataManager = seasonDataManager;
	}

	/** Singleton pattern */
	private final static ForestBGC instance = new ForestBGC();
	
	/** 
	 * Return the singleton instance of ForestBGC class.
	 * @return ForestBGC instance.
	 */
	public static ForestBGC getInstance() {
		return instance;
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.SinglePointNppEstimator#getVersion()
	 */
	@Override
	public String getVersion() {
		return "Forest-BGC model implementation based on the Joseph C. Coughlan's SIMLAT5 version 3.6.";
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.SinglePointNppEstimator#estimate()
	 */
	public SinglePointNppResults estimate() throws Exception {
		// Check input data
		if (climateDataManager == null) throw new Exception("A climate data manager is needed");
		if (siteDataManager == null) throw new Exception("A site data manager is needed");
		// Results
		ArrayList<DailyResults> dailyResults = new ArrayList<DailyResults>();
		ArrayList<CycleResults> cycleResults = new ArrayList<CycleResults>();
		// Model
		try {
			// Model cycles time data
			int startTime = siteDataManager.getData(AbstractSiteDataManager.START_TIME).intValue();
			int stopTime = siteDataManager.getData(AbstractSiteDataManager.STOP_TIME).intValue();
			int loopTime = siteDataManager.getData(AbstractSiteDataManager.LOOP_TIME).intValue();
			// State variables
			Double[] x = new Double[20];
			x[0] = siteDataManager.getData(AbstractSiteDataManager.X_1);
			x[1] = siteDataManager.getData(AbstractSiteDataManager.X_2);
			x[2] = siteDataManager.getData(AbstractSiteDataManager.X_3);
			x[3] = siteDataManager.getData(AbstractSiteDataManager.X_4);
			x[4] = siteDataManager.getData(AbstractSiteDataManager.X_5);
			x[5] = siteDataManager.getData(AbstractSiteDataManager.X_6);
			x[6] = siteDataManager.getData(AbstractSiteDataManager.X_7);
			x[7] = siteDataManager.getData(AbstractSiteDataManager.X_8);
			x[8] = siteDataManager.getData(AbstractSiteDataManager.X_9);
			x[9] = siteDataManager.getData(AbstractSiteDataManager.X_10);
			x[10] = siteDataManager.getData(AbstractSiteDataManager.X_11);
			x[11] = siteDataManager.getData(AbstractSiteDataManager.X_12);
			x[12] = siteDataManager.getData(AbstractSiteDataManager.X_13);
			x[13] = siteDataManager.getData(AbstractSiteDataManager.X_14);
			x[14] = siteDataManager.getData(AbstractSiteDataManager.X_15);
			x[15] = siteDataManager.getData(AbstractSiteDataManager.X_16);
			x[16] = siteDataManager.getData(AbstractSiteDataManager.X_17);
			x[17] = siteDataManager.getData(AbstractSiteDataManager.X_18);
			x[18] = siteDataManager.getData(AbstractSiteDataManager.X_19);
			x[19] = siteDataManager.getData(AbstractSiteDataManager.X_20);
			// State flows
			Double[] dx = new Double[20];
			for (int i=0; i<dx.length; i++) dx[i] = 0.0;
			// Flux
			Double[][] f = new Double[20][20];
			for (int i=0; i<20; i++) {
				for (int j=0; j<20; j++) f[i][j] = 0.0;
			}
			// Site constants
			Double[] b = new Double[50];
			for (int i=0; i<b.length; i++) b[i] = 0.0;
			b[0] = siteDataManager.getData(AbstractSiteDataManager.B_1);
			b[1] = siteDataManager.getData(AbstractSiteDataManager.B_2);
			b[2] = siteDataManager.getData(AbstractSiteDataManager.B_3);
			b[3] = siteDataManager.getData(AbstractSiteDataManager.B_4);
			b[4] = siteDataManager.getData(AbstractSiteDataManager.B_5);
			b[5] = siteDataManager.getData(AbstractSiteDataManager.B_6);
			b[6] = siteDataManager.getData(AbstractSiteDataManager.B_7);
			b[7] = siteDataManager.getData(AbstractSiteDataManager.B_8);
			b[8] = siteDataManager.getData(AbstractSiteDataManager.B_9);
			b[9] = siteDataManager.getData(AbstractSiteDataManager.B_10);
			b[10] = siteDataManager.getData(AbstractSiteDataManager.B_11);
			b[11] = siteDataManager.getData(AbstractSiteDataManager.B_12);
			b[12] = siteDataManager.getData(AbstractSiteDataManager.B_13);
			b[13] = siteDataManager.getData(AbstractSiteDataManager.B_14);
			b[14] = siteDataManager.getData(AbstractSiteDataManager.B_15);
			b[15] = siteDataManager.getData(AbstractSiteDataManager.B_16);
			b[16] = siteDataManager.getData(AbstractSiteDataManager.B_17);
			b[17] = siteDataManager.getData(AbstractSiteDataManager.B_18);
			b[18] = siteDataManager.getData(AbstractSiteDataManager.B_19);
			b[19] = siteDataManager.getData(AbstractSiteDataManager.B_20);
			b[20] = siteDataManager.getData(AbstractSiteDataManager.B_21);
			b[21] = siteDataManager.getData(AbstractSiteDataManager.B_22);
			b[22] = siteDataManager.getData(AbstractSiteDataManager.B_23);
			b[23] = siteDataManager.getData(AbstractSiteDataManager.B_24);
			b[24] = siteDataManager.getData(AbstractSiteDataManager.B_25);
			b[25] = siteDataManager.getData(AbstractSiteDataManager.B_26);
			b[26] = siteDataManager.getData(AbstractSiteDataManager.B_27);
			b[27] = siteDataManager.getData(AbstractSiteDataManager.B_28);
			b[28] = siteDataManager.getData(AbstractSiteDataManager.B_29);
			b[29] = siteDataManager.getData(AbstractSiteDataManager.B_30);
			b[30] = siteDataManager.getData(AbstractSiteDataManager.B_31);
			b[31] = siteDataManager.getData(AbstractSiteDataManager.B_32);
			b[32] = siteDataManager.getData(AbstractSiteDataManager.B_33);
			b[33] = siteDataManager.getData(AbstractSiteDataManager.B_34);
			b[34] = siteDataManager.getData(AbstractSiteDataManager.B_35);
			b[35] = siteDataManager.getData(AbstractSiteDataManager.B_36);
			b[36] = siteDataManager.getData(AbstractSiteDataManager.B_37);
			b[37] = siteDataManager.getData(AbstractSiteDataManager.B_38);
			b[38] = siteDataManager.getData(AbstractSiteDataManager.B_39);
			b[39] = siteDataManager.getData(AbstractSiteDataManager.B_40);
			b[40] = siteDataManager.getData(AbstractSiteDataManager.B_41);
			b[41] = siteDataManager.getData(AbstractSiteDataManager.B_42);
			b[42] = siteDataManager.getData(AbstractSiteDataManager.B_43);
			b[43] = siteDataManager.getData(AbstractSiteDataManager.B_44);
			b[44] = siteDataManager.getData(AbstractSiteDataManager.B_45);
			b[45] = siteDataManager.getData(AbstractSiteDataManager.B_46);
			b[46] = siteDataManager.getData(AbstractSiteDataManager.B_47);
			b[47] = siteDataManager.getData(AbstractSiteDataManager.B_48);
			b[48] = siteDataManager.getData(AbstractSiteDataManager.B_49);
			b[49] = siteDataManager.getData(AbstractSiteDataManager.B_50);
			// Climate parameters
			Double[] z = new Double[21];
			for (int i=0; i<z.length; i++) z[i] = 0.0;
			// Initialize intermediate G variables
			Double[] g = new Double[110];
			for (int i=0; i<g.length; i++) g[i] = 0.0;
			// Season
			int season = 1;
			// yearday loop
			for (int yearday = startTime; yearday <= loopTime; yearday ++) {
				// Ontogeny
				if (dailyDataManager != null) {
					List<DailyData> dailyDataList = dailyDataManager.getDailyData(yearday);
					if (dailyDataList != null) {
						ListIterator<DailyData> dailyDataListIterator = dailyDataList.listIterator();
						while (dailyDataListIterator.hasNext()) {
							DailyData dailyData = dailyDataListIterator.next();
							siteDataManager.setData(dailyData.getConstant(),dailyData.getValue());
						}
					}
				}
				// Philogeny
				if (seasonDataManager != null) {
					List<SeasonData> seasonDataList = seasonDataManager.getSeasonData(season);
					if (seasonDataList != null) {
						ListIterator<SeasonData> seasonDataListIterator = seasonDataList.listIterator();
						while (seasonDataListIterator.hasNext()) {
							SeasonData seasonData = seasonDataListIterator.next();
							siteDataManager.setData(seasonData.getConstant(),seasonData.getValue());
						}
					}
				}
				// Reset yearday G variables
				for (int i = 0; i < 89; i++) g[i] = 0.0;
				// Model
				switch (cnModel) {
					case ForestBGC.CNMODEL_GOWER_91:
						// climate changes
						z = climateBGC(yearday,x,b);
						// water model
						g = h2oBGC(x,b,z,g);
						// Carbon flows
						f[5][5] = g[25];
						f[10][11] = g[101];
						f[10][12] = 0.0;
						f[12][11] = g[103];
						break;
					default: // ForestBGC.CNMODEL_RUNNING_88: 
						// climate changes
						z = climateDC(yearday,x,b);
						// water model
						g = h2oDC(x,b,z,g);
						// Carbon flows
						f[5][5] = g[25];
						f[5][6] = f[5][6] + g[29] + g[30];
						break;
				}
				// Water flows
				f[1][1] = g[0] + g[5];
				f[0][0] = g[1];
				f[4][4] = g[3];
				f[0][1] = g[2];
				f[1][2] = g[17];
				f[1][3] = g[16];
				// allocate
				if (yearday % stopTime == 0) {
					// grow
					switch (cnModel) {
						case ForestBGC.CNMODEL_GOWER_91:
							// substract annual maintenance respiration
							x[5] = x[5] - g[99];
							// C/N submodel
							g = growBGC(x,b,g);
							// water flows
							f[3][3] = -x[3];
							// litterfall carbon flows
							f[7][10] = (x[7] / g[43]) * (1.0 - b[32]);
							f[7][12] = x[7] / g[43] * b[32];
							f[8][12] = x[8] * b[40];
							f[9][10] = x[9] * b[41];
							// Carbon growth flows
							f[5][6] = g[99] + g[65];
							f[5][7] = g[66];
							f[5][8] = g[67];
							f[5][9] = g[68];
							// Nitrogen flows
							f[13][13] = b[38] + b[39]; // N inputs, atmospheric decomposition, biological fixation
							// Nitrogen allocation
							f[13][14] = g[69];
							f[13][15] = g[70];
							f[13][16] = g[71];
							// leaf N retranslocation
							f[14][13] = g[72];
							// Litterfall N
							f[14][17] = (x[14] / g[43]) * (1.0 - g[42]); // leaf
							f[15][18] = 0.0; // stem
							f[16][13] = g[71]; // root
							// decomposition flows
							f[17][13] = g[93];
							f[17][18] = x[17] * g[79] * g[80] * b[32];
							f[17][19] = 0.0;
							f[18][13] = g[94];
							f[18][19] = 0.0;
							// N leaching loss
							f[13][19] = x[13] / b[37];
							break;
						default: // ForestBGC.CNMODEL_RUNNING_88:
							// C/N submodel
							g = growDC(x,b,g);
							// growth flows
							f[5][6] = f[5][6] + g[53] + g[54] + g[55];
							f[5][7] = g[56];
							f[5][8] = g[57];
							f[5][9] = g[58];
							f[7][10] = g[56];
							f[8][11] = g[60];
							f[9][12] = g[61];
							break;
					}
					// net primary production
//					npp = x[5] - g[99];
					// reset allocation G variables
					for (int i=89; i < 110; i++) g[i] = 0.0;
					// Increase season
					season++;
				}
				// flows
				for (int i = 0; i < x.length; i++) { // initialize
					dx[i] = f[i][i];
				}
				for (int i = 0; i < (x.length - 1); i++) {
					for (int j = i + 1; j < x.length; j++) {
						double part = f[j][i] - f[i][j];
						dx[i] = dx[i] + part;
						dx[j] = dx[j] - part;
					}
				}
				for (int i = 0; i < x.length; i++) { // reset flows
					for (int j = 0; j < x.length; j++) {
						f[i][j] = 0.0;
					}
				}
				// Daily Results
				DailyResults results = new DailyResults(x,dx,f,g);
				dailyResults.add(results);
				// Cycle results
				if (yearday % stopTime == 0) cycleResults.add(new CycleResults(x,dx,f,g));
				// update state variables
				for (int i = 0; i < x.length; i++) x[i] = x[i] + dx[i];
			}
			return new ForestBGCResults(name, new Date(System.currentTimeMillis()), this, dailyResults, cycleResults);
		}
		catch(Exception e) {throw e;}
	}
	
	/**
	 * Changes in climatic data, following the DC model by Running '88.
	 * @param yearday
	 * @param x state variables
	 * @param b site constants
	 * @return the climatic data
	 */
	private final Double[] climateDC(int yearday, Double[] x, Double[] b) throws Exception {
		try {
			Double xTMax = climateDataManager.getData(ClimateDataManager.T_MAX, yearday);
			Double xTMin = climateDataManager.getData(ClimateDataManager.T_MIN, yearday);
			Double dewPT = climateDataManager.getData(ClimateDataManager.DEW_POINT, yearday);
			Double xRad = climateDataManager.getData(ClimateDataManager.DAY_LENGTH, yearday);
			Double xPPT = climateDataManager.getData(ClimateDataManager.PRECISION, yearday);
			Double[] z = new Double[21];
			z[0] = 86.0;
			z[1] = new Double(yearday);
			z[2] = xPPT/1000; // PPT to meters
			z[3] = xTMax;
			z[4] = xTMin;
			Double tave = (z[3] + z[4])/2.0;
			// Compute relative humidity from min temp or dew point
			z[5] = (6.1078 * Math.exp((17.269 * dewPT)/(237.3 + dewPT))) / (6.1078 * Math.exp((17.269 * tave)/(237.3 + tave))) * 100.0;
			// Leaf area index all sides
			z[7] = xRad * b[7];
			z[9] = (b[0] * x[7])/b[4];
			// Average temperatures: soil, air, daylight & night
			if (x[0] > 0.0) z[6] = 0.0; // If snowpack exists, soil temp defined as 0 deg
			else z[6] = tave;
			z[13] = 0.212 * (z[3] - tave) + tave;
			z[14] = (z[13] + z[4])/2;
			// Compute absolute humidity deficit
			double esd = 6.1078 * Math.exp((17.269 * z[13])/(237.3 + z[13]));
			double es = z[5] / 100 * esd;
			z[15] = Math.max(esd - es, 0.0);
			z[16] = 217E-6 * z[15] / (z[13] + 273.16);
			// Compute daylight in seconds
			int xd = yearday - 79;
			if (xd < 0) xd = 286 + yearday;
			double ampl = Math.exp(7.42 + 0.045 * b[6]) / 3600;
			double day = ampl * Math.sin(xd * 0.01721) + 12;
			z[17] = day * 3600 * 0.85;					// Canopy average radiation
			if (z[9] < 1.0) z[18] = z[7]; // If LAI is < 1 then full canopy light
			else z[18] = (z[7] * (1 - Math.exp((z[9] / 2.2) * b[1]))) / ((-b[1]) * z[9] / 2.2);
			// Leaf nitrogen
			z[20] = b[25];
			// Return climate data
			return z;
		}
		catch (Exception e) {throw e;}
	}
	
	/**
	 * Changes in climatic data, following the BGC model by Gower '91.
	 * @param yearday
	 * @param x state variables
	 * @param b site constants
	 * @return the climatic data
	 */
	private final Double[] climateBGC(int yearday, Double[] x, Double[] b)  throws Exception {
		try {
			Double xTMax = climateDataManager.getData(ClimateDataManager.T_MAX, yearday);
			Double xTMin = climateDataManager.getData(ClimateDataManager.T_MIN, yearday);
			Double dewPT = climateDataManager.getData(ClimateDataManager.DEW_POINT, yearday);
			Double xRad = climateDataManager.getData(ClimateDataManager.DAY_LENGTH, yearday);
			Double xPPT = climateDataManager.getData(ClimateDataManager.PRECISION, yearday);
			Double[] z = new Double[21];
			z[0] = 1987.0;
			z[1] = new Double(yearday);
			z[2] = xPPT/1000; // PPT to meters
			z[3] = xTMax;
			z[4] = xTMin;
			Double tave = (z[3] + z[4])/2.0;
			// Compute relative humidity from min temp or dew point
			z[5] = dewPT;
			// Leaf area index all sides
			z[7] = xRad * b[7];
			z[9] = (b[0] * x[7])/b[4];
			// Phenology triggers for LAI with yearday
			if ((z[1] < b[35]) || (z[1] > b[36])) z[9] = 0.1;
			// Average temperatures: soil, air, daylight & night
			if (x[0] > 0.0) z[6] = 0.0; // If snowpack exists, soil temp defined as 0 deg
			else z[6] = tave;
			z[13] = 0.212 * (z[3] - tave) + tave;
			z[14] = (z[13] + z[4])/2;
			// Compute absolute humidity deficit
			double esd = 6.1078 * Math.exp((17.269 * z[13])/(237.3 + z[13]));
			double es = z[5] / 100 * esd;
			z[15] = Math.max(esd - es, 0.0);
			z[16] = 217E-6 * z[15] / (z[13] + 273.16);
			// Compute daylight in seconds
			int xd = yearday - 79;
			if (xd < 0) xd = 286 + yearday;
			double ampl = Math.exp(7.42 + 0.045 * b[6]) / 3600;
			double day = ampl * Math.sin(xd * 0.01721) + 12;
			z[17] = day * 3600 * 0.85;					// Canopy average radiation
			if (z[9] < 1.0) z[18] = z[7]; // If LAI is < 1 then full canopy light
			else z[18] = (z[7] * (1 - Math.exp((z[9] / 2.2) * b[1]))) / ((-b[1]) * z[9] / 2.2);
			// Leaf lignin
			z[19] = b[32];
			// Leaf nitrogen
			double maxla = b[30] * b[4] / b[0];
			double nnmax = x[14] / (maxla * b[25]);
			z[20] = (b[25] - b[26]) * nnmax + b[26];
			// Return climate data
			return z;
		}
		catch (Exception e) {throw e;}
	}

	/**
	 * Computes:
	 * 	- Daily hidrologic balance
	 * 	- Average daily canopy leaf (H2O) conductance
	 * 	- Average canopy mesophyll (CO2) conductance
	 * 	- Daily photosynthetic production
	 * 	- Daily maintenance respiration losses
	 * following the DC model by Running '88
	 * @param x state variables
	 * @param b site constants
	 * @param z climatic variables
	 * @param g G parameters
	 * @return the new g values
	 */
	private Double[] h2oDC(Double[] x, Double[] b, Double[] z, Double[] g) {
		// clone g
		Double[] dg = g.clone();
		// Snow Vs. Rain
		if (z[14] > 0.0) {
			dg[0] = (z[2] - z[9] * b[3]) * b[4];
			if (dg[0] < 0.0) dg[0] = 0.0;
			dg[1] = 0.0;
		}
		else {
			dg[1] = z[2] * b[4];
			if (dg[1] < 0.0) dg[1] = 0.0;
			dg[0] = 0.0;
		}
		// Snowmelt
		dg[2] = Math.max(b[5] * z[13] * b[4], 0.0);
		if ((x[0] - dg[2]) < 0.0) dg[2] = x[0];
		// LAI potential evaporation
		dg[3] = (z[2] * b[4]) - dg[0] - dg[1];
		// Potential evaporation with radiation limit
		dg[4] = (z[7]/2.5E+6) * b[4];
		if (dg[3] < 0.0) dg[3] = 0.0;
		if (dg[3] > dg[4]) {
			dg[5] = dg[3] - dg[4];
			dg[3] = dg[4];
		}
		else dg[5] = 0.0;
		// cold soil temperature root res. reduction
		dg[9] = Math.max(0.2 / (x[1]/b[2]), b[8]); // pre-dawn lwp
		if (z[6] == 0.0) dg[9] = dg[9] * 2.0;
		// pre-dawn lwp lc reduction
		dg[10] = Math.max(b[10] - (b[10] / (b[11] - b[8])) * (dg[9] - b[8]), 0.00005);
		// Night min temp lc reduction
		if (z[4] < 0.0) dg[11] = Math.max(dg[10] + 0.0002 * z[4], 0.00005);
		else dg[11] = dg[10] + (dg[10] * 0.00003 * (z[13] - 10));
		// abshd control of lc
		dg[12] = Math.max(dg[11] - (dg[11] * b[12] * (z[16] * 1.0E+6 - 4)), 0.00005);
		// radiation reduction fraction of lc reduction to 1.0 to 0.0
		dg[13] = Math.min(1/b[9] * z[18], 1.0);
		if (dg[13] == 0.0) dg[13] = 0.00000001;
		// final canopy avg. lc
		dg[14] = dg[13] * dg[12];
		// penman - monteith estimate of transpiration
		dg[15] = penmon(z,dg[14]);
		// canopy transpiration
		dg[16] = dg[15] * z[9] * b[4] * z[17];
		// ground water outflow
		dg[17] = Math.max(x[1] + dg[0] + dg[2] + dg[5] - b[2], 0.0);
		// leaf nitrogen effect on CO2 cond.
		dg[18] = 67.0 * z[20];
		// light effect on CO2 cond.
		dg[19] = (z[18] - b[13]) / (z[18] + b[14]);
		// temp. effect on CO2 cond.
		dg[20] = Math.max(b[22] * (b[17] - z[13]) * (z[13] - b[16]) / Math.pow(b[17] - b[16], 2), 0.0);
		// final mesophyll cond.
		dg[21] = b[15] * dg[18] * dg[19] * dg[20];
		// gross psn
		dg[22] = ((0.0006 - 0.00007) * (dg[14] / 1.6) * dg[21]) / ((dg[14] / 1.6) + dg[21]);
		// Daily gross psn/cm^2
		dg[23] = dg[22] * z[17] * z[9] * b[4];
		// Night respiration
		dg[24] = Math.max(b[18] * Math.exp(b[24] * z[14]) * (24.0 - (z[17] / 3600)) * x[7], 0.0);
		dg[95] = dg[95] + dg[24];
		// 24h net C/HA, 0.3 is CO2/C conversion
		dg[25] = (dg[23] - dg[24]) * 0.3;
		// Maintenence respiration of stem & root
		dg[29] = Math.max(b[19] * Math.exp(b[24] * z[6]) * Math.exp(0.67 * Math.log(x[8])), 0.0);
		dg[30] = Math.max(b[20] * Math.exp(b[24] * z[6]) * x[9], 0.0);
		// Maintenance respiration of stem and root
		dg[96] = dg[96] + dg[29];
		dg[97] = dg[97] + dg[30];
		// annual sums
		dg[99] = dg[99] + dg[29] + dg[30];
		// return
		return dg;
	}
	
	/**
	 * Computes:
	 * 	- Daily hidrologic balance
	 * 	- Average daily canopy leaf (H2O) conductance
	 * 	- Average canopy mesophyll (CO2) conductance
	 * 	- Daily photosynthetic production
	 * 	- Daily maintenance respiration losses
	 * following the BGC model by Gower '91
	 * @param x state variables
	 * @param b site constants
	 * @param z climatic variables
	 * @param g G parameters
	 * @return the new g values
	 */
	private Double[] h2oBGC(Double[] x, Double[] b, Double[] z, Double[] g) {
		// Clone g
		Double dg[] = g.clone();
		// Snow Vs. Rain
		if (z[14] > 0.0) {
			dg[0] = (z[2] - z[9] * b[3]) * b[4];
			if (dg[0] < 0.0) dg[0] = 0.0;
			dg[1] = 0.0;
		}
		else {
			dg[1] = z[2] * b[4];
			if (dg[1] < 0.0) dg[1] = 0.0;
			dg[0] = 0.0;
		}
		// Snowmelt
		dg[2] = Math.max(b[5] * z[13] * b[4], 0.0);
		if ((x[0] - dg[2]) < 0.0) dg[2] = x[0];
		// LAI potential evaporation
		dg[3] = (z[2] * b[4]) - dg[0] - dg[1];
		// Potential evaporation with radiation limit
		dg[4] = (z[7]/2.5E+6) * b[4];
		if (dg[3] < 0.0) dg[3] = 0.0;
		if (dg[3] > dg[4]) {
			dg[5] = dg[3] - dg[4];
			dg[3] = dg[4];
		}
		else dg[5] = 0.0;
		// soil water fraction
		dg[6] = x[1] / b[2];
		// soil water decomposition rate
		dg[7] = dg[6] * b[28];
		// soil water frac summation
		dg[89] = dg[89] + dg[6];
		// pre-dawn lwp
		dg[9] = Math.max(0.2 / dg[6], b[8]); 
		// cold soil temperature root res. reduction
		if (z[6] == 0.0) dg[9] = dg[9] * 2.0;
		// Max annual PMS
		dg[90] = Math.max(dg[90], dg[9]);
		// pre-dawn lwp lc reduction
		dg[10] = Math.max(b[10] - (b[10] / (b[11] - b[8])) * (dg[9] - b[8]), 0.00005);
		// Night min temp lc reduction
		if (z[4] < 0.0) dg[11] = Math.max(dg[10] + 0.0002 * z[4], 0.00005);
		else dg[11] = dg[10] + (dg[10] * 0.00003 * (z[13] - 10));
		// abshd control of lc
		dg[12] = Math.max(dg[11] - (dg[11] * b[12] * (z[16] * 1.0E+6 - 4)), 0.00005);
		// radiation reduction fraction of lc reduction to 1.0 to 0.0
		dg[13] = Math.min(1/b[9] * z[18], 1.0);
		if (dg[13] == 0.0) dg[13] = 0.00000001;
		// final canopy avg. lc
		dg[14] = dg[13] * dg[12];
		// penman - monteith estimate of transpiration
		dg[15] = penmon(z,dg[14]);
		// canopy transpiration
		dg[16] = dg[15] * z[9] * b[4] * z[17];
		// ground water outflow
		dg[17] = Math.max(x[1] + dg[0] + dg[2] + dg[5] - b[2], 0.0);
		// leaf nitrogen effect on CO2 cond.
		dg[18] = 18.2 * z[20] + 0.5;
		// light effect on CO2 cond.
		dg[19] = (z[18] - b[13]) / (z[18] + b[14]);
		// temp. effect on CO2 cond.
		dg[20] = Math.max(b[22] * (b[17] - z[13]) * (z[13] - b[16]) / Math.pow(b[17] - b[16], 2), 0.0);
		// final mesophyll cond.
		dg[21] = b[15] * dg[18] * dg[19] * dg[20];
		// gross psn
		dg[22] = ((0.0006 - 0.00007) * (dg[14] / 1.6) * dg[21]) / ((dg[14] / 1.6) + dg[21]);
		// Daily gross psn/cm^2
		dg[23] = dg[22] * z[17] * z[9] * b[4];
		// Night respiration
		dg[24] = Math.max(b[18] * Math.exp(b[24] * z[14]) * (24.0 - (z[17] / 3600)) * x[7], 0.0);
		dg[95] = dg[95] + dg[24];
		// 24h net C/HA, 0.3 is CO2/C conversion
		dg[25] = (dg[23] - dg[24]) * 0.3;
		// Maintenence respiration of stem & root
		dg[29] = Math.max(b[19] * Math.exp(b[24] * z[6]) * Math.exp(0.67 * Math.log(x[8])), 0.0);
		dg[30] = Math.max(b[20] * Math.exp(b[24] * z[6]) * x[9], 0.0);
		// litter C decomposition soil temperature and water average
		dg[100] = ((dg[7] + (z[6] / b[45])) / 2.0) * b[47];
		dg[101] = x[10] * dg[100] / 365;
		// soil temperature and water C decomposition fct
		dg[102] = dg[100] * b[46];
		dg[103] = x[12] * dg[102] / 365;
		// annual decomposition sums litter, soil C
		dg[104] = dg[101] + dg[104];
		dg[105] = dg[103] + dg[105];
		// temperature degree day summation
		dg[91] = dg[91] + z[6];
		// annual sums maintenance respiration and litter+soil decomposition
		dg[99] = dg[99] + dg[29] + dg[30];
		dg[109] = dg[109] + dg[101] + dg[103];
		// return
		return dg;
	}

	/**
	 * N/C growing submodel
	 * following the BGC model by Gower '91
	 * @param x state variables
	 * @param b site constants
	 * @param g G parameters
	 * @return the new g values
	 */
	private Double[] growBGC(Double[] x, Double[] b, Double[] g) {
		// clone g
		Double[] dg = g.clone();
		// psn
		double psn = Math.max(x[5], 0.0);
		// max la
		dg[39] = b[30] * b[4] / b[0];
		// annual soil water fract average
		dg[49] = dg[89] / 365.0;
		dg[89] = 0.0;
		// nitrogen availability index
		dg[50] = Math.min((x[13] / (dg[39] * b[25])) * b[34], 1.0);
		// leaf/root ratio
		dg[51] = (dg[49] + dg[50]) / 4.0;
		// Available canopy N*
		dg[40] = Math.min((x[13] * dg[51] * 2.0) / (dg[39] * b[25]), 1.0);
		// leaf N concentration
		dg[41] = (b[25] - b[26]) * dg[40] + b[26];
		// leaf retrans. fixed at 50%
		dg[42] = b[27];
		// fix lead turnover
		dg[43] = b[31];
		// leaf area c, n, h2o limits
		dg[52] = Math.max(psn * dg[51] * (1.0 - b[42]), 0.0);
		// N limit
		dg[53] = (1.0 / dg[41]) * x[13] * dg[51];
		// H2O limit
		dg[54] = (b[11] / dg[90]) * (x[7] / dg[43] * (1.0 + (b[42] / (1.0 - b[42]))));
		dg[90] = 0.0;
		// final leaf C
		dg[55] = Math.min(dg[54], Math.min(dg[53], dg[52]));
		// leaf C fraction of total PSN
		if (psn > 0.0) dg[56] = dg[55] / psn;
		else dg[56] = Math.min((x[7] / dg[43]) / psn, 0.0);
		// root C fraction of total PSN
		dg[57] = Math.min(dg[56] * (1 / dg[51]), 1.0 - dg[56]);
		// Stem C
		dg[58] = 1 - dg[56] - dg[57];
		// Actual carbon allocation
		dg[59] = psn * dg[56]; // leaf
		dg[60] = psn * dg[58]; // stem
		dg[61] = psn * dg[57]; // root
		// Growth respiration
		dg[62] = dg[59] * b[42]; // leaf
		dg[63] = dg[60] * b[43]; // stem
		dg[64] = dg[61] * b[44]; // root
		dg[65] = dg[62] + dg[63] + dg[64]; // total
		// Final net carbon allocation
		dg[66] = dg[59] - dg[62]; // leaf
		dg[67] = dg[60] - dg[63]; // stem
		dg[68] = dg[61] - dg[64]; // root
		// N allocation
		dg[69] = dg[66] * dg[41]; // leaf
		dg[70] = dg[67] * dg[41] * 0.05; // stem
		dg[71] = dg[68] * dg[41] * 0.5; // root
		// N leaf retranslocation
		dg[72] = (x[14] / dg[43]) * dg[42];
		// Decomposition
		dg[79] = dg[104] / x[10] * b[29];
		dg[80] = dg[105] / x[12] * b[29];
		// Final N mineralization from litter and soil
		dg[93] = x[17] * dg[79];
		dg[94] = x[18] * dg[80];
		dg[91] = 0.0;
		// return
		return dg;
	}

	/**
	 * N/C growing submodel
	 * following the DC model by Running '88
	 * @param x state variables
	 * @param b site constants
	 * @param g G parameters
	 * @return the new g values
	 */
	private Double[] growDC(Double[] x, Double[] b, Double[] g) {
		// clone g
		Double[] dg = g.clone();
		// Percent allocation
		dg[49] = b[29];
		dg[50] = b[30];
		dg[51] = b[31];
		// actual carbon allocation
		double tpsn = x[5];
		double lpsn = tpsn * dg[49];
		double spsn = tpsn * dg[50];
		double rpsn = tpsn * dg[51];
		// leaf, stem & root respiration
		dg[53] = lpsn * b[42];
		dg[54] = spsn * b[43];
		dg[55] = rpsn * b[44];
		// leaf, stem & root growth
		dg[56] = lpsn - dg[53];
		dg[57] = spsn - dg[54];
		dg[58] = rpsn - dg[55];
		// decomposition calculations
		double plig = 10.0;
		double aet = ((x[3] + x[4]) / b[4]) * 1000.0;
		dg[79] = ((-3.44 + (0.1 * aet)) - (0.0134 + (0.00147 * aet * plig))) / 100.0;
		// litter fall: carbon to litter
		dg[59] = dg[56] * b[39];
		dg[60] = dg[57] * b[40];
		dg[61] = dg[58] * b[41];
		// return
		return dg;
	}
	
	/**
	 * Penman-Montheith function
	 * @param z climatic data
	 * @param dg14 final canopy avg. lc
	 * @return penmon value
	 */
	private double penmon(Double[] z, Double dg14) {
		double rad = z[18] * 1000.0 / z[17];
		double gamma = 0.646 + 0.0006 * z[13];
		double t1 = z[13] + 0.5;
		double t2 = z[13] - 0.5;
		double svp1 = 6.1078 * Math.exp((17.269 * t1) / (237.0 + t1));
		double svp2 = 6.1078 * Math.exp((17.269 * t2) / (237.0 + t2));
		double slope = svp1 - svp2;
		double cp = 1.01E+3;
		double pa = 1.292 - 0.00428 * z[13];
		double ra = 5.0 / (z[9] / 2);
		double rs = 1.0 / dg14;
		double xlat = (2.501 - 0.0024 * z[13]) * 1.0E+6;
		double xtrans = ((slope * rad) + (cp * pa) * (z[15] / ra)) / (slope + gamma * (1.0 + rs/ra));
		return xtrans / (xlat * 1000.0);
	}

	@Override
	public String getCopyright() {
		return "Copyright 2009 M.A. Veganzones. Copyright 1988 J.C. Coughlan.";
	}

}