/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.swing.JProgressBar;

import es.ehu.www.ccwintco.npp.core.MultiplePointsNppEstimator;
import es.ehu.www.ccwintco.npp.core.MultiplePointsNppResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.AbstractSiteDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.ClimateDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.DailyDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SeasonDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SiteDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager.CoordinatedLai;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Implementation of the ForestBGC ecophysiological method for multiple points using climate data, specific-species data from the study site and a set of Leaf Area Index (LAI) values, one for each pont.
 * Based on the Joseph C. Coughlan's SIMLAT5 version 3.6 implementation of the Forest-BGC model. Copyright 1988 J.C. Coughlan.
 * Bibliographical references:
 * <ul>
 * <li>S.W. Running and J.C. Coughlan. A general model of forest ecosystem processes for regional applications. I. Hydrologic balance, canopy gas exchange and primary production processes. Ecological Modelling, 42(2):125-154, 1988.</li>
 * <li>S.W. Running and S.T. Gower. Forest-BGC, a general model of forest ecosystem processes for regional applications. II. Dynamic carbon allocation and nitrogen budgets. Tree Physiology, 9(1-2):147-160, 1991.</li>
 * </ul>
 * 
 */
public class GlobalForestBGC implements MultiplePointsNppEstimator {

	/** Name of he simulation */
	private String name = null;
	
	/**
	 * @return the name
	 */
	public final String getName() {
		return name;
	}

	/**
	 * @param name the name to set
	 */
	public final void setName(String name) {
		this.name = name;
	}

	/** ForestBGC single-point NPP estimator */
	private final ForestBGC forestBGC = ForestBGC.getInstance();
	
	/** Climate data manager */
	private ClimateDataManager climateDataManager = null;
	
	/** Site data manager */
	private SiteDataManager siteDataManager = null;
	
	/** Daily site data manager */
	private DailyDataManager dailyDataManager = null;
	
	/** Seasonally site data manager */
	private SeasonDataManager seasonDataManager = null;

	/** LAI  data manager */
	private LaiDataManager laiDataManager = null;
	
	/** Progress bar to monitorize estimates */
	private JProgressBar progressBar = null;
	
	/**
	 * @return the climateDataManager
	 */
	public final ClimateDataManager getClimateDataManager() {
		return climateDataManager;
	}

	/**
	 * @param climateDataManager the climateDataManager to set
	 */
	public final void setClimateDataManager(ClimateDataManager climateDataManager) {
		this.climateDataManager = climateDataManager;
	}

	/**
	 * @return the siteDataManager
	 */
	public final SiteDataManager getSiteDataManager() {
		return siteDataManager;
	}

	/**
	 * @param siteDataManager the siteDataManager to set
	 */
	public final void setSiteDataManager(SiteDataManager siteDataManager) {
		this.siteDataManager = siteDataManager;
	}

	/**
	 * @return the dailyDataManager
	 */
	public final DailyDataManager getDailyDataManager() {
		return dailyDataManager;
	}

	/**
	 * @param dailyDataManager the dailyDataManager to set
	 */
	public final void setDailyDataManager(DailyDataManager dailyDataManager) {
		this.dailyDataManager = dailyDataManager;
	}

	/**
	 * @return the seasonDataManager
	 */
	public final SeasonDataManager getSeasonDataManager() {
		return seasonDataManager;
	}

	/**
	 * @param seasonDataManager the seasonDataManager to set
	 */
	public final void setSeasonDataManager(SeasonDataManager seasonDataManager) {
		this.seasonDataManager = seasonDataManager;
	}

	/**
	 * @return the laiDataManager
	 */
	public final LaiDataManager getLaiDataManager() {
		return laiDataManager;
	}

	/**
	 * @param laiDataManager the laiDataManager to set
	 */
	public final void setLaiDataManager(LaiDataManager laiDataManager) {
		this.laiDataManager = laiDataManager;
	}

	/** Singleton pattern */
	private final static GlobalForestBGC instance = new GlobalForestBGC();
	
	/** 
	 * Return the singleton instance of GlobalForestBGC class.
	 * @return GlobalForestBGC instance.
	 */
	public static GlobalForestBGC getInstance() {
		return instance;
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.MultiplePointsNppEstimator#getVersion()
	 */
	@Override
	public String getVersion() {
		return "Forest-BGC model implementation based on the Joseph C. Coughlan's SIMLAT5 version 3.6.";
	}

	/**
	 * In case some LAI value were null (i.e, because there are points in a map that don't want to be calculated), this method will return a null result for that points.  
	 * @see es.ehu.www.ccwintco.npp.core.MultiplePointsNppEstimator#estimate()
	 */
	public MultiplePointsNppResults estimate() throws Exception {
		if (climateDataManager == null) throw new Exception("A climate data manager is needed");
		if (siteDataManager == null) throw new Exception("A site data manager is needed");
		if (laiDataManager == null) throw new Exception("A LAI data manager is needed");
		ArrayList<ForestBGCResults> results = new ArrayList<ForestBGCResults>();
		HashMap<Double,ForestBGCResults> heuristico = new HashMap<Double,ForestBGCResults>();
		try {
			forestBGC.setClimateDataManager(climateDataManager);
			forestBGC.setSiteDataManager(siteDataManager);
			forestBGC.setDailyDataManager(dailyDataManager);
			forestBGC.setSeasonDataManager(seasonDataManager);
			forestBGC.setName(name);
			Iterator<CoordinatedLai> it = laiDataManager.laiIterator();
			int num = laiDataManager.getNumData();
			int i = 0;
			while (it.hasNext()) {
				CoordinatedLai lai = it.next();
				if (lai == null) {
					results.add(null);
				}
				else if (heuristico.containsKey(lai.getLai())) {
					results.add((ForestBGCResults)heuristico.get(lai.getLai()));
				}
				else {
					forestBGC.getSiteDataManager().setData(AbstractSiteDataManager.X_8, lai.getLai());
					ForestBGCResults estimatedResults = (ForestBGCResults)forestBGC.estimate();
					results.add(estimatedResults);
					heuristico.put(lai.getLai(), estimatedResults);
				}
				if (progressBar != null) progressBar.setValue(i*100/num);
				i++;
			}
			return new GlobalForestBGCResults(name,new Date(System.currentTimeMillis()),this,results);
		}
		catch(Exception e) {throw e;}
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.MultiplePointsNppEstimator#estimate(JProgressBar)
	 */
	public MultiplePointsNppResults estimate(JProgressBar progressBar) throws Exception {
		this.progressBar = progressBar;
		try {
			return estimate();
		}
		catch (Exception e) {throw e;}
	}

	@Override
	public String getCopyright() {
		return "Copyright 2009 M.A. Veganzones. Copyright 1988 J.C. Coughlan.";
	}

}
