/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.inputs;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Implementation of ClimateDataManager to handle the climate files used by Simlat5.
 */
public class Simlat5ClimateDataManager implements ClimateDataManager {

	/** List for data allocation */
	private List<Double[]> data;
	
	/**
	 * Constructor
	 * @param file The file containing the climate data in a Simlat5 format
	 */
	public Simlat5ClimateDataManager(File file) throws Exception {
		// Initialize data allocation
		data = new ArrayList<Double[]>();
		// Read input file
		try {
			// Check file
			if (!file.isFile()) throw new Exception("Provided file is not a file");
			else {
				// Check if file is readable
				if (!file.canRead()) throw new Exception("Provided file is not readable");
				BufferedReader in = new BufferedReader(new FileReader(file));
				// Read data
				String line = in.readLine(); String[] s;
				int i = 1;
				while (line != null) {
					s = line.split("\t");
					if (s.length != (ClimateDataManager.NUM_PARAMETERS + 1) || Integer.valueOf(s[0]).intValue() != i) {
						s = line.split(" ");
						if (s.length != (ClimateDataManager.NUM_PARAMETERS + 1) || Integer.valueOf(s[0]).intValue() == i) throw new Exception("Provided file data are not valid");
					}
					Double[] d = new Double[ClimateDataManager.NUM_PARAMETERS];
					for (int j = 0; j < ClimateDataManager.NUM_PARAMETERS; j++) d[j] = Double.valueOf(s[j+1]);
					data.add(d);
					line = in.readLine();
					i++;
				}
			}
		}
		catch (Exception e) {throw new Exception("Exception ocurred: " + e.getMessage());}
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.ClimateDataManager#getData(int)
	 */
	public Double[] getData(int parameter) throws Exception {
		if (parameter < 0 || parameter > ClimateDataManager.NUM_PARAMETERS) throw new Exception("Unexisting parameter"); 
		Double[] dataOutput = new Double[this.getNumDays()];
		ListIterator<Double[]> it = data.listIterator();
		int i = 0;
		while (it.hasNext()) {
			dataOutput[i] = it.next()[parameter];
			i++;
		}
		return dataOutput;
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.ClimateDataManager#getData(int, int)
	 */
	public Double getData(int parameter, int day) throws Exception {
		if (parameter < 0 || parameter > ClimateDataManager.NUM_PARAMETERS) throw new Exception("Unexisting parameter"); 
		if (day < 1 || day > this.getNumDays()) throw new Exception("Unexisting day");
		return data.get(day - 1)[parameter];
	}

	public Double[] getAllData(int day) throws Exception {
		if (day < 1 || day > this.getNumDays()) throw new Exception("Unexisting day");
		return data.get(day - 1);
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.ClimateDataManager#getNumDays()
	 */
	public int getNumDays() {
		return data.size();
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.ClimateDataManager#setData(int, java.lang.Double[])
	 */
	public void setData(int parameter, Double[] data) throws Exception {
		if (parameter < 0 || parameter > ClimateDataManager.NUM_PARAMETERS) throw new Exception("Unexisting parameter");
		if (data.length != this.getNumDays()) throw new Exception("Provided data is inconsistent with the number of days");
		for (int i = 0; i < this.getNumDays(); i++) {
			if (data[i] == null) throw new Exception("Provided data has null values");
			Double[] oldData = this.data.get(i);
			oldData[parameter] = data[i];
			this.data.set(i, oldData);
		}
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.ClimateDataManager#setData(int, int, java.lang.Double)
	 */
	public void setData(int parameter, int day, Double data) throws Exception {
		if (parameter < 0 || parameter > ClimateDataManager.NUM_PARAMETERS) throw new Exception("Unexisting parameter"); 
		if (day < 1 || day > this.getNumDays()) throw new Exception("Unexisting day");
		if (data == null) throw new Exception("Provided data has null values");
		Double[] dataOld = this.data.get(day - 1);
		dataOld[parameter] = data;
		this.data.set(day - 1, dataOld);
	}

}
