/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.Image;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileInputStream;
import java.text.DateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import java.util.logging.Level;

import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.SpinnerNumberModel;
import javax.swing.filechooser.FileFilter;
import javax.swing.text.BadLocationException;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;

import es.ehu.www.ccwintco.npp.core.forestbgc.ForestBGC;
import es.ehu.www.ccwintco.npp.core.forestbgc.ForestBGCResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.GlobalForestBGC;
import es.ehu.www.ccwintco.npp.core.forestbgc.GlobalForestBGCResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.AbstractSiteDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.ClimateDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.DailyDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.Mod15DataLaiManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SeasonDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.Simlat5ClimateDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.Simlat5SiteDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SimpleDailyDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SimpleLaiDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SimpleSeasonDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SiteDataManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.DailyDataManager.DailyData;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager.CoordinatedLai;
import es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SeasonDataManager.SeasonData;
import es.ehu.www.ccwintco.npp.core.forestbgc.outputs.ForestBGCOutputManager;
import es.ehu.www.ccwintco.npp.core.forestbgc.outputs.Simlat5ForestBGCOutputManager;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional
 *         <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> User
 *         interface for the NPP estimation program
 */
public class NppGui extends JFrame {

	// Data managers
	/** Climate data manager */
	ClimateDataManager climateDataManager = null;
	/** Site data manager */
	SiteDataManager siteDataManager = null;
	/** Daily data manager */
	DailyDataManager dailyDataManager = null;
	/** Seasonally data manager */
	SeasonDataManager seasonDataManager = null;
	/** LAI data manager */
	LaiDataManager laiDataManager = null;
	
	// Properties
	/** Default properties */
	Properties defaultProperties = null;
	/** User properties */
	Properties userProperties = null;

	// Files
	/** CLM file */
	File clmFile = null;
	/** DAT file */
	File datFile = null;
	/** LIF file */
	File lifFile = null;
	/** SEA file */
	File seaFile = null;
	/** LAI file */
	File laiFile = null;
	
	// Folders
	/** Outputs folder */
	private File outputFolder = null;
	
	// Main Panels
	/** Panel for controls */
	JPanel controlPanel = null;
	/** Panel to show data */
	JPanel dataPanel = null;
	/** Panel to show information */
	JPanel consolePanel = null;

	// Datafiles controls
	/** Label for clm file option */
	JLabel clmFileLabel = null;
	/** Decorating text for clm files input */
	JTextField clmFileTextField = null;
	/** clm files input button */
	JButton clmFileButton = null;
	/** Label for dat file option */
	JLabel datFileLabel = null;
	/** Decorating text for dat files input */
	JTextField datFileTextField = null;
	/** dat files input button */
	JButton datFileButton = null;
	/** Label for lif file option */
	JLabel lifFileLabel = null;
	/** Decorating text for lif files input */
	JTextField lifFileTextField = null;
	/** lif files input button */
	JButton lifFileButton = null;
	/** Label for sea file option */
	JLabel seaFileLabel = null;
	/** Decorating text for sea files input */
	JTextField seaFileTextField = null;
	/** sea files input button */
	JButton seaFileButton = null;

	// LAI data controls
	/** Radio buttom to choose no-lai option */
	JRadioButton nolaiRadioButton = null;
	/** Radio buttom to choose lai file option */
	JRadioButton laiFileRadioButton = null;
	/** Decorating text for lai files input */
	JTextField laiFileTextField = null;
	/** lai files input button */
	JButton laiFileButton = null;

	// Outputs controls
	/** Check box to select resume output */
	JCheckBox resumeOutputCheckBox = null;
	/** Check box to select daily output */
	JCheckBox dailyOutputCheckBox = null;
	/** Check box to select long output */
	JCheckBox longOutputCheckBox = null;
	/** Check box to select map output */
	JCheckBox mapOutputCheckBox = null;
	/** Check box to select text format output */
	JCheckBox textOutputCheckBox = null;
	/** Ouputs folder label */
	JLabel folderOutputLabel = null;
	/** Decorating text for ouputs folder input */
	JTextField folderOutputTextField = null;
	/** ouputs folder input button */
	JButton folderOutputButton = null;
	/** map outputs options button */
	JButton optionsOutputsButton = null;

	// Run controls
	/** Starting day label */
	JLabel startdayLabel = null;
	/** Spinner to select the starting day */
	JSpinner startdaySpinner = null;
	/** Spinner model for starting day */
	SpinnerNumberModel startdaySpinnerModel = null;
	/** Stopping day label */
	JLabel stopdayLabel = null;
	/** Spinner to select the stopping day */
	JSpinner stopdaySpinner = null;
	/** Spinner model for stopping day */
	SpinnerNumberModel stopdaySpinnerModel = null;
	/** Season day label */
	JLabel seasondaysLabel = null;
	/** Spinner to select the number of days for each season */
	JSpinner seasondaysSpinner = null;
	/** Spinner model for seasons */
	SpinnerNumberModel seasondaysSpinnerModel = null;
	/** C fixation ratio label */
	JLabel carbonRatioLabel = null;
	/** Spinner to select the ratio of C fixation */
	JSpinner carbonRatioSpinner = null;
	/** Spinner model for carbon ratio */
	SpinnerNumberModel carbonRatioSpinnerModel = null;
	/** Forest-BGC run button */
	JButton runButton = null;
	/** Reset button */
	JButton resetButton = null;
	/** Progress bar */
	JProgressBar progressBar = null;

	// Data panels
	/** Tabbed pane */
	JTabbedPane dataTabbedPane = null;
	/** Climatic pane */
	JTextPane clmTextPane = null;
	/** State variables (X) pane */
	JTextPane xTextPane = null;
	/** Site constants (B) pane */
	JTextPane bTextPane = null;
	/** LAI panel */
	JTextPane laiTextPane = null;
	/** LAI scroll panel */
	JScrollPane laiScrollPane = null;
	/** MOD-15 LAI panel */
	ImagePanel mod15Pane = null;
	/** MOD-15 LAI scroll panel */
	JScrollPane mod15ScrollPane = null;
	/** Landsat LAI panel */
	ImagePanel landsatPane = null;
	/** Ontogeny (daily) pane */
	JTextPane dailyTextPane = null;
	/** Philogeny (seasonally) pane */
	JTextPane seasonallyTextPane = null;

	// Data text formats
	/** Format for texts */
	SimpleAttributeSet textAttrs = null;

	// Console panels
	JTextPane consoleTextPane = null;

	// Console text formats
	/** Format for info texts */
	SimpleAttributeSet infoAttrs = null;
	/** Format for config texts */
	SimpleAttributeSet configAttrs = null;
	/** Format for warning texts */
	SimpleAttributeSet warningAttrs = null;
	/** Format for severe texts */
	SimpleAttributeSet severeAttrs = null;

	/**
	 * Serial version UID
	 */
	private static final long serialVersionUID = -2985045036743505040L;

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		NppGui nppGui = new NppGui();
		nppGui.setVisible(true);
	}

	/**
	 * @throws HeadlessException
	 */
	public NppGui() throws HeadlessException {
		super();
		initialize();
	}

	/**
	 * Initialize the graphics properties
	 */
	private void initialize() {
		// Window properties
		this.setResizable(true);
		this.setPreferredSize(new Dimension(1024, 768));
		JFrame.setDefaultLookAndFeelDecorated(true);
		this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		this
				.setTitle("Forest-BGC Net Primary Production (NPP) Estimation Workplace");
		// Default and user's properties
		try {
			defaultProperties = new Properties();
			defaultProperties.load(new FileInputStream("bin/properties/default.props"));
			userProperties = new Properties();
			userProperties.load(new FileInputStream("bin/properties/user.props"));
		}
		catch (Exception e) {System.out.println(e.toString());} // Use defaults
		// Layout
		this.setLayout(new BorderLayout(5, 5));
		// Panels
		this.add(getControlPanel(), BorderLayout.WEST);
		JPanel monitorPanel = new JPanel();
		monitorPanel.setLayout(new BorderLayout(5, 5));
		monitorPanel.add(getDataPanel(), BorderLayout.CENTER);
		monitorPanel.add(getConsolePanel(), BorderLayout.SOUTH);
		this.add(monitorPanel, BorderLayout.CENTER);
		// Packing and maximizing
		this.pack();
		this.setExtendedState(JFrame.MAXIMIZED_BOTH);
	}

	// /////////////////////////////////////////////////////////////////////////
	// / PANELS
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the control panel
	 * 
	 * @return the control panel
	 */
	private JPanel getControlPanel() {
		if (controlPanel == null) {
			// initialize
			controlPanel = new JPanel();
			controlPanel.setPreferredSize(new Dimension(400, 600));
			// border
			controlPanel.setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createMatteBorder(2, 10, 2, 2, Color.black),
					BorderFactory.createTitledBorder(BorderFactory
							.createEmptyBorder(), "Controls")));
			// layout
			controlPanel.setLayout(new GridLayout(4, 1, 5, 5));
			// input files
			JPanel fileControlPanel = new JPanel();
			fileControlPanel
					.setBorder(BorderFactory.createTitledBorder(BorderFactory
							.createEtchedBorder(), "Load data from files"));
			fileControlPanel.setLayout(new GridLayout(4, 1));
			// / clm files
			JPanel clmFileControlPanel = new JPanel();
			clmFileLabel = new JLabel("CLM files:");
			clmFileLabel.setPreferredSize(new Dimension(90, 25));
			clmFileControlPanel.add(clmFileLabel);
			clmFileTextField = new JTextField(15);
			clmFileTextField.setPreferredSize(new Dimension(150, 20));
			clmFileTextField.setEditable(false);
			clmFileControlPanel.add(clmFileTextField);
			clmFileButton = new JButton("Select ...");
			clmFileButton.setPreferredSize(new Dimension(85, 25));
			clmFileButton.setVerticalTextPosition(AbstractButton.CENTER);
			clmFileButton.setHorizontalTextPosition(AbstractButton.LEADING);
			clmFileButton.addActionListener(new ClmFileChooseAction());
			clmFileControlPanel.add(clmFileButton);
			fileControlPanel.add(clmFileControlPanel);
			// / dat files
			JPanel datFileControlPanel = new JPanel();
			datFileLabel = new JLabel("DAT files:");
			datFileLabel.setPreferredSize(new Dimension(90, 25));
			datFileControlPanel.add(datFileLabel);
			datFileTextField = new JTextField(15);
			datFileTextField.setPreferredSize(new Dimension(150, 20));
			datFileTextField.setEditable(false);
			datFileControlPanel.add(datFileTextField);
			datFileButton = new JButton("Select ...");
			datFileButton.setPreferredSize(new Dimension(85, 25));
			datFileButton.setVerticalTextPosition(AbstractButton.CENTER);
			datFileButton.setHorizontalTextPosition(AbstractButton.LEADING);
			datFileButton.addActionListener(new DatFileChooseAction());
			datFileControlPanel.add(datFileButton);
			fileControlPanel.add(datFileControlPanel);
			// / lif files
			JPanel lifFileControlPanel = new JPanel();
			lifFileLabel = new JLabel("LIF files:");
			lifFileLabel.setPreferredSize(new Dimension(90, 25));
			lifFileControlPanel.add(lifFileLabel);
			lifFileTextField = new JTextField(15);
			lifFileTextField.setPreferredSize(new Dimension(150, 20));
			lifFileTextField.setEditable(false);
			lifFileControlPanel.add(lifFileTextField);
			lifFileButton = new JButton("Select ...");
			lifFileButton.setPreferredSize(new Dimension(85, 25));
			lifFileButton.setVerticalTextPosition(AbstractButton.CENTER);
			lifFileButton.setHorizontalTextPosition(AbstractButton.LEADING);
			lifFileButton.addActionListener(new LifFileChooseAction());
			lifFileControlPanel.add(lifFileButton);
			fileControlPanel.add(lifFileControlPanel);
			// / sea files
			JPanel seaFileControlPanel = new JPanel();
			seaFileLabel = new JLabel("SEA files:");
			seaFileLabel.setPreferredSize(new Dimension(90, 25));
			seaFileControlPanel.add(seaFileLabel);
			seaFileTextField = new JTextField(15);
			seaFileTextField.setPreferredSize(new Dimension(150, 20));
			seaFileTextField.setEditable(false);
			seaFileControlPanel.add(seaFileTextField);
			seaFileButton = new JButton("Select ...");
			seaFileButton.setPreferredSize(new Dimension(85, 25));
			seaFileButton.setVerticalTextPosition(AbstractButton.CENTER);
			seaFileButton.setHorizontalTextPosition(AbstractButton.LEADING);
			seaFileButton.addActionListener(new SeaFileChooseAction());
			seaFileControlPanel.add(seaFileButton);
			fileControlPanel.add(seaFileControlPanel);
			// / add to controlPanel
			controlPanel.add(fileControlPanel);
			// LAI data
			JPanel laidataControlPanel = new JPanel();
			laidataControlPanel.setBorder(BorderFactory.createTitledBorder(
					BorderFactory.createEtchedBorder(),
					"Leaf Area Index (LAI) regional data"));
			laidataControlPanel.setLayout(new GridLayout(2, 1));
			// / NO regional data
			JPanel nolaiFileControlPanel = new JPanel();
			nolaiRadioButton = new JRadioButton("No regional LAI data");
			nolaiFileControlPanel.add(nolaiRadioButton);
			laidataControlPanel.add(nolaiFileControlPanel);
			// / LAI files
			JPanel laiFileControlPanel = new JPanel();
			laiFileRadioButton = new JRadioButton("LAI file:");
			laiFileRadioButton.setPreferredSize(new Dimension(90, 25));
			laiFileControlPanel.add(laiFileRadioButton);
			laiFileTextField = new JTextField(15);
			laiFileTextField.setPreferredSize(new Dimension(150, 20));
			laiFileTextField.setEditable(false);
			laiFileControlPanel.add(laiFileTextField);
			laiFileButton = new JButton("Select ...");
			laiFileButton.setPreferredSize(new Dimension(85, 25));
			laiFileButton.setVerticalTextPosition(AbstractButton.CENTER);
			laiFileButton.setHorizontalTextPosition(AbstractButton.LEADING);
			laiFileButton.addActionListener(new LaiFileChooseAction());
			laiFileControlPanel.add(laiFileButton);
			laidataControlPanel.add(laiFileControlPanel);
			// / Group radio buttons
			ButtonGroup laidataButtonGroup = new ButtonGroup();
			laidataButtonGroup.add(nolaiRadioButton);
			laidataButtonGroup.add(laiFileRadioButton);
			nolaiRadioButton.setSelected(true);
			// / add to controlPanel
			controlPanel.add(laidataControlPanel);
			// Outputs
			JPanel outputsControlPanel = new JPanel();
			outputsControlPanel.setBorder(BorderFactory
					.createTitledBorder(BorderFactory.createEtchedBorder(),
							"Outputs configuration"));
			outputsControlPanel.setLayout(new GridLayout(4, 1));
			// / Tipo
			JPanel tipoOutputsControlPanel = new JPanel();
			JLabel tipoLabel = new JLabel("Type:");
			tipoOutputsControlPanel.add(tipoLabel);
			resumeOutputCheckBox = new JCheckBox("Resume");
			resumeOutputCheckBox.setPreferredSize(new Dimension(75, 25));
			String selectResume = null;
			resumeOutputCheckBox.setSelected(true);
			if (userProperties != null) selectResume = userProperties.getProperty("selectResume");
			if ((selectResume == null) && (defaultProperties != null)) selectResume = defaultProperties.getProperty("selectResume");
			if ((selectResume != null) && (selectResume.equalsIgnoreCase("false"))) resumeOutputCheckBox.setSelected(false);
			tipoOutputsControlPanel.add(resumeOutputCheckBox);
			dailyOutputCheckBox = new JCheckBox("Daily");
			dailyOutputCheckBox.setPreferredSize(new Dimension(75, 25));
			String selectDaily = null;
			if (userProperties != null)	selectDaily = userProperties.getProperty("selectDaily");
			if ((selectDaily == null) && (defaultProperties != null)) selectDaily = defaultProperties.getProperty("selectDaily");
			if ((selectDaily != null) && (selectDaily.equalsIgnoreCase("true"))) dailyOutputCheckBox.setSelected(true);
			tipoOutputsControlPanel.add(dailyOutputCheckBox);
			longOutputCheckBox = new JCheckBox("Long");
			longOutputCheckBox.setPreferredSize(new Dimension(75, 25));
			String selectLong = null;
			if (userProperties != null) selectLong = userProperties.getProperty("selectLong");
			if ((selectLong == null) && (defaultProperties != null)) selectLong = defaultProperties.getProperty("selectLong");
			if ((selectLong != null) && (selectLong.equalsIgnoreCase("true"))) longOutputCheckBox.setSelected(true);
			tipoOutputsControlPanel.add(longOutputCheckBox);
			mapOutputCheckBox = new JCheckBox("Map");
			mapOutputCheckBox.setPreferredSize(new Dimension(75, 25));
			mapOutputCheckBox.setEnabled(false);
			tipoOutputsControlPanel.add(mapOutputCheckBox);
			outputsControlPanel.add(tipoOutputsControlPanel);
			// / Format
			JPanel formatOutputsControlPanel = new JPanel();
			JLabel formatLabel = new JLabel("Format:");
			formatOutputsControlPanel.add(formatLabel);
			textOutputCheckBox = new JCheckBox("Simlat5");
			textOutputCheckBox.setPreferredSize(new Dimension(75, 25));
			String selectText = null;
			textOutputCheckBox.setSelected(true);
			if (userProperties != null) selectText = userProperties.getProperty("selectText");
			if ((selectText == null) && (defaultProperties != null)) selectText = defaultProperties.getProperty("selectText");
			if ((selectText != null) && (selectText.equalsIgnoreCase("false"))) textOutputCheckBox.setSelected(false);
			formatOutputsControlPanel.add(textOutputCheckBox);
			outputsControlPanel.add(formatOutputsControlPanel);
			// / Outputs folder
			JPanel folderOutputsControlPanel = new JPanel();
			folderOutputLabel = new JLabel("Outputs folder:");
			folderOutputLabel.setPreferredSize(new Dimension(95, 25));
			folderOutputsControlPanel.add(folderOutputLabel);
			folderOutputTextField = new JTextField(15);
			folderOutputTextField.setPreferredSize(new Dimension(130, 20));
			folderOutputTextField.setEditable(false);
			folderOutputsControlPanel.add(folderOutputTextField);
			folderOutputButton = new JButton("Select ...");
			folderOutputButton.setPreferredSize(new Dimension(85, 25));
			folderOutputButton.setVerticalTextPosition(AbstractButton.CENTER);
			folderOutputButton.setHorizontalTextPosition(AbstractButton.LEADING);
			folderOutputButton.addActionListener(new FolderOutputChooseAction());
			folderOutputsControlPanel.add(folderOutputButton);
			outputsControlPanel.add(folderOutputsControlPanel);
			// / Options
			JPanel optionsOutputsControlPanel = new JPanel();
			optionsOutputsButton = new JButton("Options ...");
			optionsOutputsButton.setPreferredSize(new Dimension(100, 25));
			optionsOutputsButton.setVerticalTextPosition(AbstractButton.CENTER);
			optionsOutputsButton.setHorizontalTextPosition(AbstractButton.LEADING);
			optionsOutputsButton.addActionListener(new OptionsOutputsAction());
			optionsOutputsButton.setEnabled(true);
			optionsOutputsControlPanel.add(optionsOutputsButton);
			outputsControlPanel.add(optionsOutputsControlPanel);
			// / add to controlPanel
			controlPanel.add(outputsControlPanel);
			// Control data
			JPanel controldataControlPanel = new JPanel();
			controldataControlPanel.setBorder(BorderFactory.createTitledBorder(
					BorderFactory.createEtchedBorder(),
					"Estimation control parameters"));
			controldataControlPanel.setLayout(new GridLayout(4, 1));
			// / Season control
			JPanel seasonControldataControlPanel = new JPanel();
			seasonControldataControlPanel.setLayout(new GridLayout(1, 3));
			Integer start = null; Integer stop = null; Integer season = null; Double carbonFixation = null;
			if (userProperties != null) {
				String startStr = userProperties.getProperty("start"); if (startStr != null) start = Integer.parseInt(startStr);
				String stopStr = userProperties.getProperty("stop"); if (stopStr != null) stop = Integer.parseInt(stopStr);
				String seasonStr = userProperties.getProperty("season"); if (seasonStr != null) season = Integer.parseInt(seasonStr);
				String carbonFixationStr = userProperties.getProperty("carbonFixation"); if (carbonFixationStr != null) carbonFixation = Double.parseDouble(carbonFixationStr);
			}
			if ((start == null) && (defaultProperties != null)) {
				String startStr = defaultProperties.getProperty("start"); if (startStr != null) start = Integer.parseInt(startStr);
			}
			if ((stop == null) && (defaultProperties != null)) {
				String stopStr = defaultProperties.getProperty("stop"); if (stopStr != null) stop = Integer.parseInt(stopStr);
			}
			if ((season == null) && (defaultProperties != null)) {
				String seasonStr = defaultProperties.getProperty("season"); if (seasonStr != null) season = Integer.parseInt(seasonStr);
			}
			if ((carbonFixation == null) && (defaultProperties != null)) {
				String carbonFixationStr = defaultProperties.getProperty("carbonFixation"); if (carbonFixationStr != null) carbonFixation = Double.parseDouble(carbonFixationStr);
			}
			// /// start day
			JPanel startdayPanel = new JPanel();
			startdayLabel = new JLabel("Start:");
			startdayLabel.setPreferredSize(new Dimension(40, 25));
			startdayPanel.add(startdayLabel);
			if ((start == null) || (start <= 0)) start = 1;
			startdaySpinnerModel = new SpinnerNumberModel(start.intValue(), 1, 10000, 10);
			startdaySpinner = new JSpinner(startdaySpinnerModel);
			startdaySpinner.setPreferredSize(new Dimension(60, 25));
			startdayPanel.add(startdaySpinner);
			seasonControldataControlPanel.add(startdayPanel);
			// /// stop day
			JPanel stopdayPanel = new JPanel();
			stopdayLabel = new JLabel("Stop:");
			stopdayLabel.setPreferredSize(new Dimension(40, 25));
			stopdayPanel.add(stopdayLabel);
			if ((stop == null) || (stop <= 0) || (stop < start)) stop = start + 364;
			stopdaySpinnerModel = new SpinnerNumberModel(stop.intValue(), 1, 10000, 10);
			stopdaySpinner = new JSpinner(stopdaySpinnerModel);
			stopdaySpinner.setPreferredSize(new Dimension(60, 25));
			stopdayPanel.add(stopdaySpinner);
			seasonControldataControlPanel.add(stopdayPanel);
			// /// Season
			JPanel seasondaysPanel = new JPanel();
			seasondaysLabel = new JLabel("Season:");
			seasondaysLabel.setPreferredSize(new Dimension(50, 25));
			seasondaysPanel.add(seasondaysLabel);
			if ((season == null) || (season <= 0)) season = 365;
			seasondaysSpinnerModel = new SpinnerNumberModel(season.intValue(), 1, 10000, 10);
			seasondaysSpinner = new JSpinner(seasondaysSpinnerModel);
			seasondaysSpinner.setPreferredSize(new Dimension(60, 25));
			seasondaysPanel.add(seasondaysSpinner);
			seasonControldataControlPanel.add(seasondaysPanel);
			controldataControlPanel.add(seasonControldataControlPanel);
			// / C fixation ratio
			JPanel carbonControldataControlPanel = new JPanel();
			carbonRatioLabel = new JLabel("Carbon fixation ratio:");
			carbonRatioLabel.setPreferredSize(new Dimension(150, 25));
			carbonControldataControlPanel.add(carbonRatioLabel);
			if ((carbonFixation == null) || (carbonFixation < 0.0) || (carbonFixation > 1.0)) carbonFixation = 0.5;
			carbonRatioSpinnerModel = new SpinnerNumberModel(carbonFixation.doubleValue(), 0.0, 1.0, 0.1);
			carbonRatioSpinner = new JSpinner(carbonRatioSpinnerModel);
			carbonRatioSpinner.setPreferredSize(new Dimension(60, 25));
			carbonControldataControlPanel.add(carbonRatioSpinner);
			controldataControlPanel.add(carbonControldataControlPanel);
			// / Run and reset
			JPanel runControldataControlPanel = new JPanel();
			runButton = new JButton("Run");
			runButton.setPreferredSize(new Dimension(150, 35));
			runButton.setVerticalTextPosition(AbstractButton.CENTER);
			runButton.setHorizontalTextPosition(AbstractButton.LEADING);
			runButton.addActionListener(new RunAction());
			runButton.setEnabled(false);
			runControldataControlPanel.add(runButton);
			resetButton = new JButton("Reset");
			resetButton.setPreferredSize(new Dimension(80, 35));
			resetButton.setVerticalTextPosition(AbstractButton.CENTER);
			resetButton.setHorizontalTextPosition(AbstractButton.LEADING);
			resetButton.addActionListener(new ResetAction());
			runControldataControlPanel.add(resetButton);
			controldataControlPanel.add(runControldataControlPanel);
			// / Progress bar
			JPanel progressControldataControlPanel = new JPanel();
			progressBar = new JProgressBar(0, 100);
			progressBar.setValue(0);
			progressBar.setStringPainted(true);
			progressBar.setPreferredSize(new Dimension(150, 25));
			progressControldataControlPanel.add(progressBar);
			controldataControlPanel.add(progressControldataControlPanel);
			// / add to controlPanel
			controlPanel.add(controldataControlPanel);
		}
		// return
		return controlPanel;
	}

	/**
	 * Gets the data panel
	 * 
	 * @return the data panel
	 */
	private JPanel getDataPanel() {
		if (dataPanel == null) {
			// initialize
			dataPanel = new JPanel();
			// border
			dataPanel.setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createMatteBorder(2, 10, 2, 2, Color.black),
					BorderFactory.createTitledBorder(BorderFactory
							.createEmptyBorder(), "Data and results")));
			// layout
			dataPanel.setLayout(new BorderLayout());
			// Tabbed pane
			dataTabbedPane = new JTabbedPane();
			dataTabbedPane.setPreferredSize(new Dimension(400, 350));
			// Clm Text Pane
			clmTextPane = new JTextPane();
			clmTextPane.setAutoscrolls(true);
			clmTextPane.setEditable(false);
			clmTextPane.setBackground(new Color(240, 240, 240));
			JScrollPane clmScrollPane = new JScrollPane(clmTextPane);
			dataTabbedPane.add(clmScrollPane, "Climatic data values");
			// X Text Pane
			xTextPane = new JTextPane();
			xTextPane.setAutoscrolls(true);
			xTextPane.setEditable(false);
			xTextPane.setBackground(new Color(240, 240, 240));
			JScrollPane xScrollPane = new JScrollPane(xTextPane);
			dataTabbedPane.add(xScrollPane,
					"State Variables (X) initial values");
			// B Text Pane
			bTextPane = new JTextPane();
			bTextPane.setAutoscrolls(true);
			bTextPane.setEditable(false);
			bTextPane.setBackground(new Color(240, 240, 240));
			JScrollPane bScrollPane = new JScrollPane(bTextPane);
			dataTabbedPane.add(bScrollPane, "Site Constants (B) values");
			// Daily Text Pane
			dailyTextPane = new JTextPane();
			dailyTextPane.setAutoscrolls(true);
			dailyTextPane.setEditable(false);
			dailyTextPane.setBackground(new Color(240, 240, 240));
			JScrollPane dailyScrollPane = new JScrollPane(dailyTextPane);
			dataTabbedPane.add(dailyScrollPane,
					"Ontogeny Site Constants replacing values");
			// Seasonally Text Pane
			seasonallyTextPane = new JTextPane();
			seasonallyTextPane.setAutoscrolls(true);
			seasonallyTextPane.setEditable(false);
			seasonallyTextPane.setBackground(new Color(240, 240, 240));
			JScrollPane seasonallyScrollPane = new JScrollPane(
					seasonallyTextPane);
			dataTabbedPane.add(seasonallyScrollPane,
					"Philogeny Site Constants replacing values");
			// add tabbed pane
			dataPanel.add(dataTabbedPane, BorderLayout.CENTER);
			// Formats
			textAttrs = new SimpleAttributeSet();
			StyleConstants.setBold(textAttrs, true);
			StyleConstants.setForeground(textAttrs, new Color(20, 20, 20));
		}
		// return
		return dataPanel;
	}

	/**
	 * Gets the console panel
	 * 
	 * @return the console panel
	 */
	private JPanel getConsolePanel() {
		if (consolePanel == null) {
			// initialize
			consolePanel = new JPanel();
			consolePanel.setPreferredSize(new Dimension(400, 250));
			// border
			consolePanel.setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createMatteBorder(2, 10, 2, 2, Color.black),
					BorderFactory.createTitledBorder(BorderFactory
							.createEmptyBorder(), "Console")));
			// layout
			consolePanel.setLayout(new BorderLayout());
			// Console Pane
			consoleTextPane = new JTextPane();
			consoleTextPane.setAutoscrolls(true);
			consoleTextPane.setEditable(false);
			consoleTextPane.setBackground(new Color(240, 240, 220));
			// Console Scroll Pane
			JScrollPane consoleScrollPane = new JScrollPane(consoleTextPane);
			// Format
			infoAttrs = new SimpleAttributeSet();
			StyleConstants.setBold(infoAttrs, true);
			StyleConstants.setForeground(infoAttrs, new Color(13, 132, 200));
			configAttrs = new SimpleAttributeSet();
			StyleConstants.setForeground(configAttrs, new Color(13, 132, 9));
			StyleConstants.setBold(configAttrs, true);
			warningAttrs = new SimpleAttributeSet();
			StyleConstants.setBold(warningAttrs, true);
			StyleConstants.setForeground(warningAttrs, new Color(200, 100, 9));
			severeAttrs = new SimpleAttributeSet();
			StyleConstants.setBold(severeAttrs, true);
			StyleConstants.setForeground(severeAttrs, Color.red);
			// add pane
			consolePanel.add(consoleScrollPane, BorderLayout.CENTER);
		}
		// return
		return consolePanel;
	}

	// /////////////////////////////////////////////////////////////////////////
	// / ACTION LISTENERS
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for choosing Clm files
	 */
	private class ClmFileChooseAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// File chooser dialog
			String clmPath = null;
			if (userProperties != null) {
				clmPath = userProperties.getProperty("clmFolder");
				if (clmPath == null) clmPath = userProperties.getProperty("dataFolder");
			}
			if ((clmPath == null) && (defaultProperties != null)) {
				clmPath = defaultProperties.getProperty("clmFolder");
				if (clmPath == null) clmPath = defaultProperties.getProperty("dataFolder");
			}
			JFileChooser fileChooser;
			if (clmPath == null) fileChooser = new JFileChooser();
			else fileChooser = new JFileChooser(clmPath);
			// Climate filter
			fileChooser.setFileFilter(new ClimateFileFilter());
			int returnVal = fileChooser.showOpenDialog(NppGui.this);
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				clmFile = fileChooser.getSelectedFile();
				clmFileTextField.setText(clmFile.getAbsolutePath());
				try {
					// log in console
					log("Loading climate file: " + clmFile.getName(),
							Level.INFO);
					// Wait cursor
					setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
					// load climatic data
					loadClimateData();
					// log in console
					log("Number of days for the model: "
							+ climateDataManager.getNumDays(), Level.INFO);
					log("Climate data successfully loaded", Level.CONFIG);
					// maximum day start, stop and season days
					startdaySpinnerModel.setMaximum(climateDataManager
							.getNumDays());
					stopdaySpinnerModel.setMaximum(climateDataManager
							.getNumDays());
					seasondaysSpinnerModel.setMaximum(climateDataManager
							.getNumDays());
					// if site data available, model can be run
					if (datFile != null) {
						runButton.setEnabled(true);
						// log in console
						log("Forest-BGC model can be runned now", Level.CONFIG);
					}
					// Normal cursor
					setCursor(null);
				} catch (Exception exc) {
					// log in console
					log(
							"The climate data file contains errors. Data couldn't be loaded.",
							Level.SEVERE);
					// nullyfy climaif file and manager
					clmFile = null;
					clmFileTextField.setText("");
					climateDataManager = null;
					// disable run button
					runButton.setEnabled(false);
					// Normal cursor
					setCursor(null);
				}
			}
		}

		/**
		 * 
		 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
		 * @author Grupo Inteligencia Computacional
		 *         <http://www.ehu.es/computationalintelligence>
		 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
		 *         Filter for climate files
		 */
		private class ClimateFileFilter extends FileFilter {

			public boolean accept(File f) {
				if (f.isDirectory()) {
					return true;
				}
				if (f != null && f.getName().toLowerCase().endsWith(".clm"))
					return true;
				else
					return false;
			}

			public String getDescription() {
				return "Climate files (*.clm)";
			}
		}

	}

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for choosing Dat files
	 */
	private class DatFileChooseAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// File chooser dialog
			String datPath = null;
			if (userProperties != null) {
				datPath = userProperties.getProperty("datFolder");
				if (datPath == null) datPath = userProperties.getProperty("dataFolder");
			}
			if ((datPath == null) && (defaultProperties != null)) {
				datPath = defaultProperties.getProperty("datFolder");
				if (datPath == null) datPath = defaultProperties.getProperty("dataFolder");
			}
			JFileChooser fileChooser;
			if (datPath == null) fileChooser = new JFileChooser();
			else fileChooser = new JFileChooser(datPath);
			// Site data filter
			fileChooser.setFileFilter(new SiteDataFileFilter());
			int returnVal = fileChooser.showOpenDialog(NppGui.this);
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				datFile = fileChooser.getSelectedFile();
				datFileTextField.setText(datFile.getAbsolutePath());
				try {
					// log in console
					log("Loading site data file: " + datFile.getName(),
							Level.INFO);
					// Wait cursor
					setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
					// load site data
					loadSiteData();
					// log in console
					if (siteDataManager.isStaticBudget())
						log(
								"The provided data indicate a 'Static' carbon budget",
								Level.INFO);
					else
						log(
								"The provided data indicate a 'Dynamic' carbon and nitrogen budget",
								Level.INFO);
					log("Site data successfully loaded", Level.CONFIG);
					// if climatic data available, model can be run
					if (clmFile != null) {
						runButton.setEnabled(true);
						// log in console
						log("Forest-BGC model can be runned now", Level.CONFIG);
					}
					// Normal cursor
					setCursor(null);
				} catch (Exception exc) {
					// log in console
					log(
							"The site data file contains errors. Data couldn't be loaded.",
							Level.SEVERE);
					// nullyfy site data file and manager
					datFile = null;
					datFileTextField.setText("");
					siteDataManager = null;
					// disable run button
					runButton.setEnabled(false);
					// Normal cursor
					setCursor(null);
				}
			}
		}

		/**
		 * 
		 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
		 * @author Grupo Inteligencia Computacional
		 *         <http://www.ehu.es/computationalintelligence>
		 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
		 *         Filter for site data files
		 */
		private class SiteDataFileFilter extends FileFilter {

			public boolean accept(File f) {
				if (f.isDirectory()) {
					return true;
				}
				if (f != null && f.getName().toLowerCase().endsWith(".dat"))
					return true;
				else
					return false;
			}

			public String getDescription() {
				return "Site data files (*.dat)";
			}

		}

	}

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for choosing Lif files
	 */
	private class LifFileChooseAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// File chooser dialog
			String lifPath = null;
			if (userProperties != null) {
				lifPath = userProperties.getProperty("lifFolder");
				if (lifPath == null) lifPath = userProperties.getProperty("dataFolder");
			}
			if ((lifPath == null) && (defaultProperties != null)) {
				lifPath = defaultProperties.getProperty("lifFolder");
				if (lifPath == null) lifPath = defaultProperties.getProperty("dataFolder");
			}
			JFileChooser fileChooser;
			if (lifPath == null) fileChooser = new JFileChooser();
			else fileChooser = new JFileChooser(lifPath);
			// Lif filter
			fileChooser.setFileFilter(new DailyFileFilter());
			int returnVal = fileChooser.showOpenDialog(NppGui.this);
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				lifFile = fileChooser.getSelectedFile();
				lifFileTextField.setText(lifFile.getAbsolutePath());
				try {
					// log in console
					log("Loading ontogeny file: " + lifFile.getName(),
							Level.INFO);
					// Wait cursor
					setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
					// load daily data
					loadDailyData();
					// log in console
					log("Ontogeny data successfully loaded", Level.CONFIG);
					// Normal cursor
					setCursor(null);
				} catch (Exception exc) {
					// log in console
					log(
							"The ontogeny data file contains errors. Data couldn't be loaded.",
							Level.SEVERE);
					// nullyfy lif file and manager
					lifFile = null;
					lifFileTextField.setText("");
					dailyDataManager = null;
					// Normal cursor
					setCursor(null);
				}
			}
		}

		/**
		 * 
		 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
		 * @author Grupo Inteligencia Computacional
		 *         <http://www.ehu.es/computationalintelligence>
		 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
		 *         Filter for ontogeny (daily) files
		 */
		private class DailyFileFilter extends FileFilter {

			public boolean accept(File f) {
				if (f.isDirectory()) {
					return true;
				}
				if (f != null && f.getName().toLowerCase().endsWith(".lif"))
					return true;
				else
					return false;
			}

			public String getDescription() {
				return "Ontogeny files (*.lif)";
			}
		}

	}

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for choosing Sea files
	 */
	private class SeaFileChooseAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// File chooser dialog
			String seaPath = null;
			if (userProperties != null) {
				seaPath = userProperties.getProperty("seaFolder");
				if (seaPath == null) seaPath = userProperties.getProperty("dataFolder");
			}
			if ((seaPath == null) && (defaultProperties != null)) {
				seaPath = defaultProperties.getProperty("seaFolder");
				if (seaPath == null) seaPath = defaultProperties.getProperty("dataFolder");
			}
			JFileChooser fileChooser;
			if (seaPath == null) fileChooser = new JFileChooser();
			else fileChooser = new JFileChooser(seaPath);
			// Sea filter
			fileChooser.setFileFilter(new SeasonallyFileFilter());
			int returnVal = fileChooser.showOpenDialog(NppGui.this);
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				seaFile = fileChooser.getSelectedFile();
				seaFileTextField.setText(seaFile.getAbsolutePath());
				try {
					// log in console
					log("Loading philogeny file: " + seaFile.getName(),
							Level.INFO);
					// Wait cursor
					setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
					// load season data
					loadSeasonallyData();
					// log in console
					log("Philogeny data successfully loaded", Level.CONFIG);
					// Normal cursor
					setCursor(null);
				} catch (Exception exc) {
					// log in console
					log(
							"The philogeny data file contains errors. Data couldn't be loaded.",
							Level.SEVERE);
					// nullyfy sea file file and manager
					seaFile = null;
					seaFileTextField.setText("");
					seasonDataManager = null;
					// Normal cursor
					setCursor(null);
				}
			}
		}

		/**
		 * 
		 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
		 * @author Grupo Inteligencia Computacional
		 *         <http://www.ehu.es/computationalintelligence>
		 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
		 *         Filter for philogeny (seasonally) files
		 */
		private class SeasonallyFileFilter extends FileFilter {

			public boolean accept(File f) {
				if (f.isDirectory()) {
					return true;
				}
				if (f != null && f.getName().toLowerCase().endsWith(".sea"))
					return true;
				else
					return false;
			}

			public String getDescription() {
				return "Philogeny files (*.sea)";
			}
		}

	}

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for choosing Lai files
	 */
	private class LaiFileChooseAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// File chooser dialog
			String laiPath = null;
			if (userProperties != null) {
				laiPath = userProperties.getProperty("laiFolder");
				if (laiPath == null) laiPath = userProperties.getProperty("dataFolder");
			}
			if ((laiPath == null) && (defaultProperties != null)) {
				laiPath = defaultProperties.getProperty("laiFolder");
				if (laiPath == null) laiPath = defaultProperties.getProperty("dataFolder");
			}
			JFileChooser fileChooser;
			if (laiPath == null) fileChooser = new JFileChooser();
			else fileChooser = new JFileChooser(laiPath);
			// Lai filters
			fileChooser.addChoosableFileFilter(new LaiFileFilter());
			fileChooser.addChoosableFileFilter(new Mod15FileFilter());
			fileChooser.setAcceptAllFileFilterUsed(false);
			int returnVal = fileChooser.showOpenDialog(NppGui.this);
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				laiFile = fileChooser.getSelectedFile();
				laiFileTextField.setText(laiFile.getAbsolutePath());
				try {
					// Remove possible previous tabs
					dataTabbedPane.remove(laiScrollPane);
					dataTabbedPane.remove(mod15ScrollPane);
					// Wait cursor
					setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
					if (laiFile.getName().endsWith(".lai")) { // LAI data
						// log in console
						log("Loading Leaf Area Index (LAI) file: " + laiFile.getName(),	Level.INFO);
						// load season data
						loadLaiData();
					}
					else if (laiFile.getName().endsWith(".hdf")) { // MOD-15 data
						// log in console
						log("Loading MODIS LAI (MOD-15) file: " + laiFile.getName(), Level.INFO);
						// load season data
						loadMod15Data();
					}
					else { // Landsat data
						// log in console
						log("Loading Landsat file: " + laiFile.getName(), Level.INFO);
						// load season data
						loadLandsatData();
					}
					// Select Lai radio button
					laiFileRadioButton.setSelected(true);
					// Enables map output selection
					mapOutputCheckBox.setEnabled(true);
					// log in console
					log("LAI data successfully loaded", Level.CONFIG);
					// Normal cursor
					setCursor(null);

				} catch (Exception exc) {
					// log in console
					log(
							"The LAI data file contains errors. Data couldn't be loaded.",
							Level.SEVERE);
					// nullyfy lai file file and manager
					laiFile = null;
					laiFileTextField.setText("");
					laiDataManager = null;
					// Normal cursor
					setCursor(null);
				}
			}
		}

		/**
		 * 
		 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
		 * @author Grupo Inteligencia Computacional
		 *         <http://www.ehu.es/computationalintelligence>
		 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
		 *         Filter for LAI files
		 */
		private class LaiFileFilter extends FileFilter {

			public boolean accept(File f) {
				if (f.isDirectory()) {
					return true;
				}
				if (f != null && f.getName().toLowerCase().endsWith(".lai"))
					return true;
				else
					return false;
			}

			public String getDescription() {
				return "Leaf Area Index (LAI) files (*.lai)";
			}
		}

		/**
		 * 
		 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
		 * @author Grupo Inteligencia Computacional
		 *         <http://www.ehu.es/computationalintelligence>
		 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
		 *         Filter for MOD-15 product files
		 */
		private class Mod15FileFilter extends FileFilter {

			public boolean accept(File f) {
				if (f.isDirectory()) {
					return true;
				}
				if (f != null && f.getName().toLowerCase().endsWith(".hdf"))
					return true;
				else
					return false;
			}

			public String getDescription() {
				return "MOD-15 HDF files (*.hdf)";
			}
		}
	}

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for choosing outputs folder
	 */
	private class FolderOutputChooseAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// File chooser dialog
			String folderOutputPath = null;
			if (userProperties != null) {
				folderOutputPath = userProperties.getProperty("outputFolder");
				if (folderOutputPath == null) folderOutputPath = userProperties.getProperty("outputFolder");
			}
			if ((folderOutputPath == null) && (defaultProperties != null)) {
				folderOutputPath = defaultProperties.getProperty("outputFolder");
				if (folderOutputPath == null) folderOutputPath = defaultProperties.getProperty("outputFolder");
			}
			JFileChooser fileChooser;
			if (folderOutputPath == null) fileChooser = new JFileChooser();
			else fileChooser = new JFileChooser(folderOutputPath);
			fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			fileChooser.setAcceptAllFileFilterUsed(false);
			int returnVal = fileChooser.showOpenDialog(NppGui.this);
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				outputFolder = fileChooser.getSelectedFile();
				folderOutputTextField.setText(outputFolder.getAbsolutePath());
				// log in console
				log("Ouputs folder selected: " + outputFolder.getAbsolutePath(), Level.INFO);
			}
		}

	}

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for choosing outputs options
	 */
	private class OptionsOutputsAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// TODO Auto-generated method stub

		}

	}

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for running the model
	 */
	private class RunAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// Wait cursor
			setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			// Disable interface
			/// Files buttons
			clmFileButton.setEnabled(false);
			datFileButton.setEnabled(false);
			lifFileButton.setEnabled(false);
			seaFileButton.setEnabled(false);
			/// LAI files buttons
			nolaiRadioButton.setEnabled(false);
			laiFileRadioButton.setEnabled(false);
			laiFileButton.setEnabled(false);
			/// Outputs
			resumeOutputCheckBox.setEnabled(false);
			dailyOutputCheckBox.setEnabled(false);
			longOutputCheckBox.setEnabled(false);
			mapOutputCheckBox.setEnabled(false);
			/// Day spinners
			startdaySpinner.setEnabled(false);
			stopdaySpinner.setEnabled(false);
			seasondaysSpinner.setEnabled(false);
			/// Carbon spinner
			carbonRatioSpinner.setEnabled(false);
			/// Run button
			runButton.setEnabled(false);
			/// Reset button
			resetButton.setEnabled(false);
			// Run tasks
			/// choose task depending on LAI data
			if (nolaiRadioButton.isSelected()) {
				// log in console
				log("Running single-point ForestBGC model", Level.CONFIG);
				// Configure progress bar
				progressBar.setIndeterminate(true);
				// set data managers
				ForestBGC.getInstance().setClimateDataManager(climateDataManager);
				ForestBGC.getInstance().setSiteDataManager(siteDataManager);
				ForestBGC.getInstance().setDailyDataManager(dailyDataManager);
				ForestBGC.getInstance().setSeasonDataManager(seasonDataManager);
				String name = datFile.getName();
				ForestBGC.getInstance().setName(name.substring(0, name.length()-4));
				try {
					// Estimate NPP
					ForestBGCResults results = (ForestBGCResults)ForestBGC.getInstance().estimate();
					// log in console
					log("Npp estimate concluded", Level.INFO);
					// Give outputs
					giveOutput(results,(Double)carbonRatioSpinnerModel.getValue());
					// Update progress bar
					progressBar.setIndeterminate(false);
					progressBar.setValue(100);
				}
				catch (Exception exc) {
					// log in console
					log("Forest-BGC model cannot estimate Npp for current data", Level.SEVERE);
					// Update progress bar
					progressBar.setIndeterminate(false);
					progressBar.setValue(0);
				}
			}
			else {
				// log in console
				log("Running global ForestBGC model", Level.CONFIG);
				// Configure progress bar
				progressBar.setIndeterminate(true);
				// set data managers
				GlobalForestBGC.getInstance().setClimateDataManager(climateDataManager);
				GlobalForestBGC.getInstance().setSiteDataManager(siteDataManager);
				GlobalForestBGC.getInstance().setDailyDataManager(dailyDataManager);
				GlobalForestBGC.getInstance().setSeasonDataManager(seasonDataManager);
				GlobalForestBGC.getInstance().setLaiDataManager(laiDataManager);
				String name = datFile.getName();
				GlobalForestBGC.getInstance().setName(name.substring(0, name.length()-4));
				try {
					// log in console
					log("Forest-BGC is going to run " + laiDataManager.getNumData() + " simulations. This could take a long time ...", Level.WARNING);
					// Estimate NPP
					GlobalForestBGCResults results = (GlobalForestBGCResults)GlobalForestBGC.getInstance().estimate(progressBar);
					// log in console
					log("Npp estimates concluded", Level.INFO);
					// Give outputs
					giveOutput(results,(Double)carbonRatioSpinnerModel.getValue());
					// Update progress bar
					progressBar.setIndeterminate(false);
					progressBar.setValue(100);
				}
				catch (Exception exc) {
					// log in console
					log("Forest-BGC model cannot estimate Npp for current data", Level.SEVERE);
					// Update progress bar
					progressBar.setIndeterminate(false);
					progressBar.setValue(0);
				}
			}
			// Enable interface
			/// Files buttons
			clmFileButton.setEnabled(true);
			datFileButton.setEnabled(true);
			lifFileButton.setEnabled(true);
			seaFileButton.setEnabled(true);
			/// LAI files buttons
			nolaiRadioButton.setEnabled(true);
			laiFileRadioButton.setEnabled(true);
			laiFileButton.setEnabled(true);
			/// Outputs
			resumeOutputCheckBox.setEnabled(true);
			dailyOutputCheckBox.setEnabled(true);
			longOutputCheckBox.setEnabled(true);
			if (laiDataManager != null) mapOutputCheckBox.setEnabled(true);
			/// Day spinners
			startdaySpinner.setEnabled(true);
			stopdaySpinner.setEnabled(true);
			seasondaysSpinner.setEnabled(true);
			/// Carbon spinner
			carbonRatioSpinner.setEnabled(true);
			/// Run button
			runButton.setEnabled(true);
			/// Reset button
			resetButton.setEnabled(true);
			// Normal cursor
			setCursor(null);
		}

	}

	/**
	 * 
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional
	 *         <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es> Action
	 *         Listener for resetting the model
	 */
	private class ResetAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			// log in console
			log("Model reset by user", Level.WARNING);
			// reset
			reset();
		}

	}

	// /////////////////////////////////////////////////////////////////////////
	// / BUSINESS LOGIC
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Manages the log outputs in console
	 * 
	 * @param message
	 * @param level
	 *            as defined in Level: CONFIG, WARNING, SEVERE and INFO
	 *            (default)
	 */
	private void log(String message, Level level) {
		StyledDocument doc = consoleTextPane.getStyledDocument();
		int length = doc.getLength();
		Date date = new Date(System.currentTimeMillis());
		try {
			if (level.equals(Level.CONFIG))
				doc.insertString(length, "["
						+ DateFormat.getDateTimeInstance(DateFormat.SHORT,
								DateFormat.SHORT).format(date) + "] " + message
						+ "\n", configAttrs);
			else if (level.equals(Level.WARNING))
				doc.insertString(length, "["
						+ DateFormat.getDateTimeInstance(DateFormat.SHORT,
								DateFormat.SHORT).format(date) + "] WARNING: "
						+ message + "\n", warningAttrs);
			else if (level.equals(Level.SEVERE))
				doc.insertString(length, "["
						+ DateFormat.getDateTimeInstance(DateFormat.SHORT,
								DateFormat.SHORT).format(date) + "] ERROR: "
						+ message + "\n", severeAttrs);
			else
				doc.insertString(length, "["
						+ DateFormat.getDateTimeInstance(DateFormat.SHORT,
								DateFormat.SHORT).format(date) + "] " + message
						+ "\n", infoAttrs);
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Loads climate data.
	 */
	private void loadClimateData() throws Exception {
		try {
			// Get data
			climateDataManager = new Simlat5ClimateDataManager(clmFile);
			Double[] tMax = climateDataManager
					.getData(ClimateDataManager.T_MAX);
			Double[] tMin = climateDataManager
					.getData(ClimateDataManager.T_MIN);
			Double[] dewPoint = climateDataManager
					.getData(ClimateDataManager.DEW_POINT);
			Double[] dayLength = climateDataManager
					.getData(ClimateDataManager.DAY_LENGTH);
			Double[] precision = climateDataManager
					.getData(ClimateDataManager.PRECISION);
			// Write data
			StyledDocument doc = clmTextPane.getStyledDocument();
			doc.insertString(doc.getLength(),
					"DAY\tT_MAX\tT_MIN\tDEW_POINT\tDAY_LENGTH\tPRECISION\n\n",
					infoAttrs);
			for (int i = 0; i < climateDataManager.getNumDays(); i++)
				doc.insertString(doc.getLength(), (i + 1) + "\t" + tMax[i]
						+ "\t" + tMin[i] + "\t" + dewPoint[i] + "\t\t"
						+ precision[i] + "\t\t" + dayLength[i] + "\t" + "\n",
						textAttrs);
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * Loads site data.
	 */
	private void loadSiteData() throws Exception {
		try {
			// Get data
			siteDataManager = new Simlat5SiteDataManager(datFile);
			// B data
			StyledDocument bdoc = bTextPane.getStyledDocument();
			bdoc.insertString(bdoc.getLength(),
					"B\tDESCRIPTION\t\t\t\tVALUE\tUNITS\n\n", infoAttrs);
			bdoc.insertString(bdoc.getLength(), "1\tSpecific leaf area\t\t\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.B_1)
					+ "\t(m^2 kg^-1\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"2\tCanopy light extinction coefficient\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_2)
							+ "\t(dimensionless)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"3\tSoil water capacity\t\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_3)
							+ "\t(m^3 ha^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"4\tWater interception coefficient\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_4)
							+ "\t(m L*^-1 day^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"5\tGround surface area\t\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_5)
							+ "\t(m^2 ha^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"6\tSnowmelt coefficient\t\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_6)
							+ "\t(m Celsius^-1 day^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(), "7\tLatitude\t\t\t\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.B_7)
					+ "\t(deg)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(), "8\t1 - surface albedo\t\t\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.B_8)
					+ "\t(dimensionless)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"9\tMinimum water potential in spring\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_9)
							+ "\t(MPa)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"10\tRadiation reducing leaf conductance threshold\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_10)
							+ "\t(Kj m⁻2 day^-1 )\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"11\tMaximum canopy average leaf conductance\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_11)
							+ "\t(m s^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"12\tLeaf water potential at stomatal closure\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_12)
							+ "\t(MPa)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"13\tSlope of absolute humidity reduction\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_13)
							+ "\t(m s^-1 Habs^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"14\tPhotosynthesis light compensation point\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_14)
							+ "\t(Kj m^-2)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"15\tPhotosynthesis maximum\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_15)
							+ "\t(Kj m^-2 day^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"16\tMaximum leaf conductance (CO2)\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_16)
							+ "\t(m s^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"17\tMinimum temperature for photosynthesis\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_17)
							+ "\t(Celsius)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"18\tMaximum temperature for photosynthesis\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_18)
							+ "\t(Celsius)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"19\tLeaf respireation coefficient\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_19)
							+ "(Celsius^-1 day^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"20\tStem respireation coefficient\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_20)
							+ "(Celsius^-1 day^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"21\tRoot respireation coefficient\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_21)
							+ "(Celsius^-1 day^-1)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(), "22\t\t\t\t\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.B_22) + "\n",
					textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"23\tTemperature effect on mesophyll condition coefficient\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_23)
							+ "(Celsius)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"24\tDecomposition temperature optimum\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_24)
							+ "(Celsius)\n", textAttrs);
			bdoc.insertString(bdoc.getLength(),
					"25\tQ10=2.3 for exponential resiration surface\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.B_25)
							+ "(Celsius)\n", textAttrs);
			if (siteDataManager.isStaticBudget()) {
				bdoc.insertString(bdoc.getLength(),
						"26\tLeaf nitrogen concentration\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_26)
								+ "\t(%)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(), "27\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_27) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "28\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_28) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "29\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_29) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"30\tLeaf carbon allocation fraction\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_30)
								+ "\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"31\tStem carbon allocation fraction\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_31)
								+ "\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"32\tRoot carbon allocation fraction\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_32)
								+ "\n", textAttrs);
				bdoc.insertString(bdoc.getLength(), "33\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_33) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "34\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_34) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "35\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_35) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "36\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_36) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "37\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_37) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "38\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_38) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "39\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_39) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"40\tLeaf drop coefficient (Coniferous:1.0, Deciduous:0.2)\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_40)
								+ "\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"41\tStem turnover coefficient\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_41)
								+ "\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"42\tRoot turnover coefficient\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_42)
								+ "\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"43\tLeaf growth respiration\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_43)
								+ "\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"44\tStem growth respiration\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_44)
								+ "\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"45\tRoot growth respiration\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_45)
								+ "\n", textAttrs);
			} else {
				bdoc.insertString(bdoc.getLength(),
						"26\tMaximum canopy average leaf nitrogen concentration\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_26)
								+ "\t(N Celsius)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"27\tMinimum canopy average leaf nitrogen concentration\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_27)
								+ "(N Celsius)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"28\tMaximum leaf nitrogen retranslocation fraction\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_28)
								+ "(fraction year^-1)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"29\tSoil litter decomposition rate factor (0-1)\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_29)
								+ "(fraction year^-1)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"30\tN/C decomposition release fraction\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_30)
								+ "(dimensionless)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"31\tMaximum leaf area index (all sides)\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_31)
								+ "(dimensionless)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"32\tLeaf turnover age\t\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_32)
								+ "(years)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"33\tLeaf lignin fraction\t\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_33)
								+ "(%)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"34\tSoil water leaf/root allocation factor\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_34)
								+ "(dimensionless)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"35\tNitrogen availability leaf/root allocation factor\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_35)
								+ "(dimensionless)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"36\tDate of spring leaf growth\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_36)
								+ "(yearday)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"37\tDate of fall leaf growth\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_37)
								+ "(yearday)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"38\tMobile N retention time\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_38)
								+ "(years)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"39\tAtmospheric N deposition\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_39)
								+ "(Kg ha^-1 year^-1)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"40\tBiological N fixation\t\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_40)
								+ "(Kg ha^-1 year^-1)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"41\tStem turnover coefficient\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_41)
								+ "(fraction year^-1)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"42\tRoot turnover coefficient\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_42)
								+ "(fraction year^-1)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"43\tLeaf growth respiration\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_43)
								+ "(Celsius)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"44\tStem growth respiration\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_44)
								+ "(Celsius)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"45\tRoot growth respiration\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_45)
								+ "(Celsius)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"46\tDecompositon optimum temperature\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_46)
								+ "(Celsius)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"47\tSoil/litter C decomosition fraction\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_47)
								+ "(dimensionless)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(),
						"48\tDecomposition rate scalar\t\t\t"
								+ siteDataManager.getData(AbstractSiteDataManager.B_48)
								+ "(dimensionless)\n", textAttrs);
				bdoc.insertString(bdoc.getLength(), "49\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_49) + "\n",
						textAttrs);
				bdoc.insertString(bdoc.getLength(), "50\t\t\t\t\t\t"
						+ siteDataManager.getData(AbstractSiteDataManager.B_50) + "\n",
						textAttrs);
			}
			// X data
			StyledDocument xdoc = xTextPane.getStyledDocument();
			xdoc.insertString(xdoc.getLength(),
					"X\tDESCRIPTION\t\tVALUE\tUNITS\n\n", infoAttrs);
			xdoc.insertString(xdoc.getLength(), "1\tSnowpack\t\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_1)
					+ "\t(m^3)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "2\tSoil water content\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_2)
					+ "\t(m^3)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "3\tWater outflow\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_3)
					+ "\t(m^3)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "4\tTranspiration\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_4)
					+ "\t(m^3)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "5\tEvaporation\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_5)
					+ "\t(m^3)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(),
					"6\tPSN\t\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.X_6)
							+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(),
					"7\tRespiraton autotrophic\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.X_7)
							+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(),
					"8\tLeaf carbon\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.X_8)
							+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(),
					"9\tStem carbon\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.X_9)
							+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "10\tRoot carbon\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_10)
					+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(),
					"11\tLeaf/Root litter carbon\t\t"
							+ siteDataManager.getData(AbstractSiteDataManager.X_11)
							+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(),
					"12\tRespiration decompositon, C\t"
							+ siteDataManager.getData(AbstractSiteDataManager.X_12)
							+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "13\tSoil carbon\t\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_13)
					+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "14\tAvailable nitrogen\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_14)
					+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(),
					"15\tLeaf nitrogen(1.5% OF X8)\t"
							+ siteDataManager.getData(AbstractSiteDataManager.X_15)
							+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "16\tStem nitrogen\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_16)
					+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "17\tRoot nitrogen\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_17)
					+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(),
					"18\tLeaf/Root litter nitrogen\t"
							+ siteDataManager.getData(AbstractSiteDataManager.X_18)
							+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "19\tSoil nitrogen\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_19)
					+ "\t(Kg)\n", textAttrs);
			xdoc.insertString(xdoc.getLength(), "20\tNitrogen loss\t\t"
					+ siteDataManager.getData(AbstractSiteDataManager.X_20)
					+ "\t(Kg)\n", textAttrs);
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * Loads ontogeny data.
	 */
	private void loadDailyData() throws Exception {
		try {
			// Get data
			dailyDataManager = new SimpleDailyDataManager(lifFile);
			List<Integer> days = dailyDataManager.getDays();
			Collections.sort(days);
			ListIterator<Integer> daysIterator = days.listIterator();
			List<Integer> constants = dailyDataManager.getReplacingConstants();
			ListIterator<Integer> constantsIterator = constants.listIterator();
			// Write data
			StyledDocument doc = dailyTextPane.getStyledDocument();
			doc.insertString(doc.getLength(), "DAY\t", infoAttrs);
			while (constantsIterator.hasNext()) doc.insertString(doc.getLength(), "B(" + constantsIterator.next() + ")\t", infoAttrs);
			doc.insertString(doc.getLength(), "\n\n", infoAttrs);
			while (daysIterator.hasNext()) {
				Integer day = daysIterator.next();
				doc.insertString(doc.getLength(), day + "\t", textAttrs);
				List<DailyData> dailyData = dailyDataManager.getDailyData(day);
				constantsIterator = constants.listIterator();
				while (constantsIterator.hasNext()) {
					Integer constant = constantsIterator.next();
					boolean notfound = true;
					ListIterator<DailyData> dailyDataIterator = dailyData.listIterator();
					while (dailyDataIterator.hasNext()) {
						DailyData data = dailyDataIterator.next();
						if (data.getConstant() == constant) {
							doc.insertString(doc.getLength(), data.getValue() + "\t", textAttrs);
							notfound = false;
							break;
						}
					}
					if (notfound) doc.insertString(doc.getLength(), "*\t", textAttrs);
				}
				doc.insertString(doc.getLength(), "\n", textAttrs);
			}
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * Loads philogeny data.
	 */
	private void loadSeasonallyData() throws Exception {
		try {
			// Get data
			seasonDataManager = new SimpleSeasonDataManager(seaFile);
			List<Integer> seasons = seasonDataManager.getSeasons();
			Collections.sort(seasons);
			ListIterator<Integer> seasonsIterator = seasons.listIterator();
			List<Integer> constants = seasonDataManager.getReplacingConstants();
			ListIterator<Integer> constantsIterator = constants.listIterator();
			// Write data
			StyledDocument doc = seasonallyTextPane.getStyledDocument();
			doc.insertString(doc.getLength(), "SEASON\t", infoAttrs);
			while (constantsIterator.hasNext()) doc.insertString(doc.getLength(), "B(" + constantsIterator.next() + ")\t", infoAttrs);
			doc.insertString(doc.getLength(), "\n\n", infoAttrs);
			while (seasonsIterator.hasNext()) {
				Integer season = seasonsIterator.next();
				doc.insertString(doc.getLength(), season + "\t", textAttrs);
				List<SeasonData> seasonallyData = seasonDataManager.getSeasonData(season);
				constantsIterator = constants.listIterator();
				while (constantsIterator.hasNext()) {
					Integer constant = constantsIterator.next();
					boolean notfound = true;
					ListIterator<SeasonData> seasonallyDataIterator = seasonallyData.listIterator();
					while (seasonallyDataIterator.hasNext()) {
						SeasonData data = seasonallyDataIterator.next();
						if (data.getConstant() == constant) {
							doc.insertString(doc.getLength(), data.getValue() + "\t", textAttrs);
							notfound = false;
							break;
						}
					}
					if (notfound) doc.insertString(doc.getLength(), "*\t", textAttrs);
				}
				doc.insertString(doc.getLength(), "\n", textAttrs);
			}
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * Loads LAI data.
	 */
	private void loadLaiData() throws Exception {
		try {
			// Get data
			laiDataManager = new SimpleLaiDataManager(laiFile);
			Iterator<CoordinatedLai> laiIterator = laiDataManager.laiIterator();
			// Write data
			laiTextPane = new JTextPane();
			laiTextPane.setAutoscrolls(true);
			laiTextPane.setEditable(false);
			laiTextPane.setBackground(new Color(240, 240, 240));
			laiScrollPane = new JScrollPane(laiTextPane);
			dataTabbedPane.add(laiScrollPane, "Leaf Area index (LAI) values");
			StyledDocument doc = laiTextPane.getStyledDocument();
			doc.insertString(doc.getLength(), "X\tY\tLAI\n\n", infoAttrs);
			while (laiIterator.hasNext()) {
				CoordinatedLai lai = laiIterator.next();
				doc.insertString(doc.getLength(), lai.getX() + "\t" + lai.getY()  + "\t" + lai.getLai() + "\n", textAttrs);
			}
		} 
		catch (Exception e) {
			throw e;
		}
	}
	
	/**
	 * Loads MOD-15 data.
	 * @throws Exception
	 */
	private void loadMod15Data() throws Exception {
		try {
			// Get data
			laiDataManager = new Mod15DataLaiManager(laiFile);
			// Visualize as an image
			Image laiImage = laiDataManager.getLaiImage();
			mod15Pane = new ImagePanel(laiImage);
			mod15Pane.setAutoscrolls(true);
			mod15Pane.setBackground(new Color(240, 240, 240));
			mod15ScrollPane = new JScrollPane(mod15Pane);
			dataTabbedPane.add(mod15ScrollPane, "MOD-15 Leaf Area index (LAI) Map");
		}
		catch (Exception e) {
			throw e;
		}
	}

	/**
	 * Loads Landsat LAI data.
	 * @throws Exception
	 */
	private void loadLandsatData() throws Exception {
		///TODO
	}
	
	private void reset() {
		// reset managers
		climateDataManager = null;
		siteDataManager = null;
		dailyDataManager = null;
		seasonDataManager = null;
		laiDataManager = null;
		// reset files
		clmFile = null;
		datFile = null;
		lifFile = null;
		seaFile = null;
		laiFile = null;
		// reset file texts
		clmFileTextField.setText("");
		datFileTextField.setText("");
		lifFileTextField.setText("");
		seaFileTextField.setText("");
		laiFileTextField.setText("");
		// Select no regional LAI data
		nolaiRadioButton.setSelected(true);
		// Reset data panels
		try {
			clmTextPane.getStyledDocument().remove(0, clmTextPane.getStyledDocument().getLength());
			xTextPane.getStyledDocument().remove(0, xTextPane.getStyledDocument().getLength());
			bTextPane.getStyledDocument().remove(0, bTextPane.getStyledDocument().getLength());
			dailyTextPane.getStyledDocument().remove(0, dailyTextPane.getStyledDocument().getLength());
			seasonallyTextPane.getStyledDocument().remove(0, seasonallyTextPane.getStyledDocument().getLength());
		}
		catch (Exception e){}
		int numTabs = dataTabbedPane.getTabCount();
		for (int i=5; i<numTabs; i++) dataTabbedPane.removeTabAt(5);
		// Outputs check boxes
		resumeOutputCheckBox.setSelected(true);
		dailyOutputCheckBox.setSelected(false);
		longOutputCheckBox.setSelected(false);
		mapOutputCheckBox.setSelected(false);
		mapOutputCheckBox.setEnabled(false);
		// Day spinners
		startdaySpinnerModel.setMaximum(10000);
		startdaySpinnerModel.setValue(1);
		stopdaySpinnerModel.setMaximum(10000);
		stopdaySpinnerModel.setValue(365);
		seasondaysSpinnerModel.setMaximum(10000);
		seasondaysSpinnerModel.setValue(365);
		// Carbon spinner
		carbonRatioSpinnerModel.setValue(0.5);
		// Run button
		runButton.setEnabled(false);
		// Progress bar
		progressBar.setValue(0);
	}
	
	/**
	 * Handle results outputs.
	 * @param results
	 */
	private void giveOutput(ForestBGCResults results, double carbonFixationRatio) {
		// output folder
		if (outputFolder == null){
			String folderOutputPath = null;
			if (userProperties != null) folderOutputPath = userProperties.getProperty("outputFolder");
			if ((folderOutputPath == null) && (defaultProperties != null)) folderOutputPath = defaultProperties.getProperty("outputFolder");
			if (folderOutputPath == null) folderOutputPath = ".";
			outputFolder = new File(folderOutputPath);
		}
		// output manager
		Simlat5ForestBGCOutputManager outputManager = Simlat5ForestBGCOutputManager.getInstance();
		outputManager.setCarbonFixationRatio(carbonFixationRatio);
		// resume output
		if (resumeOutputCheckBox.isSelected()) {
			if (textOutputCheckBox.isSelected()) {
				File resumeFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_resume.txt");
				int i = 0;
				while (resumeFile.exists()) {
					resumeFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_resume_" + i + ".txt");
					i++;
				}
				try {
					if (!resumeFile.createNewFile() || !resumeFile.canWrite()) {
						// Error
						log("Error creating resume output file.", Level.SEVERE);
					}
					// Report
					outputManager.report(results, ForestBGCOutputManager.RESUME_REPORT, resumeFile);
					// log in console
					log("Resume output file created", Level.CONFIG);
				} catch (Exception e) {
					// log in console
					log("Error creating resume output file.", Level.SEVERE);
				}
			}
		}
		// daily output
		if (dailyOutputCheckBox.isSelected()) {
			if (textOutputCheckBox.isSelected()) {
				File dailyFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_daily.txt");
				int i = 0;
				while (dailyFile.exists()) {
					dailyFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_daily_" + i + ".txt");
					i++;
				}
				try {
					if (!dailyFile.createNewFile() || !dailyFile.canWrite()) {
						// Error
						log("Error creating daily output file.", Level.SEVERE);
					}
					// Report
					outputManager.report(results, ForestBGCOutputManager.DAILY_REPORT, dailyFile);
					// log in console
					log("Daily output file created", Level.CONFIG);
				} catch (Exception e) {
					// log in console
					log("Error creating daily output file.", Level.SEVERE);
				}
			}
		}
		// long output
		if (longOutputCheckBox.isSelected()) {
			if (textOutputCheckBox.isSelected()) {
				File longFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_long.txt");
				int i = 0;
				while (longFile.exists()) {
					longFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_long_" + i + ".txt");
					i++;
				}
				try {
					if (!longFile.createNewFile() || !longFile.canWrite()) {
						// Error
						log("Error creating long output file.", Level.SEVERE);
					}
					// Report
					outputManager.report(results, ForestBGCOutputManager.LONG_REPORT, longFile);
					// log in console
					log("Long output file created", Level.CONFIG);
				} catch (Exception e) {
					// log in console
					log("Error creating long output file.", Level.SEVERE);
				}
			}
		}
	}

	/**
	 * Handle results outputs.
	 * @param results
	 */
	private void giveOutput(GlobalForestBGCResults results, double carbonFixationRatio) {
		// output fodler
		if (outputFolder == null){
			String folderOutputPath = null;
			if (userProperties != null) folderOutputPath = userProperties.getProperty("outputFolder");
			if ((folderOutputPath == null) && (defaultProperties != null)) folderOutputPath = defaultProperties.getProperty("outputFolder");
			if (folderOutputPath == null) folderOutputPath = ".";
			outputFolder = new File(folderOutputPath);
		}
		// output manager
		Simlat5ForestBGCOutputManager outputManager = Simlat5ForestBGCOutputManager.getInstance();
		outputManager.setCarbonFixationRatio(carbonFixationRatio);
		// resume output
		if (resumeOutputCheckBox.isSelected()) {
			if (textOutputCheckBox.isSelected()) {
				File resumeFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_resume.txt");
				int i = 0;
				while (resumeFile.exists()) {
					resumeFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_resume_" + i + ".txt");
					i++;
				}
				try {
					if (!resumeFile.createNewFile() || !resumeFile.canWrite()) {
						// Error
						log("Error creating resume output file.", Level.SEVERE);
					}
					// Report
					outputManager.report(results, ForestBGCOutputManager.RESUME_REPORT, resumeFile);
					// log in console
					log("Resume output file created", Level.CONFIG);
				} catch (Exception e) {
					// log in console
					log("Error creating resume output file.", Level.SEVERE);
				}
			}
		}
		// daily output
		if (dailyOutputCheckBox.isSelected()) {
			if (textOutputCheckBox.isSelected()) {
				File dailyFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_daily.txt");
				int i = 0;
				while (dailyFile.exists()) {
					dailyFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_daily_" + i + ".txt");
					i++;
				}
				try {
					if (!dailyFile.createNewFile() || !dailyFile.canWrite()) {
						// Error
						log("Error creating daily output file.", Level.SEVERE);
					}
					// Report
					outputManager.report(results, ForestBGCOutputManager.DAILY_REPORT, dailyFile);
					// log in console
					log("Daily output file created", Level.CONFIG);
				} catch (Exception e) {
					// log in console
					log("Error creating daily output file.", Level.SEVERE);
				}
			}
		}
		// long output
		if (longOutputCheckBox.isSelected()) {
			if (textOutputCheckBox.isSelected()) {
				File longFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_long.txt");
				int i = 0;
				while (longFile.exists()) {
					longFile = new File(outputFolder.getAbsolutePath() + File.separator + results.getSimulationName() + "_long_" + i + ".txt");
					i++;
				}
				try {
					if (!longFile.createNewFile() || !longFile.canWrite()) {
						// Error
						log("Error creating long output file.", Level.SEVERE);
					}
					// Report
					outputManager.report(results, ForestBGCOutputManager.LONG_REPORT, longFile);
					// log in console
					log("Long output file created", Level.CONFIG);
				} catch (Exception e) {
					// log in console
					log("Error creating long output file.", Level.SEVERE);
				}
			}
		}
	}
	
	/**
	 * Panel for displaying an image.
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
	 *
	 */
	private final class ImagePanel extends JPanel {
		
		/**
		 * 
		 */
		private static final long serialVersionUID = 2531069603340756535L;
		
		/** the image */
		private final Image image;

		/**
		 * @param image
		 */
		public ImagePanel(Image image) {
			this.image = image;
		}

		/* (non-Javadoc)
		 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
		 */
		@Override
		protected void paintComponent(Graphics g) {
			super.paintComponent(g);
			int h = image.getHeight(null);
			int w = image.getWidth(null);
			int hh = this.getHeight();
			int ww = this.getWidth();
			int x = 0; int y = 0;
			if (ww - w > 0) x = (ww - w)/2;
			else w = ww;
			if (hh - h > 0) x = (hh - h)/2;
			else h = hh;
			g.drawImage(image, x, y, w, h, null);
		}
		
	}

}