/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.inputs;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Interface for managing climate data needed for the ForestBGC ecophysiological model.
 */
public interface ClimateDataManager {

	// Parameters
	/** Max. Temperature */
	public static final int T_MAX = 0; 
	/** Min. Temperature */
	public static final int T_MIN = 1; 
	/** Dew point */
	public static final int DEW_POINT = 2; 
	/** Day lenght */
	public static final int DAY_LENGTH = 3; 
	/** Precision */
	public static final int PRECISION = 4;
	/** Num of climatic parameters */
	public static final int NUM_PARAMETERS = 5;
	
	/**
	 * Get the all days data for the given parameter.
	 * @param parameter the parameter to get.
	 * @return the data as a vector ordered by day.
	 * @throws Exception if data cannot be obtained.
	 */
	public Double[] getData(int parameter) throws Exception;
	
	/**
	 * Get the data for the given parameter at the given day.
	 * @param parameter the parameter to get.
	 * @param day the day of interest (starting by day 1).
	 * @return the data.
	 * @throws Exception if data cannot be obtained.
	 */
	public Double getData(int parameter, int day) throws Exception;

	/**
	 * Get the values of all the parameters for the given day.
	 * @param day
	 * @return
	 * @throws Exception if data cannot be obtained.
	 */
	public Double[] getAllData(int day) throws Exception;
	
	/**
	 * Set the all days data for the given parameter.
	 * @param parameter the parameter to get.
	 * @param data the data to set ordered by day. The data size has to be equal to the number of days.
	 * @throws Exception if data cannot be set.
	 */
	public void setData(int parameter, Double[] data) throws Exception;
	
	/**
	 * Set the data for the given parameter at the given day.
	 * @param parameter the parameter to get.
	 * @param day the day of interest (starting by day 1).
	 * @param data the data.
	 * @throws Exception if data cannot be obtained.
	 */
	public void setData(int parameter, int day, Double data) throws Exception;

	/**
	 * Get the number of days for which there are climate data.
	 * @return the number of days.
	 */
	public int getNumDays();

}