/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional.
 * NPP software reimplements part of the code provided by SIMLAT5 v3.6 under Copyright 1998 J.C. Coughlan.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.inputs;

import java.awt.Image;
import java.util.Iterator;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Interface for managing Leaf Area Index data.
 */
public interface LaiDataManager {

	/**
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
	 * Coordinates-based Leaf Area Index data.
	 *
	 */
	public class CoordinatedLai implements Comparable<CoordinatedLai> {

		/** Horizontal spatial coordinate (starts in 0) */
		private final Integer y;
		
		/** Vertical spatial coordinate (starts in 0) */
		private final Integer x;
		
		/** LAI value */
		private final Double lai;

		/**
		 * Constructor.
		 * @param y Horizontal spatial coordinate
		 * @param x Vertical spatial coordinate
		 * @param lai LAI value
		 */ 
		public CoordinatedLai(Integer y, Integer x, Double lai) {
			this.y = y;
			this.x = x;
			this.lai = lai;
		}

		/**
		 * @return the y
		 */
		public final Integer getY() {
			return y;
		}

		/**
		 * @return the x
		 */
		public final Integer getX() {
			return x;
		}

		/**
		 * @return the lai
		 */
		public final Double getLai() {
			return lai;
		}

		/**
		 * Returns <code>true</code> if this <code>LAI</code> is the same as the o argument.
		 *
		 * @return <code>true</code> if this <code>LAI</code> is the same as the o argument.
		 */
		public boolean equals(Object o) {
			if (this == o) {
				return true;
			}
			if (o == null) {
				return false;
			}
			if (o.getClass() != getClass()) {
				return false;
			}
			CoordinatedLai castedObj = (CoordinatedLai) o;
			return ((this.x == null ? castedObj.x == null : this.x.equals(castedObj.x))
					&& (this.y == null ? castedObj.y == null : this.y
							.equals(castedObj.y)) && (this.lai == null ? castedObj.lai == null
					: this.lai.equals(castedObj.lai)));
		}

		/* (non-Javadoc)
		 * @see java.lang.Comparable#compareTo(java.lang.Object)
		 */
		public int compareTo(CoordinatedLai o) {
			if (this.getY() < o.getY()) return -1;
			if (this.getY() > o.getY()) return 1;
			if (this.getX() < o.getX()) return -1;
			if (this.getX() > o.getX()) return 1;
			return 0;
		}
		
	}

	/**
	 * Gets the iterator to the LAI objects
	 * @return Iterator
	 * @throws Exception 
	 */
	public Iterator<CoordinatedLai> laiIterator() throws Exception;

	/**
	 * Get the number of LAI values.
	 * @return int
	 */
	public int getNumData() throws Exception;
	
	/**
	 * Gets an image of the Coordinated LAI data.
	 * @return an image
	 * @throws Exception
	 */
	public Image getLaiImage() throws Exception;

}