/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.inputs;

import java.awt.Image;
import java.awt.Toolkit;
import java.awt.image.IndexColorModel;
import java.awt.image.MemoryImageSource;
import java.io.File;
import java.util.Iterator;
import java.util.ListIterator;

import javax.swing.tree.DefaultMutableTreeNode;

import ncsa.hdf.object.Dataset;
import ncsa.hdf.object.FileFormat;
import ncsa.hdf.object.Group;
import ncsa.hdf.object.h4.H4File;
import ncsa.hdf.object.h4.H4SDS;
import ncsa.hdf.object.h5.H5File;
import ncsa.hdf.object.h5.H5ScalarDS;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Implementation of LaiDataManager to handle MOD-15 data files.
 */
public class Mod15DataLaiManager implements LaiDataManager {

	/** Data allocation */
	private byte[] lai = null;
	
	// Size of dimensions
	private long[] dims;
	
	// Scale factor for LAI value
	private double scaleFactor = 0.1;
	
	/**
	 * Constructor.
	 * @param file file with MOD-15 data.
	 * @throws FileErrorException if the file has missing data or is not a MOD-15 datafile.
	 */
	@SuppressWarnings("unchecked")
	public Mod15DataLaiManager(File file) throws Exception {
		FileFormat fileFormat = null;
		boolean h4 =true;
		// open the file
		try {
			fileFormat = H4File.getInstance(file.getAbsolutePath());
		}
		catch (Exception e1) {
			try {
				h4 = false;
				fileFormat = H5File.getInstance(file.getAbsolutePath());
			}
			catch (Exception e2) {throw new Exception("Cannot load MOD-15 file: " + file.getAbsolutePath());}
		}
		if (fileFormat == null) throw new Exception("Cannot load MOD-15 file: " + file.getAbsolutePath());
		fileFormat.open();
		// load data
		DefaultMutableTreeNode root = (DefaultMutableTreeNode)fileFormat.getRootNode();
		if (root == null) throw new Exception("Cannot load MOD-15 file: " + file.getAbsolutePath());
		Group group = (Group)((Group)root.getUserObject()).getMemberList().get(0);
		ListIterator groupIt = group.getMemberList().listIterator();
		while (groupIt.hasNext()) {
			Group member = (Group)groupIt.next();
			if (member.getName().equalsIgnoreCase("data fields")) {
				ListIterator memberIt = member.getMemberList().listIterator();
				while (memberIt.hasNext()) {
					// Find LAI layer
					Dataset dataset = (Dataset)memberIt.next();
					if (dataset.getName().equalsIgnoreCase("Lai_1km")) {
						if (h4) {
							lai = (byte[])((H4SDS)dataset).getData();
						}
						else lai = (byte[])((H5ScalarDS)dataset).getData();
						dims = dataset.getDims();
						break;
					}
				}
				break;
			}
		}
		// close file
		fileFormat.close();
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager#getNumData()
	 */
	public int getNumData() throws Exception {
		if (lai == null) throw new Exception("No data available.");
		return lai.length;
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager#laiIterator()
	 */
	public Iterator<CoordinatedLai> laiIterator() throws Exception {
		if (lai == null) throw new Exception("No data available.");
		return new Mod15Iterator<CoordinatedLai>();
	}
	
	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager#getLaiImage()
	 */
	public Image getLaiImage() throws Exception {
		int[] pixels = new int[lai.length];
		for (int i=0; i<lai.length; i++) {
			int c = (0xff) & lai[i];
			pixels[i] = ((0xff<<24) | (c<<16) | (c<<8) | c);
		}
		return Toolkit.getDefaultToolkit().createImage(new MemoryImageSource((int)dims[0],(int)dims[1],getIndexColorModel(),pixels,0,(int)dims[0]));
	}

	/**
	 * Return the indexColorSpace scale for lai values as an image.
	 * @param width
	 * @param height
	 * @return
	 * @throws Exception
	 */
	public Image getColorScale(int width, int height) throws Exception {
		int[] pixels = new int[width*height];
		for (int i=0; i<height; i++) {
			for (int j=0; j<width; j++) {
				int c = 255*j/width;
				pixels[i*height + j] = ((0xff<<24) | (c<<16) | (c<<8) | c);
			}
		}
		return Toolkit.getDefaultToolkit().createImage(new MemoryImageSource((int)dims[0],(int)dims[1],getIndexColorModel(),pixels,0,(int)dims[0]));		
	}
	
	/**
	 * Get a coordinated lai by its index.
	 * @param index
	 * @return
	 */
	public CoordinatedLai getCoordinatedLai(int index) {
		int y = index / (int)dims[0];
		int x = index % (int)dims[0];
		int laiValue = (int)(lai[index] & 0xFF);
		if (laiValue > 100) return null;
		else return new CoordinatedLai(y, x, ((double)laiValue)*scaleFactor);
	}

	/**
	 * Gets an IndexColorModel from a lai value (0-100). If lai value is more than 100 -> black.
	 * @return
	 */
	private IndexColorModel getIndexColorModel() {
		byte[] r = new byte[256];
        byte[] g = new byte[256];
        byte[] b = new byte[256];
        for (int i=0; i<101; i++) {
        	if (4*i > 255) r[i] = 0;
        	else r[i] = (byte)(255 - 4*i);
        	if (4*(100-i) > 255) b[i] = 0;
        	else b[i] = (byte)(255 - 4*(100-i));
        	if (i<50) {
	        	if (6*(50-i) > 255) g[i] = 0;
	        	else g[i] = (byte)(255 - 6*(50-i));
    		}
        	else {
	        	if (6*(i-50) > 255) g[i] = 0;
	        	else g[i] = (byte)(255 - 6*(i-50));
        	}
        }
        for (int i=101; i<256; i++) {
        	r[i] = 0;
        	g[i] = 0;
        	b[i] = 0;
        }
		return new IndexColorModel(8,256,r,g,b);	
	}

	/**
	 * Efficient iterator for MOD-15 LAI data.
	 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
	 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
	 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
	 *
	 */
	private final class Mod15Iterator<E> implements Iterator<CoordinatedLai> {

		/** Index */
		private int index = 0;
		
		/**
		 *  Constructor.
		 */
		public Mod15Iterator() {
			this.index = 0;
		}

		/**
		 * Constructor
		 * @param index
		 */
		public Mod15Iterator(int index) {
			this.index = index;
		}

		/**
		 * Can return false if it gets out of memory!!!
		 */
		public boolean hasNext() {
			return index < lai.length;
		}

		/**
		 * Can return null if it gets out of memory!!!
		 */
		public CoordinatedLai next() {
			CoordinatedLai coordinatedLai = getCoordinatedLai(index);
			index++;
			return coordinatedLai;
		}

		/**
		 * Not implemeted!!
		 */
		public void remove() {
			// NOT USED
		}
	}	
}
