/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.inputs;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Implementation of SiteDataManager to handle the site files used by Simlat5.
 */
public final class Simlat5SiteDataManager extends AbstractSiteDataManager {

	/** Data allocation */
	private Double[] data;
	
	/** Static/Dynamic budget */
	private boolean staticBudget;
	
	/**
	 * Constructor.
	 * @param file The file containing the site data in a Simlat5 format
	 */
	public Simlat5SiteDataManager(File file) throws Exception {
		// Initialize data allocation
		data = new Double[AbstractSiteDataManager.NUM_PARAMETERS];
		// Read input file
		try {
			// Check file
			if (!file.isFile()) throw new Exception("Provided file is not a file");
			else {
				// Check if file is readable
				if (!file.canRead()) throw new Exception("Provided file is not readable");
				BufferedReader in = new BufferedReader(new FileReader(file));
				// Line readed
				String line; String[] s;
				// Header
				in.readLine(); // COMMENT LINE
				in.readLine(); // CONT'D
				// START TIME
				line = in.readLine();
				if (line == null) throw new Exception("Provided file data are not valid");
				s = line.split(" ");
				data[AbstractSiteDataManager.START_TIME] = Double.valueOf(s[0]);
				// STOP TIME
				line = in.readLine();
				if (line == null) throw new Exception("Provided file data are not valid");
				s = line.split(" ");
				data[AbstractSiteDataManager.STOP_TIME] = Double.valueOf(s[0]);
				// Skip lines
				in.readLine(); // DAY
				in.readLine(); // GRW
				in.readLine(); // LNG
				in.readLine(); // KPRINT
				in.readLine(); // KBEGIN
				// LOOP TIME
				line = in.readLine();
				if (line == null) throw new Exception("Provided file data are not valid");
				s = line.split(" ");
				data[AbstractSiteDataManager.LOOP_TIME] = Double.valueOf(s[0]);
				// Skip lines
				in.readLine(); // LIFE CYCLE
				in.readLine(); // SEASONALLY
				in.readLine(); // NEGX
				in.readLine(); // NUMX
				// State variables
				for (int i=0; i<20; i++) {
					line = in.readLine();
					if (line == null) throw new Exception("Provided file data are not valid");
					s = line.split(" ");
					switch (i+1) {
					case 1: data[AbstractSiteDataManager.X_1] = Double.valueOf(s[0]); break;
					case 2: data[AbstractSiteDataManager.X_2] = Double.valueOf(s[0]); break;
					case 3: data[AbstractSiteDataManager.X_3] = Double.valueOf(s[0]); break;
					case 4: data[AbstractSiteDataManager.X_4] = Double.valueOf(s[0]); break;
					case 5: data[AbstractSiteDataManager.X_5] = Double.valueOf(s[0]); break;
					case 6: data[AbstractSiteDataManager.X_6] = Double.valueOf(s[0]); break;
					case 7: data[AbstractSiteDataManager.X_7] = Double.valueOf(s[0]); break;
					case 8: data[AbstractSiteDataManager.X_8] = Double.valueOf(s[0]); break;
					case 9: data[AbstractSiteDataManager.X_9] = Double.valueOf(s[0]); break;
					case 10: data[AbstractSiteDataManager.X_10] = Double.valueOf(s[0]); break;
					case 11: data[AbstractSiteDataManager.X_11] = Double.valueOf(s[0]); break;
					case 12: data[AbstractSiteDataManager.X_12] = Double.valueOf(s[0]); break;
					case 13: data[AbstractSiteDataManager.X_13] = Double.valueOf(s[0]); break;
					case 14: data[AbstractSiteDataManager.X_14] = Double.valueOf(s[0]); break;
					case 15: data[AbstractSiteDataManager.X_15] = Double.valueOf(s[0]); break;
					case 16: data[AbstractSiteDataManager.X_16] = Double.valueOf(s[0]); break;
					case 17: data[AbstractSiteDataManager.X_17] = Double.valueOf(s[0]); break;
					case 18: data[AbstractSiteDataManager.X_18] = Double.valueOf(s[0]); break;
					case 19: data[AbstractSiteDataManager.X_19] = Double.valueOf(s[0]); break;
					case 20: data[AbstractSiteDataManager.X_20] = Double.valueOf(s[0]); break;
					}
				}
				// Skip line
				in.readLine(); // NUMB
				// Site constants
				for (int i=0; i<45; i++) {
					line = in.readLine();
					if (line == null) throw new Exception("Provided file data are not valid");
					s = line.split(" ");
					switch (i+1) {
					case 1: data[AbstractSiteDataManager.B_1] = Double.valueOf(s[0]); break;
					case 2: data[AbstractSiteDataManager.B_2] = Double.valueOf(s[0]); break;
					case 3: data[AbstractSiteDataManager.B_3] = Double.valueOf(s[0]); break;
					case 4: data[AbstractSiteDataManager.B_4] = Double.valueOf(s[0]); break;
					case 5: data[AbstractSiteDataManager.B_5] = Double.valueOf(s[0]); break;
					case 6: data[AbstractSiteDataManager.B_6] = Double.valueOf(s[0]); break;
					case 7: data[AbstractSiteDataManager.B_7] = Double.valueOf(s[0]); break;
					case 8: data[AbstractSiteDataManager.B_8] = Double.valueOf(s[0]); break;
					case 9: data[AbstractSiteDataManager.B_9] = Double.valueOf(s[0]); break;
					case 10: data[AbstractSiteDataManager.B_10] = Double.valueOf(s[0]); break;
					case 11: data[AbstractSiteDataManager.B_11] = Double.valueOf(s[0]); break;
					case 12: data[AbstractSiteDataManager.B_12] = Double.valueOf(s[0]); break;
					case 13: data[AbstractSiteDataManager.B_13] = Double.valueOf(s[0]); break;
					case 14: data[AbstractSiteDataManager.B_14] = Double.valueOf(s[0]); break;
					case 15: data[AbstractSiteDataManager.B_15] = Double.valueOf(s[0]); break;
					case 16: data[AbstractSiteDataManager.B_16] = Double.valueOf(s[0]); break;
					case 17: data[AbstractSiteDataManager.B_17] = Double.valueOf(s[0]); break;
					case 18: data[AbstractSiteDataManager.B_18] = Double.valueOf(s[0]); break;
					case 19: data[AbstractSiteDataManager.B_19] = Double.valueOf(s[0]); break;
					case 20: data[AbstractSiteDataManager.B_20] = Double.valueOf(s[0]); break;
					case 21: data[AbstractSiteDataManager.B_21] = Double.valueOf(s[0]); break;
					case 22: data[AbstractSiteDataManager.B_22] = Double.valueOf(s[0]); break;
					case 23: data[AbstractSiteDataManager.B_23] = Double.valueOf(s[0]); break;
					case 24: data[AbstractSiteDataManager.B_24] = Double.valueOf(s[0]); break;
					case 25: data[AbstractSiteDataManager.B_25] = Double.valueOf(s[0]); break;
					case 26: data[AbstractSiteDataManager.B_26] = Double.valueOf(s[0]); break;
					case 27: data[AbstractSiteDataManager.B_27] = Double.valueOf(s[0]); break;
					case 28: data[AbstractSiteDataManager.B_28] = Double.valueOf(s[0]); break;
					case 29: data[AbstractSiteDataManager.B_29] = Double.valueOf(s[0]); break;
					case 30: data[AbstractSiteDataManager.B_30] = Double.valueOf(s[0]); break;
					case 31: data[AbstractSiteDataManager.B_31] = Double.valueOf(s[0]); break;
					case 32: data[AbstractSiteDataManager.B_32] = Double.valueOf(s[0]); break;
					case 33: data[AbstractSiteDataManager.B_33] = Double.valueOf(s[0]); break;
					case 34: data[AbstractSiteDataManager.B_34] = Double.valueOf(s[0]); break;
					case 35: data[AbstractSiteDataManager.B_35] = Double.valueOf(s[0]); break;
					case 36: data[AbstractSiteDataManager.B_36] = Double.valueOf(s[0]); break;
					case 37: data[AbstractSiteDataManager.B_37] = Double.valueOf(s[0]); break;
					case 38: data[AbstractSiteDataManager.B_38] = Double.valueOf(s[0]); break;
					case 39: data[AbstractSiteDataManager.B_39] = Double.valueOf(s[0]); break;
					case 40: data[AbstractSiteDataManager.B_40] = Double.valueOf(s[0]); break;
					case 41: data[AbstractSiteDataManager.B_41] = Double.valueOf(s[0]); break;
					case 42: data[AbstractSiteDataManager.B_42] = Double.valueOf(s[0]); break;
					case 43: data[AbstractSiteDataManager.B_43] = Double.valueOf(s[0]); break;
					case 44: data[AbstractSiteDataManager.B_44] = Double.valueOf(s[0]); break;
					case 45: data[AbstractSiteDataManager.B_45] = Double.valueOf(s[0]); break;
					}
				}
				// Site constants (Gower model)
				staticBudget = false;
				for (int i=45; i<50; i++) {
					line = in.readLine();
					if (line == null || line.contains("NUMIGP") || line.contains("NUMIZP")) {
						data[AbstractSiteDataManager.B_46] = null;
						data[AbstractSiteDataManager.B_47] = null;
						data[AbstractSiteDataManager.B_48] = null;
						data[AbstractSiteDataManager.B_49] = null;
						data[AbstractSiteDataManager.B_50] = null;
						staticBudget = true;
						break;
					}
					else s = line.split(" ");
					switch (i+1) {
					case 46: data[AbstractSiteDataManager.B_46] = Double.valueOf(s[0]); break;
					case 47: data[AbstractSiteDataManager.B_47] = Double.valueOf(s[0]); break;
					case 48: data[AbstractSiteDataManager.B_48] = Double.valueOf(s[0]); break;
					case 49: data[AbstractSiteDataManager.B_49] = Double.valueOf(s[0]); break;
					case 50: data[AbstractSiteDataManager.B_50] = Double.valueOf(s[0]); break;
					}
				}
			}
		}
		catch (Exception e) {throw new Exception("Exception ocurred: " + e.getMessage());}			
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SiteDataManager#getData(int)
	 */
	public Double getData(int parameter) throws Exception {
		try {
			return data[parameter];
		}
		catch (Exception e) {throw new Exception("Unexisting parameter");}
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SiteDataManager#setData(int, java.lang.Double)
	 */
	public void setData(int parameter, Double value) throws Exception {
		try {
			this.data[parameter] = value;
		}
		catch (Exception e) {throw new Exception("Unexisting parameter");}
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SiteDataManager#isStaticBudget()
	 */
	public boolean isStaticBudget() {
		return staticBudget;
	}

}
