/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.inputs;

import java.awt.Image;
import java.awt.Toolkit;
import java.awt.image.MemoryImageSource;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Implementation of LaiDataManager to handle files with coordinates-based LAI values.
 */
public class SimpleLaiDataManager implements LaiDataManager {

	/** Data allocation */
	private List<CoordinatedLai> data;
	
	/** Minimum and maximum ccordinates */
	int minX, minY, maxX, maxY;

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager#getNumData()
	 */
	public int getNumData() {
		return data.size();
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager#laiIterator()
	 */
	public Iterator<CoordinatedLai> laiIterator() {
		return data.iterator();
	}

	/**
	 * Constructor.
	 * @param file file with LAI data.
	 * @throws FileErrorException if the file has missing data or is not a LAI datafile.
	 */
	public SimpleLaiDataManager(File file) throws Exception {
		// Initialize data allocation
		data = new ArrayList<CoordinatedLai>();
		// Read input file
		try {
			// Check file
			if (!file.isFile()) throw new Exception("Provided file is not a file");
			else {
				// Check if file is readable
				if (!file.canRead()) throw new Exception("Provided file is not readable");
				BufferedReader in = new BufferedReader(new FileReader(file));
				// Read data
				String line = in.readLine();
				String[] s;
				minX = Integer.MAX_VALUE; minY = Integer.MAX_VALUE;
				maxX = Integer.MIN_VALUE; maxY = Integer.MIN_VALUE;
				while (line != null){
					s = line.split("\t");
					if (s.length != 3) {
						s = line.split(" ");
						if (s.length != 3) {
							throw new Exception("Provided file data are not valid");
						}
					}
					else {
						int y = Integer.valueOf(s[0]);
						int x = Integer.valueOf(s[1]);
						if (y < minY) minY = y;
						if (y > maxY) maxY = y;
						if (x < minX) minX = x;
						if (x > maxX) maxX = x;
						data.add(new CoordinatedLai(y, x, Double.valueOf(s[2])));
					}
					line = in.readLine();
				}
			}
			Collections.sort(data);
		}
		catch (Exception e) {throw new Exception("Exception ocurred: " + e.getMessage());}
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.LaiDataManager#getLaiImage()
	 */
	public Image getLaiImage() throws Exception {
		int[] pixels = new int[(maxX - minX)*(maxY - minY)];
		ListIterator<CoordinatedLai> it = data.listIterator();
		// indíces
		int x=0; int y=0;
		while (it.hasNext()) {
			CoordinatedLai clai = it.next();
			for (int i=y; i<(clai.getY()-minY); i++) {
				for (int j=0; j<(maxX - minX); j++) {
					pixels[i*(maxX - minX)+j] = 0;
				}
				y++;
			}
			for (int i=x; i<(clai.getX()-minX); i++) {
				pixels[y*(maxX - minX)+i] = 0;
				x++;
			}
			int c = (0xff) & clai.getLai().byteValue();
			pixels[y*(maxX-minX)+x] = ((0xff<<24) | (c<<16) | (c<<8) | c);
			x++;
			if (x == (maxX-minX)) {x = 0; y++;}
		}
		return Toolkit.getDefaultToolkit().createImage(new MemoryImageSource((maxX-minX),(maxY-minY),pixels,0,(maxX-minX)));
	}
	
}
