/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional.
 * NPP software reimplements part of the code provided by SIMLAT5 v3.6 under Copyright 1998 J.C. Coughlan.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.inputs;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Implementation of SeasonDataManager to manage site information season data from a simple file.
 */
public final class SimpleSeasonDataManager implements SeasonDataManager {

	/** Data */
	private SortedMap<Integer,List<SeasonData>> data;
	
	/** Constants to be replaced */
	private List<Integer> constants;
	
	/** Constructor */
	public SimpleSeasonDataManager(File file) throws Exception {
		// Initialize data allocation
		data = new TreeMap<Integer,List<SeasonData>>();
		// Read input file
		try {
			// Check file
			if (!file.isFile()) throw new Exception("Provided file is not a file");
			else {
				// Check if file is readable
				if (!file.canRead()) throw new Exception("Provided file is not readable");
				BufferedReader in = new BufferedReader(new FileReader(file));
				// Lines and words
				String line; String[] s;
				// Read comment ".sea comment line"
				line = in.readLine();
				if (line == null) throw new Exception("Empty or invalid file");
				// Read variables
				line = in.readLine();
				if (line == null) throw new Exception("Empty or invalid file");
				s = line.trim().split("\\s+");
				boolean tabular = false;
				try {
					if (s == null || Integer.parseInt(s[0]) > 50 || Integer.parseInt(s[0]) < 1) tabular = true;
				}
				catch (NumberFormatException e){tabular = true;}
				if (tabular) {
					s = line.split("\t");
					try {
						if (s == null || Integer.parseInt(s[0]) > 50 || Integer.parseInt(s[0]) < 1) throw new Exception("Invalid file");
					}
					catch (NumberFormatException e){throw new Exception("Invalid file");}
				}
				constants = new ArrayList<Integer>();
				for (int i=0; i<s.length; i++) constants.add(Integer.parseInt(s[i]));
				// Read data
				line = in.readLine();
				while (line != null){
					if (tabular) s = line.split("\t");
					else s = line.trim().split("\\s+");
					if (s.length != constants.size() + 1) throw new Exception("Provided file data are not valid");
					// Day
					Integer season;
					try {
						season = Integer.parseInt(s[0]);
					}
					catch (NumberFormatException e){throw new Exception("Invalid file");}
					// Variables
					List<SeasonData> seasonDataList = new ArrayList<SeasonData>();
					for (int i = 1; i < s.length; i++) {
						if (s[i].equalsIgnoreCase("*"));
						else {
							try {
								SeasonData seasonData = new SeasonData(constants.get(i-1), Double.parseDouble(s[i]));
								seasonDataList.add(seasonData);
							}
							catch (NumberFormatException e){throw new Exception("Invalid file");}
						}
					}
					data.put(season, seasonDataList);
					line = in.readLine();
				}
			}
		}
		catch (Exception e) {throw new Exception("Exception ocurred: " + e.getMessage());}
	}
	
	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.DailyDataManager#getSeasonData()
	 */
	public List<SeasonData> getSeasonData(int season) {
		return data.get(new Integer(season));
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SeasonDataManager#getSeasons()
	 */
	public List<Integer> getSeasons() {
		return new ArrayList<Integer>(data.keySet());
	}

	/* (non-Javadoc)
	 * @see es.ehu.www.ccwintco.npp.core.forestbgc.inputs.SeasonDataManager#getReplacingConstants()
	 */
	public List<Integer> getReplacingConstants() {
		return constants;
	}

}
