/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.outputs;

import java.io.File;
import java.util.List;

import es.ehu.www.ccwintco.npp.core.MultiplePointsNppResults;
import es.ehu.www.ccwintco.npp.core.OutputManager;
import es.ehu.www.ccwintco.npp.core.SinglePointNppResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.ForestBGCResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.GlobalForestBGCResults;


/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Abstract class that implements the OuputManager for Forest-BGC.
 * Provides three different kind of reports similar to the ones provided by Simlat5: resume, long and daily.
 * The reported variables can be modified by the correponding methods.
 */
public abstract class ForestBGCOutputManager implements OutputManager {

	/** Resume report */
	public static final int RESUME_REPORT = 0; 

	/** Long report */
	public static final int LONG_REPORT = 1;
	
	/** Daily report */
	public static final int DAILY_REPORT = 2; 
	
	/**
	 * List of state variables (X) of the Forest-BGC model that are going to be added to the report. 
	 */
	private List<Integer> stateVariablesReported = null;

	/**
	 * List of G variables (G) of the Forest-BGC model that are going to be added to the report. 
	 */
	private List<Integer> gVariablesReported = null;
	
	/** Carbon fixation ratio */
	private double carbonFixationRatio = 0.5;

	/**
	 * @return the stateVariablesReported
	 */
	public final List<Integer> getStateVariablesReported() {
		return stateVariablesReported;
	}

	/**
	 * @param stateVariablesReported the stateVariablesReported to set
	 */
	public final void setStateVariablesReported(List<Integer> stateVariablesReported) {
		this.stateVariablesReported = stateVariablesReported;
	}

	/**
	 * @return the gVariablesReported
	 */
	public final List<Integer> getGVariablesReported() {
		return gVariablesReported;
	}

	/**
	 * @param variablesReported the gVariablesReported to set
	 */
	public final void setGVariablesReported(List<Integer> variablesReported) {
		gVariablesReported = variablesReported;
	}

	/**
	 * @return the carbonFixationRatio
	 */
	public final double getCarbonFixationRatio() {
		return carbonFixationRatio;
	}

	/**
	 * @param carbonFixationRatio the carbonFixationRatio to set. Must be a value in the range [0.0 1.0]. If not, default value (0.5) should be considered.
	 */
	public final void setCarbonFixationRatio(double carbonFixationRatio) {
		if (carbonFixationRatio < 0.0 || carbonFixationRatio > 1.0) carbonFixationRatio = 0.5;
		else this.carbonFixationRatio = carbonFixationRatio;
	}

	/**
	 * Reports the ForestBGC output into the provided file.
	 * @param results the result to be reported.
	 * @param the kind of report.
	 * @param file the file where the report is writed. It has to be an existing and writable file. The report is appended to the end of the file.
	 * @throws Exception
	 */
	public abstract void report(ForestBGCResults results, int report, File file) throws Exception;
	
	/**
	 * Reports the ForestBGC output into the provided file.
	 * @param results the result to be reported.
	 * @param the kind of report.
	 * @param file the file where the report is writed. It has to be an existing and writable file. The report is appended to the end of the file.
	 * @throws Exception
	 */
	public abstract void report(GlobalForestBGCResults results, int report, File file) throws Exception;

	@Override
	/**
	 * Check if results correspond to Forest-BGC results and call for a resumed report.
	 */
	public void report(SinglePointNppResults results, File file)
			throws Exception {
		if (results == null || results.getClass() != ForestBGCResults.class) throw new Exception("Error converting parameter 'results': " + results.toString());
		try {
			report((ForestBGCResults)results, ForestBGCOutputManager.RESUME_REPORT, file);
		}
		catch (Exception e) {throw e;}
	}

	@Override
	/**
	 * Check if results correspond to Forest-BGC results and call for a resumed report.
	 */
	public void report(MultiplePointsNppResults results, File file)
			throws Exception {
		if (results == null || results.getClass() != GlobalForestBGCResults.class) throw new Exception("Error converting parameter 'results': " + results.toString());
		try {
			report((GlobalForestBGCResults)results, ForestBGCOutputManager.RESUME_REPORT, file);
		}
		catch (Exception e) {throw e;}
	}

}
