/**
 * Copyright 2009 Miguel Angel Veganzones, Grupo Inteligencia Computacional, Universidad del País Vasco (UPV/EHU).
 * NPP software provides different Net Primary Production (NPP) estimators.
 * 
 * This file is part of NPP software.
 *
 * NPP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NPP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NPP. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package es.ehu.www.ccwintco.npp.core.forestbgc.outputs;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.text.DecimalFormat;
import java.util.List;
import java.util.ListIterator;

import es.ehu.www.ccwintco.npp.core.forestbgc.CycleResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.DailyResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.ForestBGCResults;
import es.ehu.www.ccwintco.npp.core.forestbgc.GlobalForestBGCResults;


/**
 * @author Miguel A. Veganzones <miguelangel.veganzones@ehu.es>
 * @author Grupo Inteligencia Computacional <http://www.ehu.es/computationalintelligence>
 * @author Universidad del País Vasco (UPV/EHU) <http://www.ehu.es>
 * Implements the OuputManager for Forest-BGC simple (text files) reports.
 * Provides three different kind of reports similar to the ones provided by Simlat5: resume, long and daily.
 * For each kind of report there are some default variables that are reported, but it is possible to report 
 * additional variables using the methods provided by ForestBGCOutputManager.
 */
public final class Simlat5ForestBGCOutputManager extends ForestBGCOutputManager {

	/** Singleton pattern */
	private static final Simlat5ForestBGCOutputManager instance = new Simlat5ForestBGCOutputManager();
	
	/**
	 * @return the instance
	 */
	public static final Simlat5ForestBGCOutputManager getInstance() {
		return instance;
	}

	/**
	 * Constructor 
	 */
	private Simlat5ForestBGCOutputManager() {
	}

	/**
	 * Reports the ForestBGC output into the provided file.
	 * @param results the result to be reported.
	 * @param the kind of report.
	 * @param file the file where the report is writed. It has to be an existing and writable file. The report is appended to the end of the file.
	 * @throws Exception
	 */
	public void report(ForestBGCResults results, int report, File file) throws Exception {
		if (file == null || !file.exists() || !file.isFile() || !file.canWrite()) throw new Exception("File " + file.getAbsolutePath() + " is not valid");
		BufferedWriter writer = new BufferedWriter(new FileWriter(file));
		try {
			if (results == null) throw new Exception("Null results found");
			// some variables
			ListIterator<CycleResults> cycleIt;
			ListIterator<DailyResults> dailyIt;
			int season; int day;
			DecimalFormat formatter = new DecimalFormat("#.#");
			List<Integer> xAdditional = this.getStateVariablesReported();
			List<Integer> gAdditional = this.getGVariablesReported();
			// report header
			writer.append("Simulation: " + results.getSimulationName() + "\n");
			writer.append("Date: " + results.getSimulationDate().toString() + "\n\n");
			writer.append(results.getEstimator().getVersion() + "\n");
			writer.append(results.getEstimator().getCopyright() + "\n\n");
			writer.append("---------------------------------------------------------\n\n");
			switch(report) {
			case ForestBGCOutputManager.RESUME_REPORT:
				writer.append("\tX4\t");
				writer.append("X6\t");
				writer.append("X8\t");
				writer.append("X9\t");
				if (xAdditional != null && !xAdditional.isEmpty()) {
					ListIterator<Integer> it = xAdditional.listIterator();
					while (it.hasNext()) {
						int value = it.next().intValue();
						if (value != 4 && value != 6 && value != 8 && value != 9)
							writer.append("X" + value + "\t");
					}
				}
				writer.append("G58\t");
				writer.append("G59\t");
				writer.append("G66\t");
				writer.append("G100");
				if (gAdditional != null && !gAdditional.isEmpty()) {
					writer.append("\t");
					ListIterator<Integer> it = xAdditional.listIterator();
					while (it.hasNext()) {
						int value = it.next().intValue();
						if (value != 58 && value != 59 && value != 66 && value != 100)
							writer.append("G" + value + "\t");
					}
				}
				writer.append("\n");
				writer.append("Season\t");
				writer.append("Trans.\t");
				writer.append("PSN\t");
				writer.append("Leaf C\t");
				writer.append("Stem C\t");
				if (xAdditional != null && !xAdditional.isEmpty()) {
					ListIterator<Integer> it = xAdditional.listIterator();
					while (it.hasNext()) {
						int value = it.next().intValue();
						if (value != 4 && value != 6 && value != 8 && value != 9)
							writer.append("\t");
					}
				}
				writer.append("S_grw\t");
				writer.append("R_grw\t");
				writer.append("g_rsp\t");
				writer.append("m_rsp\t");
				if (gAdditional != null && !gAdditional.isEmpty()) {
					ListIterator<Integer> it = xAdditional.listIterator();
					while (it.hasNext()) {
						int value = it.next().intValue();
						if (value != 58 && value != 59 && value != 66 && value != 100)
							writer.append("\t");
					}
				}
				writer.append("NPP\t");
				writer.append("C Fix\n");
				cycleIt = results.getCycleResultsIterator();
				season = 0;
				while (cycleIt.hasNext()) {
					CycleResults cycle = cycleIt.next();
					if (cycle == null) throw new Exception("Null results found");
					season++;
					writer.append(season + "\t");
					// Report state variables
					Double[] x = cycle.getX();
					writer.append(formatter.format(x[3]) + "\t");
					writer.append(formatter.format(x[5]) + "\t");
					writer.append(formatter.format(x[7]) + "\t");
					writer.append(formatter.format(x[8]) + "\t");
					if (xAdditional != null && !xAdditional.isEmpty()) {
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 4 && value != 6 && value != 8 && value != 9)
								writer.append(formatter.format(x[value-1]) + "\t");
						}
					}
					// Report G variables
					Double[] g = cycle.getG();
					writer.append(formatter.format(g[57]) + "\t");
					writer.append(formatter.format(g[58]) + "\t");
					writer.append(formatter.format(g[65]) + "\t");
					writer.append(formatter.format(g[99]) + "\t");
					if (gAdditional != null && !gAdditional.isEmpty()) {
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 58 && value != 59 && value != 66 && value != 100)
								writer.append(formatter.format(g[value-1]) + "\t");
						}
					}
					// Report NPP and Carbon Fixation
					writer.append(formatter.format(cycle.getNpp()) + "\t");
					writer.append(formatter.format(cycle.getCarbonFixation(this.getCarbonFixationRatio())) + "\t");
				}
				break;
			case ForestBGCOutputManager.DAILY_REPORT:
				writer.append("\tX1\t");
				writer.append("X2\t");
				writer.append("X3\t");
				writer.append("X4\t");
				writer.append("X5\t");
				writer.append("X6\t");
				if (xAdditional != null && !xAdditional.isEmpty()) {
					ListIterator<Integer> it = xAdditional.listIterator();
					while (it.hasNext()) {
						int value = it.next().intValue();
						if (value != 1 && value != 2 && value != 3 && value != 4 && value != 5 && value != 6)
							writer.append("X" + value + "\t");
					}
				}
				writer.append("G100");
				if (gAdditional != null && !gAdditional.isEmpty()) {
					writer.append("\t");
					ListIterator<Integer> it = xAdditional.listIterator();
					while (it.hasNext()) {
						int value = it.next().intValue();
						if (value != 100) writer.append("G" + value + "\t");
					}
				}
				writer.append("\n");
				writer.append("Day\t");
				writer.append("Snow\t");
				writer.append("Soil\t");
				writer.append("Outflow\t");
				writer.append("Trans.\t");
				writer.append("Evapor.\t");
				writer.append("PSN\t");
				if (xAdditional != null && !xAdditional.isEmpty()) {
					ListIterator<Integer> it = xAdditional.listIterator();
					while (it.hasNext()) {
						int value = it.next().intValue();
						if (value != 1 && value != 2 && value != 3 && value != 4 && value != 5 && value != 6) writer.append("\t");
					}
				}
				writer.append("m_rsp");
				if (gAdditional != null && !gAdditional.isEmpty()) {
					writer.append("\t");
					ListIterator<Integer> it = xAdditional.listIterator();
					while (it.hasNext()) {
						int value = it.next().intValue();
						if (value != 100) writer.append("\t");
					}
				}
				writer.append("\n");
				dailyIt = results.getDailyResultsIterator();
				day = 0;
				while (dailyIt.hasNext()) {
					DailyResults daily = dailyIt.next();
					if (daily == null) throw new Exception("Null results found");
					day++;
					writer.append(day + "\t");
					// Report state variables
					Double[] x = daily.getX();
					writer.append(formatter.format(x[0]) + "\t");
					writer.append(formatter.format(x[1]) + "\t");
					writer.append(formatter.format(x[2]) + "\t");
					writer.append(formatter.format(x[3]) + "\t");
					writer.append(formatter.format(x[4]) + "\t");
					writer.append(formatter.format(x[5]) + "\t");
					if (xAdditional != null && !xAdditional.isEmpty()) {
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 1 && value != 2 && value != 3 && value != 4 && value != 5 && value != 6) writer.append(formatter.format(x[value-1]) + "\t");
						}
					}
					// Report G variables
					Double[] g = daily.getG();
					writer.append(formatter.format(g[99]));
					if (gAdditional != null && !gAdditional.isEmpty()) {
						writer.append("\t");
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 100) writer.append(formatter.format(g[value-1]) + "\t");
						}
					}
					writer.append("\n");
				}
				break;
			case ForestBGCOutputManager.LONG_REPORT:
				// Report constants

				// Report climatic variables
				
				// Report initial state variables
				
				// Daily report
				dailyIt = results.getDailyResultsIterator();
				day = 0;
				while (dailyIt.hasNext()) {
					DailyResults daily = dailyIt.next();
					if (daily == null) throw new Exception("Null results found");
					day++;
					writer.append("Day " + day + "\n\n");
					// Report state variables
					writer.append("State variables:\n");
					Double[] x = daily.getX();
					for (int i=0; i<x.length; i++) {
						writer.append("X[" + (i+1) + "]: " + formatter.format(x[i]) + "\t");
						if (i % 10 == 9) writer.append("\n");
					}
					writer.append("\n");
					// Report G variables
					writer.append("G variables:\n");
					Double[] g = daily.getG();
					for (int i=0; i<g.length; i++) {
						writer.append("G[" + (i+1) + "]: " + formatter.format(g[i]) + "\t");
						if (i % 10 == 9) writer.append("\n");
					}
					writer.append("\n");
					// Report Dx
					writer.append("State change variables:\n");
					Double[] dx = daily.getDx();
					for (int i=0; i<dx.length; i++) {
						writer.append("dx[" + (i+1) + "]: " + formatter.format(dx[i]) + "\t");
						if (i % 10 == 9) writer.append("\n");
					}
					writer.append("\n");
					// Report Flux
					writer.append("Flux variables:\n");
					Double[][] f = daily.getF();
					for (int i=0; i<dx.length; i++) {
						for (int j=0; j<f[i].length; j++) {
							writer.append("f[" + (i+1) + "][" + (j+1) + "]: " + formatter.format(f[i][j]) + "\t");
						}
						writer.append("\n");
					}
					writer.append("\n\n");
				}
				break;
			default:
				writer.close();
				throw new Exception("Kind of report not valid: " + report);
			}
			writer.close();
		}
		catch (Exception e) {
			writer.close();
			throw e;
		}
	}
	
	/**
	 * Reports the ForestBGC output into the provided file.
	 * @param results the result to be reported.
	 * @param the kind of report.
	 * @param file the file where the report is writed. It has to be an existing and writable file. The report is appended to the end of the file.
	 * @throws Exception
	 */
	public void report(GlobalForestBGCResults results, int report, File file) throws Exception {
		if (file == null || !file.exists() || !file.isFile() || !file.canWrite()) throw new Exception("File " + file.getAbsolutePath() + " is not valid");
		BufferedWriter writer = new BufferedWriter(new FileWriter(file));
		try {
			// some variables
			ListIterator<CycleResults> cycleIt;
			ListIterator<DailyResults> dailyIt;
			int season; int day;
			int point = 0;
			DecimalFormat formatter = new DecimalFormat("#.#");
			List<Integer> xAdditional = this.getStateVariablesReported();
			List<Integer> gAdditional = this.getGVariablesReported();
			ListIterator<ForestBGCResults> resultsIt = results.getResultsIterator();
			// report header
			writer.append("Simulation: " + results.getSimulationName() + "\n");
			writer.append("Date: " + results.getSimulationDate().toString() + "\n\n");
			writer.append(results.getEstimator().getVersion() + "\n");
			writer.append(results.getEstimator().getCopyright() + "\n\n");
			writer.append("---------------------------------------------------------\n\n");
			// report
			if (resultsIt == null || !resultsIt.hasNext()) throw new Exception("Null results found");
			while (resultsIt.hasNext()) {
				ForestBGCResults pointResults = resultsIt.next();
				point++;
				writer.append("Point " + point + ":\n\n");
				switch(report) {
				case ForestBGCOutputManager.RESUME_REPORT:
					cycleIt = pointResults.getCycleResultsIterator();
					writer.append("\tX4\t");
					writer.append("X6\t");
					writer.append("X8\t");
					writer.append("X9\t");
					if (xAdditional != null && !xAdditional.isEmpty()) {
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 4 && value != 6 && value != 8 && value != 9) writer.append("X" + value + "\t");
						}
					}
					writer.append("G58\t");
					writer.append("G59\t");
					writer.append("G66\t");
					writer.append("G100");
					if (gAdditional != null && !gAdditional.isEmpty()) {
						writer.append("\t");
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 58 && value != 59 && value != 66 && value != 100)
								writer.append("G" + value + "\t");
						}
					}
					writer.append("\n");
					writer.append("Season\t");
					writer.append("Trans.\t");
					writer.append("PSN\t");
					writer.append("Leaf C\t");
					writer.append("Stem C\t");
					if (xAdditional != null && !xAdditional.isEmpty()) {
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 4 && value != 6 && value != 8 && value != 9)
								writer.append("\t");
						}
					}
					writer.append("S_grw\t");
					writer.append("R_grw\t");
					writer.append("g_rsp\t");
					writer.append("m_rsp\t");
					if (gAdditional != null && !gAdditional.isEmpty()) {
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 58 && value != 59 && value != 66 && value != 100)
								writer.append("\t");
						}
					}
					writer.append("NPP\t");
					writer.append("C Fix\n");
					cycleIt = pointResults.getCycleResultsIterator();
					season = 0;
					while (cycleIt.hasNext()) {
						CycleResults cycle = cycleIt.next();
						if (cycle == null) throw new Exception("Null results found");
						season++;
						writer.append(season + "\t");
						// Report state variables
						Double[] x = cycle.getX();
						writer.append(formatter.format(x[3]) + "\t");
						writer.append(formatter.format(x[5]) + "\t");
						writer.append(formatter.format(x[7]) + "\t");
						writer.append(formatter.format(x[8]) + "\t");
						if (xAdditional != null && !xAdditional.isEmpty()) {
							ListIterator<Integer> it = xAdditional.listIterator();
							while (it.hasNext()) {
								int value = it.next().intValue();
								if (value != 4 && value != 6 && value != 8 && value != 9)
									writer.append(formatter.format(x[value-1]) + "\t");
							}
						}
						// Report G variables
						Double[] g = cycle.getG();
						writer.append(formatter.format(g[57]) + "\t");
						writer.append(formatter.format(g[58]) + "\t");
						writer.append(formatter.format(g[65]) + "\t");
						writer.append(formatter.format(g[99]) + "\t");
						if (gAdditional != null && !gAdditional.isEmpty()) {
							ListIterator<Integer> it = xAdditional.listIterator();
							while (it.hasNext()) {
								int value = it.next().intValue();
								if (value != 58 && value != 59 && value != 66 && value != 100)
									writer.append(formatter.format(g[value-1]) + "\t");
							}
						}
						// Report NPP and Carbon Fixation
						writer.append(formatter.format(cycle.getNpp()) + "\t");
						writer.append(formatter.format(cycle.getCarbonFixation(this.getCarbonFixationRatio())) + "\t");
						writer.append("\n\n");
					}
					break;
				case ForestBGCOutputManager.DAILY_REPORT:
					writer.append("\tX1\t");
					writer.append("X2\t");
					writer.append("X3\t");
					writer.append("X4\t");
					writer.append("X5\t");
					writer.append("X6\t");
					if (xAdditional != null && !xAdditional.isEmpty()) {
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 1 && value != 2 && value != 3 && value != 4 && value != 5 && value != 6)
								writer.append("X" + value + "\t");
						}
					}
					writer.append("G100");
					if (gAdditional != null && !gAdditional.isEmpty()) {
						writer.append("\t");
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 100) writer.append("G" + value + "\t");
						}
					}
					writer.append("\n");
					writer.append("Day\t");
					writer.append("Snow\t");
					writer.append("Soil\t");
					writer.append("Outflow\t");
					writer.append("Trans.\t");
					writer.append("Evapor.\t");
					writer.append("PSN\t");
					if (xAdditional != null && !xAdditional.isEmpty()) {
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 1 && value != 2 && value != 3 && value != 4 && value != 5 && value != 6) writer.append("\t");
						}
					}
					writer.append("m_rsp");
					if (gAdditional != null && !gAdditional.isEmpty()) {
						writer.append("\t");
						ListIterator<Integer> it = xAdditional.listIterator();
						while (it.hasNext()) {
							int value = it.next().intValue();
							if (value != 100) writer.append("\t");
						}
					}
					writer.append("\n");
					dailyIt = pointResults.getDailyResultsIterator();
					day = 0;
					while (dailyIt.hasNext()) {
						DailyResults daily = dailyIt.next();
						if (daily == null) throw new Exception("Null results found");
						day++;
						writer.append(day + "\t");
						// Report state variables
						Double[] x = daily.getX();
						writer.append(formatter.format(x[0]) + "\t");
						writer.append(formatter.format(x[1]) + "\t");
						writer.append(formatter.format(x[2]) + "\t");
						writer.append(formatter.format(x[3]) + "\t");
						writer.append(formatter.format(x[4]) + "\t");
						writer.append(formatter.format(x[5]) + "\t");
						if (xAdditional != null && !xAdditional.isEmpty()) {
							ListIterator<Integer> it = xAdditional.listIterator();
							while (it.hasNext()) {
								int value = it.next().intValue();
								if (value != 1 && value != 2 && value != 3 && value != 4 && value != 5 && value != 6) writer.append(formatter.format(x[value-1]) + "\t");
							}
						}
						// Report G variables
						Double[] g = daily.getG();
						writer.append(formatter.format(g[99]));
						if (gAdditional != null && !gAdditional.isEmpty()) {
							writer.append("\t");
							ListIterator<Integer> it = xAdditional.listIterator();
							while (it.hasNext()) {
								int value = it.next().intValue();
								if (value != 100) writer.append(formatter.format(g[value-1]) + "\t");
							}
						}
						writer.append("\n");
					}
					break;
				case ForestBGCOutputManager.LONG_REPORT:
					// Report constants

					// Report climatic variables
					
					// Report initial state variables
						
					// Daily report
					dailyIt = pointResults.getDailyResultsIterator();
					day = 0;
					while (dailyIt.hasNext()) {
						DailyResults daily = dailyIt.next();
						if (daily == null) throw new Exception("Null results found");
						day++;
						writer.append("Day " + day + "\n\n");
						// Report state variables
						writer.append("State variables:\n");
						Double[] x = daily.getX();
						for (int i=0; i<x.length; i++) {
							writer.append("X[" + (i+1) + "]: " + formatter.format(x[i]) + "\t");
							if (i % 10 == 9) writer.append("\n");
						}
						writer.append("\n");
						// Report G variables
						writer.append("G variables:\n");
						Double[] g = daily.getG();
						for (int i=0; i<g.length; i++) {
							writer.append("G[" + (i+1) + "]: " + formatter.format(g[i]) + "\t");
							if (i % 10 == 9) writer.append("\n");
						}
						writer.append("\n");
						// Report Dx
						writer.append("State change variables:\n");
						Double[] dx = daily.getDx();
						for (int i=0; i<dx.length; i++) {
							writer.append("dx[" + (i+1) + "]: " + formatter.format(dx[i]) + "\t");
							if (i % 10 == 9) writer.append("\n");
						}
						writer.append("\n");
						// Report Flux
						writer.append("Flux variables:\n");
						Double[][] f = daily.getF();
						for (int i=0; i<dx.length; i++) {
							for (int j=0; j<f[i].length; j++) {
								writer.append("f[" + (i+1) + "][" + (j+1) + "]: " + formatter.format(f[i][j]) + "\t");
							}
							writer.append("\n");
						}
						writer.append("\n\n");
					}
					break;
				default:
					writer.close();
					throw new Exception("Kind of report not valid: " + report);
				}
				writer.append("---------------------------------------------------------\n\n");
			}
			writer.close();
		}
		catch (Exception e) {
			writer.close();
			throw e;
		}
	}
}
