// By Ramón Moreno
// natafresa@gmail.com
// October 2010


using System;
using System.Collections;
using System.Collections.Generic;

using Emgu.CV.Structure;
using Emgu.CV;
using Emgu.CV.CvEnum;
using Emgu.Util;
using Emgu.Util.TypeEnum;


namespace WindowsFormsApplication1
{
    public class sphericImage
    {
        private double[,] Zenith; //double matrix for zenith angles
        private double[,] Azimuth; //double matrix for azimuth angles
        private double[,] L; //double matrix for vector longs

        private int rows; // image size
        private int col;

        Image<Bgr, byte> Im; //OpenCv Image (color [0-255])



        public sphericImage(Image<Bgr, byte> I)
        {


            rows = I.Size.Height;
            col = I.Size.Width;

            Zenith = new double[rows, col];
            Azimuth = new double[rows, col];
            L = new double[rows, col];
            Im = I;

            /// temporal variables
            double R, G, B;
            // double den,zen,azi;
            Bgr miColor;


            for (int f = 0; f < rows; f++)
            {
                for (int c = 0; c < col; c++)
                {
                    miColor = I[f, c];
                    R = miColor.Red;
                    G = miColor.Green;
                    B = miColor.Blue;

                    double l = 0;

                    // put the vector long
                    l = Math.Sqrt((R * R) + (G * G) + (B * B));
                    L[f, c] = l;

                    Zenith[f, c] = Math.Atan(G / R);// zen;

                    Azimuth[f, c] = Math.Acos(B / l);// azi;
                }
            }
        }// end cosntructor

        /// <summary>
        /// Return the matrix L 
        /// </summary>
        /// <returns></returns>
        public double[,] getL() { return this.L; }

        /// <summary>
        /// return the Zenith matrix 
        /// </summary>
        /// <returns></returns>
        public double[,] getZenith() { return this.Zenith; }

        /// <summary>
        /// return the Azimuth matrix 
        /// </summary>
        /// <returns></returns>
        public double[,] getAzimuth() { return this.Azimuth; }



        /// <summary>
        /// Return the L parameter normalized to the range [0,1]
        /// </summary>
        /// <returns></returns>
        public double[,] getNormalizedL()
        {
            double[,] M = new double[rows, col];
            double den =  Math.Sqrt(255 * 255 * 3); //maximum value

            for (int f = 0; f < rows; f++)
                for (int c = 0; c < col; c++)
                    M[f, c] = L[f, c] / den;

            return M;
        }


        /// <summary>
        ///  Return the Zenith parameter normalized to the range [0,1]
        /// </summary>
        /// <returns></returns>
        public double[,] getNormalizedZenith()
        {
            double[,] M = new double[rows, col];
            double den = Math.PI / 2; //maximum valur

            for (int f = 0; f < rows; f++)
                for (int c = 0; c < col; c++)
                    M[f, c] = Zenith[f, c] / den;

            return M;
        }

        /// <summary>
        /// return the Azimuth parameter normalized to the range [0,1]
        /// </summary>
        /// <returns></returns>
        public double[,] getNormalizedZenith()
        {
            double[,] M = new double[rows, col];
            double den = Math.PI / 2;  //maximum valur

            for (int f = 0; f < rows; f++)
                for (int c = 0; c < col; c++)
                    M[f, c] = Azimuth[f, c] / den;

            return M;
        }



        /// <summary>
        /// This function calculate the euclidean distance between to points,
        /// In this case, this function is designed to work with Zenith and Azimuth parameters
        /// </summary>
        /// <param name="a1">
        /// A <see cref="System.double"/>
        /// </param>The zenith parameter of the 'a' point
        /// <param name="a2">
        /// A <see cref="System.double"/>
        /// </param>The Azimuth parameter of the 'a' point
        /// <param name="b1">
        /// A <see cref="System.double"/>
        /// </param>The zenith parameter of the 'b' point
        /// <param name="b2">
        /// A <see cref="System.double"/>
        /// </param>The Azimuth parameter of the 'b' point
        /// <returns>
        /// A <see cref="System.double"/>
        /// </returns>
        private double distanceEU(double a1, double a2, double b1, double b2)
        {
            double d1, d2;
            d1 = (a1 - b1);
            d2 = (a2 - b2);
            return Math.Sqrt((d1 * d1) + (d2 * d2));
        }

        /// <summary>
        /// this function calculate the pseuso Sobel chromatic gradient,
        /// for this goal we use the spherical coordinates.
        /// we don't use the border image pixels.
        /// </summary>
        /// <returns>
        /// A gradient intensity image in double format
        /// A <see cref="System.double[,]"/>
        /// </returns>

        public double[,] SobelChromaticGradient()
        {

            double[,] IR = new double[this.rows, this.col];
            //the gradient image

            //Temporal variables for distence bettween neigboor pixels
            double E1, E2, E3, E4, E5, E6;
            //temporal variables for neighboor pixels information
            // Z1,Z2,Z3
            // Z4,Z5,Z6
            // Z7,Z8,Z9
            double[,] Z = new double[10, 2];


            // for each pixel in the image
            //we dont process the border pixels
            for (int f = 1; f < rows - 1; f++)
            {
                for (int c = 1; c < col - 1; c++)
                {

                    // for the 8 neighboors
                    int cont = 1;
                    for (int cl = -1; cl < 2; cl++)
                    {
                        for (int fl = -1; fl < 2; fl++)
                        {
                            //saving the neiboors information in Z
                            Z[cont, 0] = this.Zenith[fl + f, cl + c];
                            Z[cont++, 1] = this.Azimuth[fl + f, cl + c];


                        }
                    }
                    // calculating the 6 angular distances
                    E1 = distanceEU(Z[1, 0], Z[1, 1], Z[3, 0], Z[3, 1]);
                    E2 = distanceEU(Z[6, 0], Z[6, 1], Z[4, 0], Z[4, 1]);
                    E3 = distanceEU(Z[9, 0], Z[9, 1], Z[7, 0], Z[7, 1]);
                    E4 = distanceEU(Z[7, 0], Z[7, 1], Z[1, 0], Z[1, 1]);
                    E5 = distanceEU(Z[8, 0], Z[8, 1], Z[2, 0], Z[2, 1]);
                    E6 = distanceEU(Z[9, 0], Z[9, 1], Z[3, 0], Z[3, 1]);

                    // Asign the pseudo Sobel gradient
                    IR[f, c] = E1 + 2 * E2 + E3 + E4 + 2 * E5 + E6;
                }
            }
            return IR;

        }

        /// <summary>
        /// this function return the pseudo Prewitt chromatic gradient
        /// The image border pixels are not processed (=0)
        /// </summary>
        /// <returns>
        /// the chromatic intensisty gradient in doube format
        /// A <see cref="System.double[,]"/>
        /// </returns>

        public double[,] PrewittChromaticGradient()
        {
            double[,] IR = new double[this.rows, this.col];
            //the gradient image

            //Temporal variables for distence bettween neigboor pixels
            double E1, E2, E3, E4, E5, E6;
            //temporal variables for neighboor pixels information
            // Z1,Z2,Z3
            // Z4,Z5,Z6
            // Z7,Z8,Z9
            double[,] Z = new double[10, 2];


            // for each pixel in the image
            //we dont process the border pixels
            for (int f = 1; f < rows - 1; f++)
            {
                for (int c = 1; c < col - 1; c++)
                {

                    // for the 8 neighboors
                    int cont = 1;
                    for (int cl = -1; cl < 2; cl++)
                    {
                        for (int fl = -1; fl < 2; fl++)
                        {
                            //saving the neiboors information in Z
                            Z[cont, 0] = this.Zenith[fl + f, cl + c];
                            Z[cont++, 1] = this.Azimuth[fl + f, cl + c];

                        }
                    }
                    // calculating the 6 angular distances
                    E1 = distanceEU(Z[1, 0], Z[1, 1], Z[3, 0], Z[3, 1]);
                    E2 = distanceEU(Z[6, 0], Z[6, 1], Z[4, 0], Z[4, 1]);
                    E3 = distanceEU(Z[9, 0], Z[9, 1], Z[7, 0], Z[7, 1]);
                    E4 = distanceEU(Z[7, 0], Z[7, 1], Z[1, 0], Z[1, 1]);
                    E5 = distanceEU(Z[8, 0], Z[8, 1], Z[2, 0], Z[2, 1]);
                    E6 = distanceEU(Z[9, 0], Z[9, 1], Z[3, 0], Z[3, 1]);

                    // Asign the pseudo Prewitt gradient
                    IR[f, c] = E1 + E2 + E3 + E4 + E5 + E6;
                }
            }
            return IR;

        }

        /// <summary>
        /// This function return a sweet chrmatic gradient. this function only use the 4-neighboors
        /// It's based in the Prewitt mask
        /// </summary>
        /// <returns>
        /// the chromatic intensisty gradient in doube format
        /// A <see cref="System.double[,]"/>
        /// </returns>

        public double[,] SwChromaticGradient()
        {
            double[,] IR = new double[this.rows, this.col];
            //the gradient image

            //Temporal variables for distence bettween neigboor pixels
            double E2, E5;
            //temporal variables for neighboor pixels information
            // Z1,Z2,Z3
            // Z4,Z5,Z6
            // Z7,Z8,Z9
            double[,] Z = new double[5, 2];


            // for each pixel in the image
            //we dont process the border pixels
            for (int f = 1; f < rows - 1; f++)
            {
                for (int c = 1; c < col - 1; c++)
                {

                    //saving the neiboors information in Z
                    Z[1, 0] = this.Zenith[f - 1, c];
                    Z[1, 1] = this.Azimuth[f - 1, c];

                    Z[2, 0] = this.Zenith[f, c - 1];
                    Z[2, 1] = this.Azimuth[f, c - 1];

                    Z[3, 0] = this.Zenith[f, c + 1];
                    Z[3, 1] = this.Azimuth[f, c + 1];

                    Z[4, 0] = this.Zenith[f + 1, c];
                    Z[4, 1] = this.Azimuth[f + 1, c];


                    // calculating the 6 angular distances

                    E2 = distanceEU(Z[1, 0], Z[1, 1], Z[4, 0], Z[4, 1]);
                    E5 = distanceEU(Z[3, 0], Z[3, 1], Z[2, 0], Z[2, 1]);


                    // Asign the pseudo Sobel gradient
                    IR[f, c] = E2 + E5;
                }
            }
            return IR;

        }

        /// <summary>
        /// Return the intensity Prewitt gradient
        /// </summary>
        /// <returns></returns>
        public double[,] PrewittIntensityGradient()
        {
            double[,] IR = new double[this.rows, this.col]; //return matrix

            double a1, a2, a3, a4, a6, a7, a8, a9;

            // WE use as intensity the normalized L
            double[,] I = this.getNormalizedL();

            double var;
            for (int f = 1; f < this.rows - 1; f++)
                for (int c = 1; c < this.col - 1; c++)
                {
                    a1 = I[f - 1, c - 1];
                    a2 = I[f - 1, c];
                    a3 = I[f - 1, c + 1];
                    a4 = I[f, c - 1];
                    a6 = I[f, c + 1];
                    a7 = I[f - 1, c - 1];
                    a8 = I[f + 1, c];
                    a9 = I[f + 1, c + 1];


                    var = Math.Abs(((a3 + a6 + a9) - (a1 + a4 + a7))) + Math.Abs(((a1 + a2 + a3) - (a7 + a8 + a9)));
                    //trucate the values greater then 1
                    if (var > 1)
                        IR[f, c] = 1;
                    else
                        IR[f, c] = var;


                }


            return IR;
        }



        /// <summary>
        /// function used for hybrid gradient for the estimation of sinosoidal values
        /// </summary>
        /// <param name="a"></param>
        /// minimum threshold
        /// <param name="b"></param>
        /// maximum threshold
        /// <returns></returns>
        private double[,] MatrixValuesForMix(double a, double b)
        {
            double[,] IR = new double[this.rows, this.col];
            // WE use as intensity the normalized L
            double[,] I = this.getNormalizedL();

            double var;
            for (int f = 0; f < this.rows; f++)
                for (int c = 0; c < this.col; c++)
                {
                    var = I[f, c];
                    if (var <= a)
                        IR[f, c] = 0;
                    else
                        if (var >= b)
                            IR[f, c] = 1;
                        else
                        {
                            //look for the sinosoidal value
                            IR[f, c] = .5 + Math.Cos(((var - a) / (b - a)) * Math.PI) / 2;
                        }

                }
            return IR;
        }
        /// <summary>
        /// This function calculate a hybrid gradient. 
        /// Pixels with values less that 'a' are used for an intensity gradient. 
        /// Pixels with values greather that 'b' are used for a chromatic gradient.
        /// Pixels with valies between 'a' and 'b' are used for hybrid gradient
        /// </summary>
        /// <param name="a"></param>
        /// minimun threshold for the hybrid gradient
        /// <param name="b"></param>
        /// maximun threshold for the hybrid gradient
        /// <returns></returns>
        public double[,] HybridGradient(double a, double b)
        {
            ///auxiliar function that calculate th matix values for mixing following a sinosoidal rule     

            double[,] IR = new double[this.rows, this.col]; //return matrix
            double[,] IG = PrewittIntensityGradient(); //intensity gradient
            double[,] CG = PrewittChromaticGradient(); //chromatic gradient
            double[,] MM = MatrixValuesForMix(a, b); //senosoidal values for mixing


            for (int f = 0; f < this.rows; f++)
                for (int c = 0; c < this.col; c++)
                    IR[f, c] = MM[f, c] * IG[f, c] + (1 - MM[f, c]) * CG[f, c];


            return IR;
        }


        //////  FUNCTIONs FOR WATERSHED SEGMENTATION ////////

        /// <summary>
        /// this function return the maximum of a matrix
        /// </summary>
        /// <param name="M"></param>
        /// input matrix
        /// <param name="rows"></param>
        /// row size
        /// <param name="col"></param>
        /// column size
        /// <returns></returns>

        private double max(double[,] M, int rows, int col)
        {
            double r = 0;

            for (int i = 0; i < rows; i++)
                for (int j = 0; j < col; j++)
                    if (M[i, j] > r)
                    {
                        r = M[i, j];
                    }

            return r;
        }

        /// <summary>
        /// this function return the maximum of a matrix
        /// </summary>
        /// <param name="M"></param>
        /// input matrix
        /// <param name="rows"></param>
        /// row size
        /// <param name="col"></param>
        /// column size
        /// <returns></returns>

        private int max(int[,] M, int rows, int col)
        {
            int r = 0;

            for (int i = 0; i < rows; i++)
                for (int j = 0; j < col; j++)
                    if (M[i, j] > r)
                    {
                        r = M[i, j];
                    }

            return r;
        }

        /// <summary>
        /// this function return  the minumum matrix value 
        /// </summary>
        /// <param name="M"></param>
        /// input matrix
        /// <param name="rows"></param>
        /// rows size
        /// <param name="col"></param>
        /// column size
        /// <returns></returns>
        private double min(double[,] M, int rows, int col)
        {
            double r = 100000000; //inizialize with a big value


            for (int i = 0; i < rows; i++)
                for (int j = 0; j < col; j++)
                    if (M[i, j] < r)
                    {
                        r = M[i, j];

                    }

            return r;
        }

        /// <summary>
        /// this fucntion look the occurrences of 'value' in the double matrix M. Return a bidimensional matrix with the coordinates of this occurrences.
        /// </summary>
        /// <param name="M"></param>
        /// Input matrix
        /// <param name="rows"></param>
        /// row size
        /// <param name="col"></param>
        /// col size
        /// <param name="value"></param>
        /// arraylist of int[2]
        /// <returns></returns>
        private List<int[]> find(double[,] M, int rows, int col, double value)
        {
            List<int[]> r = new List<int[]>();


            for (int i = 0; i < rows; i++)
                for (int j = 0; j < col; j++)
                    if (M[i, j] == value)
                    {
                        int[] coordinates = new int[2];
                        coordinates[0] = i;
                        coordinates[1] = j;
                        r.Add(coordinates);
                    }
            return r;
        }

        /// <summary>
        /// this fucntion look the occurrences of values smaller than 'value' in the double matrix M. Return a bidimensional matrix with the coordinates of this occurrences.
        /// </summary>
        /// <param name="M"></param>
        /// Input matrix
        /// <param name="rows"></param>
        /// row size
        /// <param name="col"></param>
        /// col size
        /// <param name="value"></param>
        /// arraylist of int[2]
        /// <returns></returns>
        private List<int[]> findless(double[,] M, int rows, int col, double value)
        {
            List<int[]> r = new List<int[]>();


            for (int i = 0; i < rows; i++)
                for (int j = 0; j < col; j++)
                    if (M[i, j] <= value)
                    {
                        int[] coordinates = new int[2];
                        coordinates[0] = i;
                        coordinates[1] = j;
                        r.Add(coordinates);
                    }
            return r;
        }


        /// <summary>
        /// this fucntion look the occurrences of 'value' in the integer  matrix M. Return a bidimensional matrix with the coordinates of this occurrences.
        /// </summary>
        /// <param name="M"></param>
        /// Input matrix
        /// <param name="rows"></param>
        /// row size
        /// <param name="col"></param>
        /// col size
        /// <param name="value"></param>
        /// arraylist of int[2]
        /// <returns></returns>
        private List<int[]> find(int[,] M, int rows, int col, int value)
        {
            List<int[]> r = new List<int[]>();


            for (int i = 0; i < rows; i++)
                for (int j = 0; j < col; j++)
                    if (M[i, j] == value)
                    {
                        int[] coordinates = new int[2];
                        coordinates[0] = i;
                        coordinates[1] = j;
                        r.Add(coordinates);
                    }
            return r;
        }

        /// <summary>
        /// this fucntion look the occurrences of values smaller than 'value' in the integer  matrix M. Return a bidimensional matrix with the coordinates of this occurrences.
        /// </summary>
        /// <param name="M"></param>
        /// Input matrix
        /// <param name="rows"></param>
        /// row size
        /// <param name="col"></param>
        /// col size
        /// <param name="value"></param>
        /// arraylist of int[2]
        /// <returns></returns>
        private List<int[]> findless(int[,] M, int rows, int col, int value)
        {
            List<int[]> r = new List<int[]>();


            for (int i = 0; i < rows; i++)
                for (int j = 0; j < col; j++)
                    if (M[i, j] == value)
                    {
                        int[] coordinates = new int[2];
                        coordinates[0] = i;
                        coordinates[1] = j;
                        r.Add(coordinates);
                    }
            return r;
        }

        /// <summary>
        /// this function return a labeling matrix, where each closed region is labeled with a number.
        /// </summary>
        /// <param name="M"></param>
        /// Input integer matrix
        /// <param name="rows"></param>
        /// rows size
        /// <param name="col"></param>
        /// cols size
        /// <returns>Integer Matrix</returns>

        private int[,] bwlabel(int[,] M, int rows, int col)
        {

            int[,] IR = new int[rows, col]; // matrix label

            int labelCounter = 0;
            double val;
            int a1, a2, a3, a4, a6, a7, a8, a9; //neighboor matrix
            a1 = 0; a2 = 0; a3 = 0; a4 = 0; a6 = 0; a7 = 0; a8 = 0; a9 = 0;

            for (int f = 0; f < rows; f++)
                for (int c = 0; c < col; c++)
                {
                    val = M[f, c];
                    if ((val > 0) & (IR[f, c] == 0))// is a new pixel without label
                    {   //look at the neighboors
                        if (f - 1 >= 0 & c - 1 >= 0)
                            a1 = IR[f - 1, c - 1];
                        if (f - 1 >= 0)
                            a2 = IR[f - 1, c];
                        if (f - 1 >= 0 & c + 1 < col)
                            a3 = IR[f - 1, c + 1];
                        if (c - 1 >= 0)
                            a4 = IR[f, c - 1];
                        if (c + 1 < col)
                            a6 = IR[f, c + 1];
                        if (f + 1 < rows & c - 1 >= 0)
                            a7 = IR[f + 1, c - 1];
                        if (f + 1 < rows)
                            a8 = IR[f + 1, c];
                        if (f + 1 < rows & c + 1 < col)
                            a9 = IR[f + 1, c + 1];

                        /// we have only three possible cases:
                        /// no one of the neighboors has label -> thes we asign a new label
                        /// the neighboors has only one label -> then we asign this label
                        /// ne neighboors has different labels -> assing a new label to all neighboors (mergin)

                        /// no one of the neighboors has label -> thes we asign a new label
                        if ((a1 == 0) & (a2 == 0) & (a3 == 0) & (a4 == 0) & (a6 == 0) & (a7 == 0) & (a8 == 0) & (a9 == 0))
                        {
                            IR[f, c] = ++labelCounter;
                        }
                        else
                        {
                            // look the neighboor labels
                            int lc = 0; // local label counter
                            int[] labels = new int[8];
                            if (a1 > 0) { labels[lc++] = a1; }
                            if (a2 > 0) { labels[lc++] = a2; }
                            if (a3 > 0) { labels[lc++] = a3; }
                            if (a4 > 0) { labels[lc++] = a4; }
                            if (a6 > 0) { labels[lc++] = a6; }
                            if (a7 > 0) { labels[lc++] = a7; }
                            if (a8 > 0) { labels[lc++] = a8; }
                            if (a9 > 0) { labels[lc++] = a9; }
                            /// the neighboors has only one label -> then we asign this label
                            if (lc >= 1) //exist at least one label
                            {
                                int[] dflb = new int[8]; //counter of different labels
                                int dflbCounter = 0;
                                int _lb = 0;
                                for (int k = 0; k < lc; k++)
                                {
                                    _lb = labels[k];
                                    int esta = 0;
                                    for (int kk = 0; kk < k; kk++)
                                        if (labels[kk] == _lb)
                                            esta = 1;
                                    if (esta == 0)
                                    {
                                        dflb[dflbCounter++] = _lb;
                                    }
                                }
                                /// the neighboors has only one label -> then we asign this label
                                if (dflbCounter == 1)
                                    IR[f, c] = dflb[0];
                                else
                                {
                                    /// the neighboors has different labels -> assing a new label to all neighboors (mergin)
                                    // we select a valid only the first label, the rest of regions with different labels are
                                    // changed to this one.
                                    int LABEL = dflb[0];
                                    IR[f, c] = dflb[0];
                                    /// mergin region
                                    for (int k = 1; k < dflbCounter; k++)
                                    {
                                        int tmpLABEL = dflb[k];
                                        List<int[]> px = find(IR, rows, col, tmpLABEL);
                                        int[] coord;
                                        IEnumerator e = px.GetEnumerator();
                                        while (e.MoveNext())
                                        {
                                            coord = (int[])e.Current;
                                            IR[coord[0], coord[1]] = LABEL;
                                        }

                                    }
                                }
                            }
                        }
                    }

                }// end for
            /// count the amount of regions and asing a ordered label
            /// 


            return bwOrdering(IR);

        }

        /// <summary>
        /// sort a labeled matrix [0-#labels amount]
        /// </summary>
        /// <param name="IR"></param>
        /// <returns></returns>
        private int[,] bwOrdering(int[,] IR)
        {

            List<int> Labels = new List<int>();
            IEnumerator eLabels;
            int label = 0;
            for (int f = 0; f < rows; f++)
                for (int c = 0; c < col; c++)
                {
                    label = IR[f, c];
                    if (!Labels.Contains(label))
                        Labels.Add(label);
                }


            int tmpLabel;

            int[,] IR2 = new int[rows, col];

            //  ArrayList ocurr;
            IEnumerator e2;

            int labelCounter2 = 0;
            int bigestRegion = 0;
            int memberOfBigestRegion = 0;

            eLabels = Labels.GetEnumerator();
            while (eLabels.MoveNext())
            {
                tmpLabel = (int)eLabels.Current;
                List<int[]> ocurr2 = find(IR, rows, col, tmpLabel);
                //look for the bigest region
                if (ocurr2.Count > memberOfBigestRegion)
                {
                    bigestRegion = labelCounter2;
                    memberOfBigestRegion = ocurr2.Count;
                }

                e2 = ocurr2.GetEnumerator();
                int[] coord;
                while (e2.MoveNext())
                {
                    coord = (int[])e2.Current;
                    IR2[coord[0], coord[1]] = labelCounter2;
                }
                labelCounter2 = labelCounter2 + 1;

            }
            ///optimizing our code for watershed, we set the background to 0, it's the bigest region. 
            ///
            ///If the bigest region is labeled with '0', nothing to do.

            List<int[]> regionCero, regionBigest;
            IEnumerator regionCeroNumerator, regionBigestNumerator;
            if (bigestRegion != 0)
            {
                int[] coord;

                regionCero = find(IR2, rows, col, 0);
                regionCeroNumerator = regionCero.GetEnumerator();

                regionBigest = find(IR2, rows, col, 1);
                regionBigestNumerator = regionBigest.GetEnumerator();

                //label with '0'
                while (regionBigestNumerator.MoveNext())
                {
                    coord = (int[])regionBigestNumerator.Current;
                    IR2[coord[0], coord[1]] = 0;
                }
                //change the label
                while (regionCeroNumerator.MoveNext())
                {
                    coord = (int[])regionCeroNumerator.Current;
                    IR2[coord[0], coord[1]] = bigestRegion;
                }


            }
            return IR2;
        }

        /// <summary>
        /// image segementation based on watershed and with chromatic criterium. 
        /// </summary>
        /// <param name="IG"></param>
        /// Intensity image
        /// <param name="threshold"></param>
        /// Threshold for region mergin
        /// <param name="steps"></param>
        /// amount of steps (slides)
        /// <returns>ArrayList with three values. first a int[,] is the labeled image. Second and thrird double[,] and double[,]. Are the zenith and azimuthal angles</returns>
        public ArrayList watershed(double[,] IG, double threshold, int steps)
        {



            // In this matriz we are going to labeling all pixels
            int[,] BW = new int[this.rows, this.col];
            Boolean[,] Labeled = new Boolean[this.rows, this.col]; //matrix wich save if one pixel have been labeled
            //initialize to false
            for (int i = 0; i < this.rows; i++)
                for (int j = 0; j < this.col; j++)
                    Labeled[i, j] = false;

            //the correspondence label, chrmaticity and members amount
            double[] LbChr = new double[4]; //[ label, zenith, azimuth, #memebers]
            List<double[]> LbChrList = new List<double[]>(); //list of labeled regions. It has Lbchr types.



            double mx = max(IG, this.rows, this.col);
            double mn = min(IG, this.rows, this.col);

            double df = mx - mn;
            double jump = df / (double)steps;

            double thi = mn + jump;

            List<int[]> V = findless(IG, this.rows, this.col, thi);

            IEnumerator e = V.GetEnumerator();
            //put this values to 1 in BW
            int[] coord; // generic variable for coordinates catching

            while (e.MoveNext())
            {
                coord = (int[])e.Current;
                BW[coord[0], coord[1]] = 1;
            }

            BW = bwlabel(BW, this.rows, this.col); // MAtrix labeling

            //for each region, we'll save the label, the azimuth and zenith means and the  members amount.
            int nReg = max(BW, this.rows, this.col);

            for (int i = 0; i <= nReg; i++)
            {
                List<int[]> members = find(BW, this.rows, this.col, i);
                IEnumerator membersNumerator = members.GetEnumerator();
                double zenith_sum = 0;
                double azimuth_sum = 0;
                double[] LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]

                for(int i2 =0; i2 <members.Count; i2++)                
                {
                    coord = members[i2];
                    zenith_sum = zenith_sum + this.Zenith[coord[0], coord[1]];
                    azimuth_sum = azimuth_sum + this.Azimuth[coord[0], coord[1]];
                }
                LbChrAux[0] = (double)i;
                LbChrAux[1] = zenith_sum / (double)members.Count;
                LbChrAux[2] = azimuth_sum / (double)members.Count;
                LbChrAux[3] = (double)members.Count;

                //save this information
                LbChrList.Add(LbChrAux);

            }

            //initialize Labeled with BW
            for (int i = 0; i < this.rows; i++)
                for (int j = 0; j < this.col; j++)
                    if (BW[i, j] > 0)
                        Labeled[i, j] = true;



            /// Iteration Process for all image
            /// 
            for (int iterations = 2; iterations <= steps; iterations++)
            {
                thi = mn + (iterations * jump);
                List<int[]> selected = findless(IG, this.rows, this.col, thi);
                
                int a1, a2, a3, a4, a6, a7, a8, a9;//for labels

                // double d1, d2, d3, d4, d6, d7, d8, d9;//for distances

                /// For each selected pixel and not visited
                /// 
                for(int i2 =0; i2 < selected.Count;i2++)                
                {
                    int f, c;
                    a1 = 0; a2 = 0; a3 = 0; a4 = 0; a6 = 0; a7 = 0; a8 = 0; a9 = 0; //clean
                    coord = selected[i2];
                    f = coord[0];
                    c = coord[1];
                    if (Labeled[f, c] == false)//it's an unclasified pixel
                    {
                        Labeled[f, c] = true; //we are going to label it.
                        /// et voila the tomatoes. we must to label the new pixel
                        /// Fot this goal, we'll look the neighboors chromatic information
                        /// and calculte the chromatic distance. In function of this value we assign the new
                        /// label.
                        /// look the neighboors labels
                        /// 
                        if (f - 1 >= 0 & c - 1 >= 0)
                            a1 = BW[f - 1, c - 1];
                        if (f - 1 >= 0)
                            a2 = BW[f - 1, c];
                        if (f - 1 >= 0 & c + 1 < this.col)
                            a3 = BW[f - 1, c + 1];
                        if (c - 1 >= 0)
                            a4 = BW[f, c - 1];
                        if (c + 1 < this.col)
                            a6 = BW[f, c + 1];
                        if (f + 1 < this.rows & c - 1 >= 0)
                            a7 = BW[f + 1, c - 1];
                        if (f + 1 < this.rows)
                            a8 = BW[f + 1, c];
                        if (f + 1 < this.rows & c + 1 < this.col)
                            a9 = BW[f + 1, c + 1];

                        if ((a1 == 0) & (a2 == 0) & (a3 == 0) & (a4 == 0) & (a6 == 0) & (a7 == 0) & (a8 == 0) & (a9 == 0))
                        {
                            //it hasn't got labeled neighboors
                            int newLabel = LbChrList.Count; //new label

                            BW[f, c] = newLabel;

                            // save this information in LbChrList
                            double[] LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
                            LbChrAux[0] = (double)newLabel; //label
                            LbChrAux[1] = Zenith[f, c];
                            LbChrAux[2] = Azimuth[f, c];
                            LbChrAux[3] = 1; //#members
                            LbChrList.Add(LbChrAux);
                        }
                        else
                        {
                            //if in the neighboorgs there is only a label, asign this label
                            // look the neighboor labels
                            int lc = 0; // local label counter
                            int[] labels = new int[8];
                            if (a1 > 0) { labels[lc++] = a1; }
                            if (a2 > 0) { labels[lc++] = a2; }
                            if (a3 > 0) { labels[lc++] = a3; }
                            if (a4 > 0) { labels[lc++] = a4; }
                            if (a6 > 0) { labels[lc++] = a6; }
                            if (a7 > 0) { labels[lc++] = a7; }
                            if (a8 > 0) { labels[lc++] = a8; }
                            if (a9 > 0) { labels[lc++] = a9; }
                            /// the neighboors has only one label -> then we asign this label
                            if (lc > 0) //exist at least one label
                            {
                                int[] dflb = new int[8]; //counter of different labels
                                int dflbCounter = 0;
                                int _lb = 0;
                                for (int k = 0; k < lc; k++)
                                {
                                    _lb = labels[k];
                                    int esta = 0;
                                    for (int kk = 0; kk < k; kk++)
                                        if (labels[kk] == _lb)
                                            esta = 1;
                                    if (esta == 0)
                                        dflb[dflbCounter++] = _lb;

                                }
                                /// the neighboors has only one label -> then we asign this label
                                if (dflbCounter == 1)
                                {
                                    int thisLabel = dflb[0]; //label
                                    BW[f, c] = thisLabel;

                                    // actualize  this information in LbChrList
                                    double[] LbChrOld = new double[4];
                                    double[] LbChrNew = new double[4];
                                    
                                    LbChrOld = LbChrList[thisLabel];// (double[])tmpNumerator.Current;

                                    LbChrNew[0] = thisLabel; //label
                                    LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + Zenith[f, c]) / (LbChrOld[3] + 1); // new zenith mean
                                    LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + Azimuth[f, c]) / (LbChrOld[3] + 1); // new azimuth mean
                                    LbChrNew[3] = LbChrOld[3] + 1; //# members

                                    LbChrList.RemoveAt(thisLabel);
                                    LbChrList.Insert(thisLabel, LbChrNew); // actualize the region information


                                }
                                else
                                {
                                    /// the neighboors has different labels 
                                    /// We must look for the chromatic dintance to them. and if some of them are smaler than 
                                    /// 'threshold' we must merging then or asing this label. and actualize this information.
                                    /// 
                                    //lc is the labels neighboor count
                                    //labels is a vector with the neighboors labels
                                    double[,] distancesM = new double[dflbCounter, dflbCounter]; // matrix for the distances between the regions
                                    //inicializamos a 1
                                    for (int i = 0; i < dflbCounter; i++)
                                        for (int j = 0; j < dflbCounter; j++)
                                            distancesM[i, j] = 1;

                                    double[] distancesV = new double[dflbCounter]; //distance between the regions and the pixels[f,c]
                                    //inicializamos a 1
                                    for (int i = 0; i < dflbCounter; distancesV[i++] = 1) ;

                                    //fill distancesM
                                    //diagonal superior
                                    double[] Chr1 = new double[2]; //chromaticity of label1
                                    double[] Chr2 = new double[2]; //chromaticity of label2


                                    int label1, label2;
                                    double v0, v1;

                                    for (int ii = 0; ii < dflbCounter; ii++)
                                        for (int jj = ii + 1; jj < dflbCounter; jj++)
                                        {
                                            label1 = dflb[ii];
                                            label2 = dflb[jj];

                                           

                                            LbChr = new double[4]; //clean

                                            LbChr = LbChrList[label1];// tmpNumerator.Current;
                                            Chr1[0] = LbChr[1];
                                            Chr1[1] = LbChr[2];

                                            LbChr = new double[4]; //clean

                                            LbChr = LbChrList[label2];// tmpNumerator.Current;                                           
                                            Chr2[0] = LbChr[1];
                                            Chr2[1] = LbChr[2];

                                            v0 = (Chr1[0] - Chr2[0]) * (Chr1[0] - Chr2[0]);
                                            v1 = (Chr1[1] - Chr2[1]) * (Chr1[1] - Chr2[1]);
                                            distancesM[ii, jj] = Math.Sqrt(v0 + v1);

                                        }

                                    //fill distancesV
                                    Chr1[0] = Zenith[f, c];
                                    Chr1[1] = Azimuth[f, c];
                                    for (int ii = 0; ii < dflbCounter; ii++)
                                    {
                                        label1 = dflb[ii];
                                       

                                        LbChr = LbChrList[label1];// tmpNumerator.Current;

                                        Chr2[0] = LbChr[1];
                                        Chr2[1] = LbChr[2];

                                        v0 = (Chr1[0] - Chr2[0]) * (Chr1[0] - Chr2[0]);
                                        v1 = (Chr1[1] - Chr2[1]) * (Chr1[1] - Chr2[1]);
                                        distancesV[ii] = Math.Sqrt(v0 + v1);

                                    }

                                    // look if some neighboor label is close enough
                                    double mind = 1000000000;//minimum distance
                                    int lmind = 0;

                                    for (int ii = 0; ii < dflbCounter; ii++)
                                        if (distancesV[ii] < mind)
                                        {
                                            mind = distancesV[ii];
                                            lmind = dflb[ii];///sure??
                                        }

                                    ///LABELING
                                    ///

                                    BW[f, c] = lmind;
                                    // update ChrList

                                    // tmp new member
                                    LbChr = new double[4]; //clean
                                    LbChr[1] = Zenith[f, c];
                                    LbChr[2] = Azimuth[f, c];

                                    // actualize  this information in LbChrList
                                    double[] LbChrOld = new double[4];
                                    double[] LbChrNew = new double[4];
                                   

                                    LbChrOld = LbChrList[lmind];// tmpNumerator.Current;

                                    LbChrNew[0] = lmind; //label
                                    LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + LbChr[1]) / (LbChrOld[3] + 1); // new zenith mean
                                    LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + LbChr[2]) / (LbChrOld[3] + 1); // new azimuth mean
                                    LbChrNew[3] = LbChrOld[3] + 1; //# members

                                    LbChrList.RemoveAt(lmind);
                                    LbChrList.Insert(lmind, LbChrNew); // actualize the region information

                                    ///else nothing to do, it's a watershed label ==0
                                    ///

                                    ///REGION MERGIN, look in distancesM
                                    /// sellect the minimum, only can mix two regions
                                    double minMatrixDistance = 10000;
                                    int LABEL1 = 0;
                                    int LABEL2 = 0;

                                    //look for the minimum value
                                    for (int ii = 0; ii < dflbCounter; ii++)
                                        for (int jj = ii + 1; jj < dflbCounter; jj++)
                                            if (distancesM[ii, jj] < threshold)
                                            {
                                                minMatrixDistance = distancesM[ii, jj];
                                                LABEL1 = dflb[ii];
                                                LABEL2 = dflb[jj];
                                            }

                                    if (minMatrixDistance < threshold)
                                    {///region mergin

                                        //select all pixels labeled with label1 and asign label2,and update this information in LbChrList
                                        List<int[]> withLabel1 = find(BW, this.rows, this.col, LABEL1);
                                        IEnumerator withLabel1Numerator = withLabel1.GetEnumerator();
                                        while (withLabel1Numerator.MoveNext())
                                        {
                                            coord = (int[])withLabel1Numerator.Current;
                                            BW[coord[0], coord[1]] = LABEL2;
                                        }

                                        // actualize  this information in LbChrList
                                        double[] LbChrOld1 = new double[4];
                                        double[] LbChrOld2 = new double[4];
                                        double[] LbChrNew2 = new double[4];
                                        
                                        LbChrOld1 = LbChrList[LABEL1];// tmpNumerator.Current;
                                        
                                        LbChrOld2 = LbChrList[LABEL2];// tmpNumerator.Current;

                                        LbChrNew2[0] = LABEL2; //label
                                        LbChrNew2[1] = (LbChrOld1[1] * LbChrOld1[3] + LbChrOld2[1] * LbChrOld2[3]) / (LbChrOld1[3] + LbChrOld2[3]); // new zenith mean
                                        LbChrNew2[2] = (LbChrOld1[2] * LbChrOld1[3] + LbChrOld2[2] * LbChrOld2[3]) / (LbChrOld1[3] + LbChrOld2[3]); // new azimuth mean
                                        LbChrNew2[3] = LbChrOld1[3] + LbChrOld2[3]; //# members

                                    
                                        LbChrList[LABEL2]= LbChrNew2; // actualize the region information

                                        //clear de label1 info
                                       
                                        LbChrList[LABEL1]= null;
                                        ///


                                    }

                                }
                            }
                        }


                    }
                }


            }

            ArrayList R = new ArrayList();
            
            int[,] BW2 = bwOrdering(BW);
            R.Add(BW2);

            List<double[]> RL = new List<double[]>(); // relation with the label and their chromaticity (Zen and Azi)

            RL.Add( new double[2]);// is 0 by default
            double[] aux = new double[2];
            int LABELX = 0;//clean            

            int nLabels = max(BW2, rows,col);
            int nf =0; // clean
            int nc = 0;
            
            //for each label
            for (int LABEL = 1; LABEL <= nLabels; LABEL++)
            {
                bool salir = false;
                // look for the first occurrence of this label in BW2
                for (int i = 0; i < rows; i++)
                {
                    for (int j = 0; j < col; j++)
                        if (BW2[i, j] == LABEL)
                        {
                            nf = i; nc = j; 
                            salir = true;
                            break;
                        }
                    if (salir)
                        break;
                }
                LABELX = BW[nf, nc];
                aux = new double[2];//clear
                aux[0] = LbChrList[LABELX][1];//zenith
                aux[1] = LbChrList[LABELX][2];//azimuth
                RL.Add(aux);

            }


            R.Add(RL);
            return R; // a labeled image
            
        }
        /// <summary>
        /// watershed segementation with chromatic criterium. watershed is visible.
        /// </summary>
        /// <param name="IG"></param>
        /// Intensity image
        /// <param name="threshold"></param>
        /// Threshold for region mergin
        /// <param name="steps"></param>
        /// amount of steps (slides)
        /// <returns>ArrayList with three values. first a int[,] is the labeled image. Second and thrird double[,] and double[,]. Are the zenith and azimuthal angles</returns>
        public ArrayList watershedV(double[,] IG, double threshold, int steps)
        {



            // In this matriz we are going to labeling all pixels
            int[,] BW = new int[this.rows, this.col];

            //matrix wich save if one pixel have been labeled
            Boolean[,] Labeled = new Boolean[this.rows, this.col];

            //initialize to false
            for (int i = 0; i < this.rows; i++)
                for (int j = 0; j < this.col; j++)
                    Labeled[i, j] = false;


            //the correspondence label, chrmaticity and members amount
            double[] LbChr = new double[4]; //[ label, zenith, azimuth, #memebers]
            List<double[]> LbChrList = new List<double[]>(); //list of labeled regions. It has Lbchr types.

            double mn = min(IG, this.rows, this.col);
            double jump = (max(IG, this.rows, this.col) - mn) / (double)steps;

            double thi = mn + jump; //new threshold intensity

            //find the pixels with intensity lower than thi
            //s = find(IG < thi);
            //BW(s) = 1;

            List<int[]> LoadedPixels = findless(IG, this.rows, this.col, thi);
            IEnumerator LoadedPixelsNumerator = LoadedPixels.GetEnumerator();
            //put this values to 1 in BW
            int[] coord; // generic variable for coordinates catching
            while (LoadedPixelsNumerator.MoveNext())
            {
                coord = (int[])LoadedPixelsNumerator.Current;
                BW[coord[0], coord[1]] = 1;
            }

            //labeling
            // [BW, n_LbChr] = bwlabel(BW);
            BW = bwlabel(BW, this.rows, this.col); // MAtrix labeling

            //inicializamos LbChrList
            //for each region, we'll save the label, the azimuth and zenith means and the  members amount.
            int nReg = max(BW, this.rows, this.col);

            for (int i = 1; i <= nReg; i++)
            {
                List<int[]> members = find(BW, this.rows, this.col, i);
               
                double zenith_sum = 0;
                double azimuth_sum = 0;
                double[] LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
                coord = new int[2];//clean

                for (int i2 = 0; i2 < members.Count; i2++)
                {
                    coord = members[i2];
                    zenith_sum = zenith_sum + this.Zenith[coord[0], coord[1]];
                    azimuth_sum = azimuth_sum + this.Azimuth[coord[0], coord[1]];
                    //mark as labeled
                    Labeled[coord[0], coord[1]] = true;
                }

                //while (membersNumerator.MoveNext())
                //{
                //    coord = membersNumerator.Current;
                //    zenith_sum = zenith_sum + this.Zenith[coord[0], coord[1]];
                //    azimuth_sum = azimuth_sum + this.Azimuth[coord[0], coord[1]];
                //    //mark as labeled
                //    Labeled[coord[0], coord[1]] = true;
                //}
                LbChrAux[0] = (double)i;
                LbChrAux[1] = zenith_sum / (double)members.Count;
                LbChrAux[2] = azimuth_sum / (double)members.Count;
                LbChrAux[3] = (double)members.Count;

                //save this information
                LbChrList.Add(LbChrAux);

            }


            //iteration for all image

            for (int iterations = 2; iterations < steps; iterations++)
            {

                //temporal matrix with the pixels for labeling
                int[,] toLabel = new int[this.rows, this.col];

                thi = mn + (jump * iterations);//new treshold intensity

                //look for the new unlabeled pixels
                List<int[]> members = findless(IG, this.rows, this.col, thi);
               // IEnumerator membersNumerator = members.GetEnumerator();
                //int[] coord = new int[2];
                for(int i = 0; i< members.Count;i++)               
                {
                    coord = members[i];
                    if (Labeled[coord[0], coord[1]] == false)//is unlabeled
                        toLabel[coord[0], coord[1]] = 1; //mark for labeling

                }

                List<int[]> PixelsToLabeling = find(toLabel, this.rows, this.col, 1);
                //IEnumerator PixelsToLabelingNumerator = PixelsToLabeling.GetEnumerator();

                //for each new member to labeling

                for(int i3 = 0 ; i3 < PixelsToLabeling.Count; i3++)
               
                {
                    int f, c, a1, a2, a3, a4, a6, a7, a8, a9;
                    coord = PixelsToLabeling[i3];
                    f = coord[0];
                    c = coord[1];
                    //mark this pixel as labeled
                    Labeled[f, c] = true;

                    // miramos en sus 8 vecinos, 
                    // si no tiene vecinos -> asignamos etiqueta nueva 
                    // si tiene un vecino etiquetado,-> le asignamos su etiqueta
                    // si tiene varios vecinos, con la misma etiqueta,-> asignamos la etiqueta
                    // si tiene varios vecinos, con diferentes etiquetas,-> 
                    //     si las distancias cromáticas entre los vecinos son > thr -> asignamos la etiqueta del vecino menor distancia cromática
                    //     si tiene vecinos, con distancia cromática < thr -> unimos las regiones vecinas con una etiqueta de las dos y al px le asignmos la misma

                    // miramos en sus 8 vecinos, 
                    //contamos cuantos hay y los guardamos en una lista

                    a1 = 0; a2 = 0; a3 = 0; a4 = 0; a6 = 0; a7 = 0; a8 = 0; a9 = 0; //clean

                    if (f - 1 >= 0 & c - 1 >= 0)
                        a1 = BW[f - 1, c - 1];
                    if (f - 1 >= 0)
                        a2 = BW[f - 1, c];
                    if (f - 1 >= 0 & c + 1 < this.col)
                        a3 = BW[f - 1, c + 1];
                    if (c - 1 >= 0)
                        a4 = BW[f, c - 1];
                    if (c + 1 < this.col)
                        a6 = BW[f, c + 1];
                    if (f + 1 < this.rows & c - 1 >= 0)
                        a7 = BW[f + 1, c - 1];
                    if (f + 1 < this.rows)
                        a8 = BW[f + 1, c];
                    if (f + 1 < this.rows & c + 1 < this.col)
                        a9 = BW[f + 1, c + 1];

                    if ((a1 == 0) & (a2 == 0) & (a3 == 0) & (a4 == 0) & (a6 == 0) & (a7 == 0) & (a8 == 0) & (a9 == 0))
                    {
                        //it hasn't got labeled neighboors
                        int newLabel = LbChrList.Count; //new label

                        BW[f, c] = newLabel;

                        // save this information in LbChrList
                        double[] LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
                        LbChrAux[0] = (double)newLabel; //label
                        LbChrAux[1] = Zenith[f, c];
                        LbChrAux[2] = Azimuth[f, c];
                        LbChrAux[3] = 1; //#members
                        LbChrList.Add(LbChrAux);
                    }
                    else//the current location has some labeled neighboor
                    { //count the labels amoung
                        int lc = 0; // local label counter
                        int[] labels = new int[8];
                        if (a1 > 0) { labels[lc++] = a1; }
                        if (a2 > 0) { labels[lc++] = a2; }
                        if (a3 > 0) { labels[lc++] = a3; }
                        if (a4 > 0) { labels[lc++] = a4; }
                        if (a6 > 0) { labels[lc++] = a6; }
                        if (a7 > 0) { labels[lc++] = a7; }
                        if (a8 > 0) { labels[lc++] = a8; }
                        if (a9 > 0) { labels[lc++] = a9; }

                        int[] dflb = new int[8]; //vector counter of different labels
                        int dflbCounter = 0;
                        int _lb = 0;
                        for (int k = 0; k < lc; k++)
                        {
                            _lb = labels[k];
                            int esta = 0;
                            for (int kk = 0; kk < k; kk++)
                                if (labels[kk] == _lb)
                                    esta = 1;
                            if (esta == 0)
                                dflb[dflbCounter++] = _lb;
                        }

                        if (dflbCounter == 1)
                        {
                            //only has a neighboor
                            // si solo tiene un vecinos con una etiqueta ->actualizamos los valores crom de la etiqueta y le asignamos su etiqueta,
                            int thisLabel = dflb[0]; //label
                            BW[f, c] = thisLabel;

                            // actualize  this information in LbChrList
                            double[] LbChrOld = new double[4];
                            double[] LbChrNew = new double[4];
                            
                            LbChrOld = LbChrList[thisLabel];// (double[])tmpNumerator.Current;

                            LbChrNew[0] = thisLabel; //label
                            LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + Zenith[f, c]) / (LbChrOld[3] + 1); // new zenith mean
                            LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + Azimuth[f, c]) / (LbChrOld[3] + 1); // new azimuth mean
                            LbChrNew[3] = LbChrOld[3] + 1; //# members

                            
                            LbChrList[thisLabel]= LbChrNew; // actualize the region information


                        }
                        else
                        {
                            //has more than one neighboor
                            // Hallamos la distancia entre las labels de vecinos
                            double[,] distancesM = new double[dflbCounter, dflbCounter]; // matrix for the distances between the regions
                            //inicializamos a 1
                            for (int i = 0; i < dflbCounter; i++)
                                for (int j = 0; j < dflbCounter; j++)
                                    distancesM[i, j] = 1;

                            double[] distancesV = new double[dflbCounter]; //distance between the regions and the pixels[f,c]
                            //inicializamos a 1
                            for (int i = 0; i < dflbCounter; distancesV[i++] = 1) ;

                            //fill distancesM
                            //diagonal superior
                            double[] Chr1 = new double[2]; //chromaticity of label1
                            double[] Chr2 = new double[2]; //chromaticity of label2


                            int label1 = 0;
                            int label2 = 0;
                            double v0, v1;

                            for (int i = 0; i < dflbCounter; i++)
                                for (int j = i + 1; j < dflbCounter; j++)
                                {
                                    label1 = dflb[i];
                                    label2 = dflb[j];

                                    
                                    LbChr = new double[4]; //clean

                                    LbChr = LbChrList[label1];// tmpNumerator.Current;
                                    Chr1[0] = LbChr[1];
                                    Chr1[1] = LbChr[2];

                                    
                                    LbChr = new double[4]; //clean

                                    LbChr = LbChrList[label2];// tmpNumerator.Current;                                           
                                    Chr2[0] = LbChr[1];
                                    Chr2[1] = LbChr[2];

                                    v0 = (Chr1[0] - Chr2[0]) * (Chr1[0] - Chr2[0]);
                                    v1 = (Chr1[1] - Chr2[1]) * (Chr1[1] - Chr2[1]);
                                    distancesM[i, j] = Math.Sqrt(v0 + v1);

                                }//distancesM is filled

                            // Mezclamos las dos regiones con menor d si es menor que threshold
                            double minM = 10000;

                            for (int i = 0; i < dflbCounter; i++)
                                for (int j = i + 1; j < dflbCounter; j++)
                                    if (distancesM[i, j] < minM)
                                    {
                                        label1 = dflb[i];
                                        label2 = dflb[j];
                                        minM = distancesM[i, j];
                                    }


                            if (minM < threshold)//mix the regions
                            {

                                //add the current pixel tos ome of this labels
                                int thisLabel = label1;
                                BW[f, c] = thisLabel;


                                // actualize  this information in LbChrList
                                double[] LbChrOld = new double[4];
                                double[] LbChrNew = new double[4];
                               
                                LbChrOld = LbChrList[thisLabel];// (double[])tmpNumerator.Current;

                                LbChrNew[0] = thisLabel; //label
                                LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + Zenith[f, c]) / (LbChrOld[3] + 1); // new zenith mean
                                LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + Azimuth[f, c]) / (LbChrOld[3] + 1); // new azimuth mean
                                LbChrNew[3] = LbChrOld[3] + 1; //# members

                               
                                LbChrList[thisLabel]= LbChrNew; // actualize the region information


                                //mixing process
                                //select all pixels labeled with label1 and asign label2,and update this information in LbChrList
                                List<int[]> withLabel1 = find(BW, this.rows, this.col, label1);
                               // IEnumerator withLabel1Numerator = withLabel1.GetEnumerator();

                                for(int i4 =0; i4 <withLabel1.Count; i4++)
                                {
                                    coord = withLabel1[i4];
                                    BW[coord[0], coord[1]] = label2;
                                }

                                // actualize  this information in LbChrList
                                double[] LbChrOld1 = new double[4];
                                double[] LbChrOld2 = new double[4];
                                double[] LbChrNew2 = new double[4];

                               
                                LbChrOld1 = LbChrList[label1];// tmpNumerator.Current;

                                
                                LbChrOld2 = LbChrList[label2];// tmpNumerator.Current;

                                LbChrNew2[0] = label2; //label
                                LbChrNew2[1] = (LbChrOld1[1] * LbChrOld1[3] + LbChrOld2[1] * LbChrOld2[3]) / (LbChrOld1[3] + LbChrOld2[3]); // new zenith mean
                                LbChrNew2[2] = (LbChrOld1[2] * LbChrOld1[3] + LbChrOld2[2] * LbChrOld2[3]) / (LbChrOld1[3] + LbChrOld2[3]); // new azimuth mean
                                LbChrNew2[3] = LbChrOld1[3] + LbChrOld2[3]; //# members

                                
                                LbChrList[label2]= LbChrNew2; // actualize the region information

                                //clear de label1 info
                                
                                LbChrList[label1]= null;
                                ///


                            }//end mixing regions
                            else
                            {//there isn't close regions, we'll asign the closest region label
                                int lmind = 0;
                                BW[f, c] = lmind;
                                // update ChrList

                                // tmp new member
                                LbChr = new double[4]; //clean
                                LbChr[1] = Zenith[f, c];
                                LbChr[2] = Azimuth[f, c];

                                // actualize  this information in LbChrList
                                double[] LbChrOld = new double[4];
                                double[] LbChrNew = new double[4];
                               
                                LbChrOld = LbChrList[lmind];// tmpNumerator.Current;

                                LbChrNew[0] = lmind; //label
                                LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + LbChr[1]) / (LbChrOld[3] + 1); // new zenith mean
                                LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + LbChr[2]) / (LbChrOld[3] + 1); // new azimuth mean
                                LbChrNew[3] = LbChrOld[3] + 1; //# members

                                
                                LbChrList[lmind]= LbChrNew; // actualize the region information



                            }



                        }


                    }//end current location has some neighboor



                }//pixel labeled



            }//end iterations

            ArrayList R = new ArrayList();

            int[,] BW2 = bwOrdering(BW);
            R.Add(BW2);

            List<double[]> RL = new List<double[]>(); // relation with the label and their chromaticity (Zen and Azi)

            RL.Add(new double[2]);// is 0 by default
            double[] aux = new double[2];
            int LABELX = 0;//clean            

            int nLabels = max(BW2, rows, col);
            int nf = 0; // clean
            int nc = 0;

            //for each label
            for (int LABEL = 1; LABEL <= nLabels; LABEL++)
            {
                bool salir = false;
                // look for the first occurrence of this label in BW2
                for (int i = 0; i < rows; i++)
                {
                    for (int j = 0; j < col; j++)
                        if (BW2[i, j] == LABEL)
                        {
                            nf = i; nc = j;
                            salir = true;
                            break;
                        }
                    if (salir)
                        break;
                }
                LABELX = BW[nf, nc];
                aux = new double[2];//clear
                aux[0] = LbChrList[LABELX][1];//zenith
                aux[1] = LbChrList[LABELX][2];//azimuth
                RL.Add(aux);

            }


            R.Add(RL);
            return R; // a labeled image with the index chromaticities
            


        }

        
        public ArrayList fastSegmentation(double threshold)
        {
            // variables for stadistical behavior
            int merged = 0;
            int copy1 = 0;
            int copy2 = 0;
            int new1 = 0;
            int new2 = 0;

            int[,] BW = new int[this.rows, this.col]; //matrix label
            int a1, a2, a3, a4; //neighboors
            double[] LbChr = new double[4]; //[ label, zenith, azimuth, #memebers]
            List<double[]> LbChrList = new List<double[]>(); //list of labeled regions. It has Lbchr types.

            List<List<int[]>> LaLista = new List<List<int[]>>();

            //put a label to the first pixel/////////////////////////////////////////////////
            LbChrList.Add(LbChr); //inicialize the first as 0

            int newLabel = LbChrList.Count; //new label

            BW[0, 0] = newLabel;
            // save this information in LbChrList
            double[] LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
            LbChrAux[0] = newLabel; //label
            LbChrAux[1] = Zenith[0, 0];
            LbChrAux[2] = Azimuth[0, 0];
            LbChrAux[3] = 1; //#members
            LbChrList.Add(LbChrAux);
            //put a label to the first pixel/////////////////////////////////////////////////



            //for all pixels in the image, and only for one time
            // current pixel is not labeled
            for (int f = 0; f < this.rows; f++)           
                for (int c = 0; c < this.col; c++)
                {

                    

                    a1 = 0; a2 = 0; a3 = 0; a4 = 0; //clean
                    //load
                    if (f - 1 >= 0 & c - 1 >= 0)
                        a1 = BW[f - 1, c - 1];
                    if (f - 1 >= 0)
                        a2 = BW[f - 1, c];
                    if (f - 1 >= 0 & c + 1 < this.col)
                        a3 = BW[f - 1, c + 1];
                    if (c - 1 >= 0)
                        a4 = BW[f, c - 1];
                    // we always have a label as minimum
                    int lc = 0; // local label counter
                    int[] labels = new int[4];
                    if (a1 > 0) { labels[lc++] = a1; }
                    if (a2 > 0) { labels[lc++] = a2; }
                    if (a3 > 0) { labels[lc++] = a3; }
                    if (a4 > 0) { labels[lc++] = a4; }

                    /// the neighboors has only one label -> then we asign this label

                    int[] dflb = new int[8]; //counter of different labels
                    int dflbCounter = 0;
                    int _lb = 0;
                    for (int k = 0; k < lc; k++)
                    {
                        _lb = labels[k];
                        int esta = 0;
                        for (int kk = 0; kk < k; kk++)
                            if (labels[kk] == _lb)
                                esta = 1;
                        if (esta == 0)
                            dflb[dflbCounter++] = _lb;
                    }

                    if (dflbCounter == 1)
                    {
                        //only has a neighboor
                        int LABEL = dflb[0];
                        // calculate the distance to this label
                        double d,v0,v1;
                        double[] Chr1 = new double[2];
                        double[] Chr2 = new double[2];

                        LbChr = new double[4]; //clean
                       
                        LbChr =  LbChrList[LABEL];
                        
                        Chr1[0] = LbChr[1];
                        Chr1[1] = LbChr[2];                        
                        
                                                          
                        Chr2[0] = Zenith[f,c];;
                        Chr2[1] = Azimuth[f,c];

                        v0 = (Chr1[0] - Chr2[0]) * (Chr1[0] - Chr2[0]);
                        v1 = (Chr1[1] - Chr2[1]) * (Chr1[1] - Chr2[1]);
                        d = Math.Sqrt(v0 + v1);
                        


                        if (d < threshold)
                        {

                            copy1++;

                            int thisLabel = dflb[0]; //label
                            BW[f, c] = thisLabel;

                            // actualize  this information in LbChrList
                            double[] LbChrOld = new double[4];
                            double[] LbChrNew = new double[4];
                            LbChrOld =  LbChrList[thisLabel];

                            
                            LbChrNew[0] = thisLabel; //label
                            LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + Zenith[f, c]) / (LbChrOld[3] + 1); // new zenith mean
                            LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + Azimuth[f, c]) / (LbChrOld[3] + 1); // new azimuth mean
                            LbChrNew[3] = LbChrOld[3] + 1; //# members

                           // LbChrList.RemoveAt(thisLabel);
                           // LbChrList.Insert(thisLabel, LbChrNew); // actualize the region information
                            LbChrList[thisLabel]=LbChrNew; // actualize the region information
                        }
                        else
                        {//asign new label
                            new1++;

                            newLabel = LbChrList.Count; //new label

                            BW[f, c] = newLabel;
                            // save this information in LbChrList
                            LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
                            LbChrAux[0] = newLabel; //label
                            LbChrAux[1] = Zenith[f, c];
                            LbChrAux[2] = Azimuth[f, c];
                            LbChrAux[3] = 1; //#members
                            LbChrList.Add(LbChrAux);

                        }


                    }
                    else
                    {
                        //look for the distances matrix
                        /// the neighboors has different labels 
                        /// We must look for the chromatic dintance to them. and if some of them are smaler than 
                        /// 'threshold' we must merging then or asing this label. and actualize this information.
                        /// 
                        //lc is the labels neighboor count
                        //labels is a vector with the neighboors labels
                        double[,] distancesM = new double[dflbCounter, dflbCounter]; // matrix for the distances between the regions
                        //inicializamos a 1
                        for (int i = 0; i < dflbCounter; i++)
                            for (int j = 0; j < dflbCounter; j++)
                                distancesM[i, j] = 1;

                        double[] distancesV = new double[dflbCounter]; //distance between the regions and the pixels[f,c]
                        //inicializamos a 1
                        for (int i = 0; i < dflbCounter; distancesV[i++] = 1) ;

                        //fill distancesM
                        //diagonal superior
                        double[] Chr1 = new double[2]; //chromaticity of label1
                        double[] Chr2 = new double[2]; //chromaticity of label2


                        int label1 =0;
                        int label2 = 0 ;
                        
                        double v0, v1;

                        for (int ii = 0; ii < dflbCounter; ii++)
                            for (int jj = ii + 1; jj < dflbCounter; jj++)
                            {
                                label1 = dflb[ii];
                                label2 = dflb[jj];

                                LbChr = new double[4]; //clean

                                LbChr = LbChrList[label1];// tmpNumerator.Current;
                                Chr1[0] = LbChr[1];
                                Chr1[1] = LbChr[2];

                                LbChr = new double[4]; //clean

                                LbChr = LbChrList[label2];// tmpNumerator.Current;                                           
                                Chr2[0] = LbChr[1];
                                Chr2[1] = LbChr[2];

                                v0 = (Chr1[0] - Chr2[0]) * (Chr1[0] - Chr2[0]);
                                v1 = (Chr1[1] - Chr2[1]) * (Chr1[1] - Chr2[1]);
                                distancesM[ii, jj] = Math.Sqrt(v0 + v1);
                            }//distancesM is fill

                        // Mezclamos las dos regiones con menor d si es menor que threshold
                        double minM = 10000;

                        for (int i = 0; i < dflbCounter; i++)
                            for (int j = i + 1; j < dflbCounter; j++)
                                if (distancesM[i, j] < minM)
                                {
                                    label1 = dflb[i];
                                    label2 = dflb[j];
                                    minM = distancesM[i, j];
                                }



                        //// variables for mergin
                        double[] LbChrOld = new double[4];
                        double[] LbChrNew = new double[4];
                        int[] coord;
                        double[] LbChrOld1 = new double[4];
                        double[] LbChrOld2 = new double[4];
                        double[] LbChrNew2 = new double[4];

                        if (minM < threshold)//mix the regions
                        {
                          
                            merged++;
                            //////////////////////////////////////////////////////////////////////////////////////
                            /////////////////////////       REGION MERGIN      ///////////////////////////////////
                            //////////////////////////////////////////////////////////////////////////////////////
                            //wE are going to change the label of the little region
                            double membersL1 = LbChrList[label1][3];
                            double membersL2 = LbChrList[label2][3];
                            int labelToKeep = 0;
                            int labelToChange = 0;

                            if (membersL1 < membersL2)
                            {
                                labelToChange = label1;
                                labelToKeep = label2;
                            }
                            else
                            {
                                labelToChange = label2;
                                labelToKeep = label1;
                            }

                            //int thisLabel = label1; //label to keep
                            BW[f, c] = labelToKeep;


                            // actualize  this information in LbChrList

                          
                            LbChrOld = LbChrList[labelToKeep];// (double[])tmpNumerator.Current;

                            LbChrNew[0] = labelToKeep; //label
                            LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + Zenith[f, c]) / (LbChrOld[3] + 1); // new zenith mean
                            LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + Azimuth[f, c]) / (LbChrOld[3] + 1); // new azimuth mean
                            LbChrNew[3] = LbChrOld[3] + 1; //# members

                            LbChrList[labelToKeep]=LbChrNew;

                            //mixing process
                            //select all pixels labeled with label1 and asign label2,and update this information in LbChrList
                            List<int[]> withLabel1 = find(BW, this.rows, this.col, labelToChange);
                            
                            //int[] coord;
                            for(int i2 =0; i2<withLabel1.Count; i2++)
                            {
                                coord = withLabel1[i2];
                                BW[coord[0], coord[1]] = labelToKeep;
                            }

                            // actualize  this information in LbChrList

                            LbChrOld1 = LbChrList[labelToKeep];//

                            LbChrOld2 = LbChrList[labelToChange];// 
                            LbChrNew2[0] = labelToKeep; //label
                            LbChrNew2[1] = (LbChrOld1[1] * LbChrOld1[3] + LbChrOld2[1] * LbChrOld2[3]) / (LbChrOld1[3] + LbChrOld2[3]); // new zenith mean
                            LbChrNew2[2] = (LbChrOld1[2] * LbChrOld1[3] + LbChrOld2[2] * LbChrOld2[3]) / (LbChrOld1[3] + LbChrOld2[3]); // new azimuth mean
                            LbChrNew2[3] = LbChrOld1[3] + LbChrOld2[3]; //# members

                            LbChrList[labelToKeep]= LbChrNew2; // actualize the region information

                            //clear de label1 info
                            LbChrList[labelToChange]= null;

                            //////////////////////////////////////////////////////////////////////////////////////
                            /////////////////////////   END REGION MERGIN      ///////////////////////////////////
                            //////////////////////////////////////////////////////////////////////////////////////
                            ///
                        }
                        else
                        {// no hay parecido entre los vecinos

                            // look if some neighboor label is close enough
                            double mind = 1000000000;//minimum distance
                            int lmind = 0;

                            //fill distancesV
                            Chr1 = new double[2]; //chromaticity of label1 clear
                            Chr2 = new double[2]; //chromaticity of label2 clear
                            Chr1[1] = Azimuth[f, c];
                            Chr1[0] = Zenith[f, c];
                            label1 = 0;
                         
                            for (int ii = 0; ii < dflbCounter; ii++)
                                  {
                                    label1 = dflb[ii];                                  

                                    LbChr = new double[4]; //clean

                                    LbChr = LbChrList[label1];// tmpNumerator.Current;
                                    Chr1[0] = LbChr[1];
                                    Chr1[1] = LbChr[2];

                                    v0 = (Chr1[0] - Chr2[0]) * (Chr1[0] - Chr2[0]);
                                    v1 = (Chr1[1] - Chr2[1]) * (Chr1[1] - Chr2[1]);
                                    distancesV[ii] = Math.Sqrt(v0 + v1);
                                }//distancesV is fill


                            for (int ii = 0; ii < dflbCounter; ii++)
                                if (distancesV[ii] < mind)
                                {
                                    mind = distancesV[ii];
                                    lmind = dflb[ii];///sure??
                                }

                            ///LABELING
                            ///
                            if (mind < threshold)
                            {
                                copy2++;

                                BW[f, c] = lmind;
                                // update ChrList

                                // tmp new member
                                LbChr = new double[4]; //clean
                                LbChr[1] = Zenith[f, c];
                                LbChr[2] = Azimuth[f, c];

                                // actualize  this information in LbChrList
                               // double[] LbChrOld = new double[4];
                               // double[] LbChrNew = new double[4];
                               

                                LbChrOld = LbChrList[lmind];// tmpNumerator.Current;

                                LbChrNew[0] = lmind; //label
                                LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + LbChr[1]) / (LbChrOld[3] + 1); // new zenith mean
                                LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + LbChr[2]) / (LbChrOld[3] + 1); // new azimuth mean
                                LbChrNew[3] = LbChrOld[3] + 1; //# members

                                
                                LbChrList[lmind]= LbChrNew; // actualize the region information
                            }
                            else
                            {// new label
                                new2++;

                                newLabel = LbChrList.Count; //new label

                                BW[f, c] = newLabel;
                                // save this information in LbChrList
                                LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
                                LbChrAux[0] = newLabel; //label
                                LbChrAux[1] = Zenith[f, c];
                                LbChrAux[2] = Azimuth[f, c];
                                LbChrAux[3] = 1; //#members
                                LbChrList.Add(LbChrAux);
                            }

                        }

                    }//end else
                  //  sw.Stop();
                }//end for. all point are been labeled

            ArrayList R = new ArrayList();

            int[,] BW2 = bwOrdering(BW);
            R.Add(BW2);

            List<double[]> RL = new List<double[]>(); // relation with the label and their chromaticity (Zen and Azi)

            RL.Add(new double[2]);// is 0 by default
            double[] aux = new double[2];
            int LABELX = 0;//clean            

            int nLabels = max(BW2, rows, col);
            int nf = 0; // clean
            int nc = 0;

            //for each label
            for (int LABEL = 1; LABEL <= nLabels; LABEL++)
            {
                bool salir = false;
                // look for the first occurrence of this label in BW2
                for (int i = 0; i < rows; i++)
                {
                    for (int j = 0; j < col; j++)
                        if (BW2[i, j] == LABEL)
                        {
                            nf = i; nc = j;
                            salir = true;
                            break;
                        }
                    if (salir)
                        break;
                }
                LABELX = BW[nf, nc];
                aux = new double[2];//clear
                aux[0] = LbChrList[LABELX][1];//zenith
                aux[1] = LbChrList[LABELX][2];//azimuth
                RL.Add(aux);

            }


            R.Add(RL);
            return R; // a labeled image with the index chromaticities
            

        }

        public ArrayList fastSegmentation2(double threshold, double Ap, double Bp, double Cp)
        {
            // variables for stadistical behavior
            int merged = 0;
            int copy1 = 0;
            int copy2 = 0;
            int new1 = 0;
            int new2 = 0;

            int[,] BW = new int[this.rows, this.col]; //matrix label

            int[,] neighboors = new int[4, 3];//[label, row, col]

            double[] LbChr = new double[4]; //[ label, zenith, azimuth, #memebers]
            List<double[]> LbChrList = new List<double[]>(); //list of labeled regions. It has Lbchr types.


            //put a label to the first pixel/////////////////////////////////////////////////
            LbChrList.Add(LbChr); //inicialize the first as 0

            int newLabel = LbChrList.Count; //new label

            BW[0, 0] = newLabel;
            // save this information in LbChrList
            double[] LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
            LbChrAux[0] = newLabel; //label
            LbChrAux[1] = Zenith[0, 0];
            LbChrAux[2] = Azimuth[0, 0];
            LbChrAux[3] = 1; //#members
            LbChrList.Add(LbChrAux);
            //put a label to the first pixel/////////////////////////////////////////////////





  

            //for all pixels in the image, and only for one time
            // current pixel is not labeled
            for (int f = 0; f < this.rows; f++)
                for (int c = 0; c < this.col; c++)
                {

                    //clean
                    neighboors = new int[4, 3];//[label, row, col] 
                    //load
                    if (f - 1 >= 0 & c - 1 >= 0)
                    {
                        neighboors[0,0] = BW[f - 1, c - 1];
                        neighboors[0,1] = f - 1;
                        neighboors[0,2] = c - 1;
                    }
                    if (f - 1 >= 0)
                    {
                        neighboors[1, 0] = BW[f - 1, c];
                        neighboors[1,1] = f - 1;
                        neighboors[1,2] = c;
                    }
                    if (f - 1 >= 0 & c + 1 < this.col)
                    {
                        neighboors[2, 0] = BW[f - 1, c + 1];
                        neighboors[2, 1] = f - 1;
                        neighboors[2, 2] = c + 1;
                    }
                    if (c - 1 >= 0)
                    {
                        neighboors[3, 0] = BW[f, c - 1];
                        neighboors[3, 1] = f;
                        neighboors[3, 2] = c - 1;
                    }

                    //if we have some labeled neighboor (exception first iteration)
                    if(neighboors[0,0] != 0 |neighboors[1,0] != 0 | neighboors[2,0] != 0 | neighboors[3,0] != 0 ){

                    // we always have a label as minimum
                    /// the neighboors has only one label -> then we asign this label

                    int[] dflb = new int[4]; //counter of different labels (as maximum 4)
                    int dflbCounter = 0;
                    int _lb = 0;
                    for (int i = 0; i < 4; i++)
                    {
                        if (neighboors[i, 0] != 0)//only have in account labels different to 0                        {
                        {
                            _lb = neighboors[i, 0];
                            int esta = 0;
                            for (int j = 0; j < i; j++)
                                if (neighboors[j, 0] == _lb)
                                    esta = 1;
                            if (esta == 0)
                                dflb[dflbCounter++] = _lb;
                        }
                    }

                    if (dflbCounter == 1)
                    {
                        //only has a neighboor
                        int LABEL = dflb[0];
                        
                        // calculate the distances to the neighboors with this label
                        double[] distancesV = new double[4];//distances vector
                        for (int i = 0; i < 4; distancesV[i++] = 1) ;//initialize

                        //fill distancesV                        
                        for (int i =0; i<4;i++)
                            if(neighboors[i,0] !=0) // if it has label look for the distance to this pixel/label
                            {
                                double Lzen =0; //zenith angle of the neigboor label
                                double Lazi =0; //azimuth angle of the neigboor label
                                double Li =0; // intensity of the neighboor pixel
                                int labeli = neighboors[i,0];
                                Lzen = LbChrList[labeli][1];
                                Lazi = LbChrList[labeli][2];
                                switch (i)
                                {
                                    case 0:
                                        Li = L[f-1,c-1];
                                        break;
                                    case 1:
                                        Li = L[f-1,c];
                                        break;
                                    case 2:
                                        Li = L[f-1,c+1];
                                        break;
                                    default: //3
                                        Li = L[f,c-1];
                                        break;
                                }


                            distancesV[i] = distanceH(Zenith[f,c],Azimuth[f,c],L[f,c],Lzen, Lazi,Li,Ap,Bp,Cp);
                            }

                        //look for the little
                        double dmin = 1; //initialize to a hight value
                        for (int i = 0; i < 4; i++)
                            if (distancesV[i] < dmin)
                                dmin = distancesV[i];


                        if (dmin < threshold)
                        {

                            copy1++;

                            int thisLabel = dflb[0]; //label
                            BW[f, c] = thisLabel;

                            // actualize  this information in LbChrList
                            double[] LbChrOld = new double[4];
                            double[] LbChrNew = new double[4];
                            LbChrOld = LbChrList[thisLabel];


                            LbChrNew[0] = thisLabel; //label
                            LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + Zenith[f, c]) / (LbChrOld[3] + 1); // new zenith mean
                            LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + Azimuth[f, c]) / (LbChrOld[3] + 1); // new azimuth mean
                            LbChrNew[3] = LbChrOld[3] + 1; //# members

                            // LbChrList.RemoveAt(thisLabel);
                            // LbChrList.Insert(thisLabel, LbChrNew); // actualize the region information
                            LbChrList[thisLabel] = LbChrNew; // actualize the region information
                        }
                        else
                        {//asign new label
                            new1++;

                            newLabel = LbChrList.Count; //new label

                            BW[f, c] = newLabel;
                            // save this information in LbChrList
                            LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
                            LbChrAux[0] = newLabel; //label
                            LbChrAux[1] = Zenith[f, c];
                            LbChrAux[2] = Azimuth[f, c];
                            LbChrAux[3] = 1; //#members
                            LbChrList.Add(LbChrAux);

                        }


                    }
                    else
                    {
                        //look for the distances matrix
                        /// the neighboors has different labels 
                        /// We must look for the chromatic dintance to them. and if some of them are smaler than 
                        /// 'threshold' we must merging then or asing this label. and actualize this information.
                        /// 
                        //lc is the labels neighboor count
                        //labels is a vector with the neighboors labels
                        double[,] distancesM = new double[4,4]; // matrix for the distances between the regions
                        //inicializamos a 1
                        for (int i = 0; i < 4; i++)
                            for (int j = 0; j < 4; j++)
                                distancesM[i, j] = 1;

                        double[] distancesV = new double[4]; //distance between the regions and the pixels[f,c]
                        //inicializamos a 1
                        for (int i = 0; i < 4; distancesV[i++] = 1) ;

                        //fill distancesM
                        //diagonal superior

                        for (int i = 0; i < 4; i++)
                            for (int j = i + 1; j < 4;j++)
                                if (neighboors[i, 0] != 0 & neighboors[j, 0] != 0 & neighboors[i, 0] !=  neighboors[j, 0])// if both has label and are different each other                         {
                                {
                                    //es la distancia entre i y j
                                double Lzen =0; //zenith angle of the neigboor label
                                double Lazi =0; //azimuth angle of the neigboor label
                                double Li =0; // intensity of the neighboor pixel

                                double Pzen =0; //other pixel information
                                double Pazi =0;
                                double Pi =0;

                                int labeli = neighboors[i, 0];
                                int labelj = neighboors[j, 0];

                                Lzen = LbChrList[labeli][1];
                                Lazi = LbChrList[labeli][2];
                                
                                Pzen = LbChrList[labelj][1];
                                Pazi = LbChrList[labelj][2];

                                //look for the intensity i
                                switch (i)
                                {
                                    case 0:
                                        Li = L[f-1,c-1];
                                        break;
                                    case 1:
                                        Li = L[f-1,c];
                                        break;
                                    case 2:
                                        Li = L[f-1,c+1];
                                        break;
                                    default: //3
                                        Li = L[f,c-1];
                                        break;
                                }
                                    //look for the intensity j
                                switch (j)
                                {
                                    case 0:
                                        Pi = L[f-1,c-1];
                                        break;
                                    case 1:
                                        Pi = L[f-1,c];
                                        break;
                                    case 2:
                                        Pi = L[f-1,c+1];
                                        break;
                                    default: //3
                                        Pi = L[f,c-1];
                                        break;
                                }

                            
                            distancesM[i, j] = distanceH(Pzen,Pazi,Pi,Lzen, Lazi,Li,Ap,Bp,Cp);;
                            }
                                                                    
                            //distancesM is fill

                        // Mezclamos las dos regiones con menor d si es menor que threshold
                        double minM = 1;
                        int label1 =0;
                        int label2 =0;
                        for (int i = 0; i < 4; i++)
                            for (int j = i + 1; j < 4; j++)
                                if (distancesM[i, j] < minM)
                                {
                                    label1 = neighboors[i,0];
                                    label2 = neighboors[j,0];
                                    minM = distancesM[i, j];
                                }



                        //// variables for mergin
                        double[] LbChrOld = new double[4];
                        double[] LbChrNew = new double[4];
                        int[] coord;
                        double[] LbChrOld1 = new double[4];
                        double[] LbChrOld2 = new double[4];
                        double[] LbChrNew2 = new double[4];

                        if (minM < threshold)//mix the regions
                        {

                            merged++;
                            //////////////////////////////////////////////////////////////////////////////////////
                            /////////////////////////       REGION MERGIN      ///////////////////////////////////
                            //////////////////////////////////////////////////////////////////////////////////////
                            //wE are going to change the label of the little region
                            double membersL1 = LbChrList[label1][3];
                            double membersL2 = LbChrList[label2][3];
                            int labelToKeep = 0;
                            int labelToChange = 0;

                            if (membersL1 < membersL2)
                            {
                                labelToChange = label1;
                                labelToKeep = label2;
                            }
                            else
                            {
                                labelToChange = label2;
                                labelToKeep = label1;
                            }

                            //int thisLabel = label1; //label to keep
                            BW[f, c] = labelToKeep;


                            // actualize  this information in LbChrList


                            LbChrOld = LbChrList[labelToKeep];// (double[])tmpNumerator.Current;

                            LbChrNew[0] = labelToKeep; //label
                            LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + Zenith[f, c]) / (LbChrOld[3] + 1); // new zenith mean
                            LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + Azimuth[f, c]) / (LbChrOld[3] + 1); // new azimuth mean
                            LbChrNew[3] = LbChrOld[3] + 1; //# members

                            LbChrList[labelToKeep] = LbChrNew;

                            //mixing process
                            //select all pixels labeled with label1 and asign label2,and update this information in LbChrList
                            List<int[]> withLabel1 = find(BW, this.rows, this.col, labelToChange);

                            //int[] coord;
                            for (int i2 = 0; i2 < withLabel1.Count; i2++)
                            {
                                coord = withLabel1[i2];
                                BW[coord[0], coord[1]] = labelToKeep;
                            }

                            // actualize  this information in LbChrList

                            LbChrOld1 = LbChrList[labelToKeep];//

                            LbChrOld2 = LbChrList[labelToChange];// 
                            LbChrNew2[0] = labelToKeep; //label
                            LbChrNew2[1] = (LbChrOld1[1] * LbChrOld1[3] + LbChrOld2[1] * LbChrOld2[3]) / (LbChrOld1[3] + LbChrOld2[3]); // new zenith mean
                            LbChrNew2[2] = (LbChrOld1[2] * LbChrOld1[3] + LbChrOld2[2] * LbChrOld2[3]) / (LbChrOld1[3] + LbChrOld2[3]); // new azimuth mean
                            LbChrNew2[3] = LbChrOld1[3] + LbChrOld2[3]; //# members

                            LbChrList[labelToKeep] = LbChrNew2; // actualize the region information

                            //clear de label1 info
                            LbChrList[labelToChange] = null;

                            //////////////////////////////////////////////////////////////////////////////////////
                            /////////////////////////   END REGION MERGIN      ///////////////////////////////////
                            //////////////////////////////////////////////////////////////////////////////////////
                            ///
                        }
                        else
                        {// no hay parecido entre los vecinos

                            // look if some neighboor label is close enough
                            double mind = 1000000000;//minimum distance
                            int lmind = 0;

                            distancesV = new double[4];//distances vector clean
                            for (int i = 0; i < 4; distancesV[i++] = 1) ;//initialize

                            //fill distancesV                    
                        
                            for (int i =0; i<4;i++)
                            if(neighboors[i,0] !=0) // if it has label look for the distance to this pixel/label
                            {
                                double Lzen =0; //zenith angle of the neigboor label
                                double Lazi =0; //azimuth angle of the neigboor label
                                double Li =0; // intensity of the neighboor pixel

                                int labeli = neighboors[i, 0];

                                Lzen = LbChrList[labeli][1];
                                Lazi = LbChrList[labeli][2];
                                switch (i)
                                {
                                    case 0:
                                        Li = L[f-1,c-1];
                                        break;
                                    case 1:
                                        Li = L[f-1,c];
                                        break;
                                    case 2:
                                        Li = L[f-1,c+1];
                                        break;
                                    default: //3
                                        Li = L[f,c-1];
                                        break;
                                }


                            distancesV[i] = distanceH(Zenith[f,c],Azimuth[f,c],L[f,c],Lzen, Lazi,Li,Ap,Bp,Cp);
                            }


                            for (int i = 0; i < 4; i++)
                                if (distancesV[i] < mind)
                                {
                                    mind = distancesV[i];
                                    lmind = neighboors[i, 0];///sure??
                                }

                            ///LABELING
                            ///
                            if (mind < threshold)
                            {
                                copy2++;

                                BW[f, c] = lmind;
                                // update ChrList

                                // tmp new member
                                LbChr = new double[4]; //clean
                                LbChr[1] = Zenith[f, c];
                                LbChr[2] = Azimuth[f, c];

                                // actualize  this information in LbChrList
                                // double[] LbChrOld = new double[4];
                                // double[] LbChrNew = new double[4];


                                LbChrOld = LbChrList[lmind];// tmpNumerator.Current;

                                LbChrNew[0] = lmind; //label
                                LbChrNew[1] = (LbChrOld[1] * LbChrOld[3] + LbChr[1]) / (LbChrOld[3] + 1); // new zenith mean
                                LbChrNew[2] = (LbChrOld[2] * LbChrOld[3] + LbChr[2]) / (LbChrOld[3] + 1); // new azimuth mean
                                LbChrNew[3] = LbChrOld[3] + 1; //# members


                                LbChrList[lmind] = LbChrNew; // actualize the region information
                            }
                            else
                            {// new label
                                new2++;

                                newLabel = LbChrList.Count; //new label

                                BW[f, c] = newLabel;
                                // save this information in LbChrList
                                LbChrAux = new double[4]; //[ label, zenith, azimuth, #memebers]
                                LbChrAux[0] = newLabel; //label
                                LbChrAux[1] = Zenith[f, c];
                                LbChrAux[2] = Azimuth[f, c];
                                LbChrAux[3] = 1; //#members
                                LbChrList.Add(LbChrAux);
                            }
                        }

                        }

                    }//end else
                    //  sw.Stop();
                }//end for. all point are been labeled
       

            ArrayList R = new ArrayList();

            int[,] BW2 = bwOrdering(BW);
            R.Add(BW2);

            List<double[]> RL = new List<double[]>(); // relation with the label and their chromaticity (Zen and Azi)

            RL.Add(new double[2]);// is 0 by default
            double[] aux = new double[2];
            int LABELX = 0;//clean            

            int nLabels = max(BW2, rows, col);
            int nf = 0; // clean
            int nc = 0;

            //for each label
            for (int LABEL = 1; LABEL <= nLabels; LABEL++)
            {
                bool salir = false;
                // look for the first occurrence of this label in BW2
                for (int i = 0; i < rows; i++)
                {
                    for (int j = 0; j < col; j++)
                        if (BW2[i, j] == LABEL)
                        {
                            nf = i; nc = j;
                            salir = true;
                            break;
                        }
                    if (salir)
                        break;
                }
                LABELX = BW[nf, nc];
                aux = new double[2];//clear
                aux[0] = LbChrList[LABELX][1];//zenith
                aux[1] = LbChrList[LABELX][2];//azimuth
                RL.Add(aux);

            }


            R.Add(RL);
            return R; // a labeled image with the index chromaticities
            

        }


       /// <summary>
       /// this function return the chromatic distace between two points
       /// </summary>
       /// <param name="a"></param>
       /// <returns></returns>
        private double distanceCH(int a_row, int a_col, int b_row, int b_col)
        {
            double a_zenith = Zenith[a_row, a_col];
            double a_azimuth = Azimuth[a_row, a_col];
            double b_zenith = Zenith[b_row, b_col];
            double b_azimuth = Azimuth[b_row, b_col];

            double d1 = (a_zenith - b_zenith) * (a_zenith - b_zenith);
            double d2 = (a_azimuth - b_azimuth) * (a_azimuth - b_azimuth);

            return Math.Sqrt(d1 + d2) / Math.PI / 2;// normalize to the range [0-1];
        }


        /// <summary>
        /// return the chromatic distence between to lines expresed by their angles
        /// </summary>
        /// <param name="a_zenith"></param>
        /// <param name="a_azimuth"></param>
        /// <param name="b_zenith"></param>
        /// <param name="b_azimuth"></param>
        /// <returns></returns>
        private double distanceCH(double a_zenith, double a_azimuth, double b_zenith, double b_azimuth)
        {


            double d1 = (a_zenith - b_zenith) * (a_zenith - b_zenith);
            double d2 = (a_azimuth - b_azimuth) * (a_azimuth - b_azimuth);

            return Math.Sqrt(d1 + d2)/Math.PI/2;// normalize to the range [0-1]
        }

        /// <summary>
        /// This function return the Intensity distence between two points (a and b)
        /// </summary>
        /// <param name="a_row"></param>
        /// <param name="a_col"></param>
        /// <param name="b_row"></param>
        /// <param name="b_col"></param>
        /// <returns></returns>
        private double distanceI(int a_row, int a_col, int b_row, int b_col)
        {
            double den = Math.Sqrt(255 * 255 * 3); //maximum value

            double l1 = L[a_row, a_col]/den; //normalized
            double l2 = L[b_row, b_col]/den;

            return Math.Abs(l1 - l2);
        }

        /// <summary>
        /// The input values are in the range [0-255]
        /// </summary>
        /// <param name="L1"></param>
        /// <param name="L2"></param>
        /// <returns>return the normalized value to [0-1]</returns>
        private double distanceI(double L1, double L2)
        {
            double den = Math.Sqrt(255 * 255 * 3); //maximum value

            double l1 = L1 / den; //normalized
            double l2 = L2 / den;

            return Math.Abs(l1 - l2);
        }


        /// <summary>
        /// Auxiliar function for the hybrid distance
        /// </summary>
        /// <param name="val"></param>
        /// intput intensity parameter
        /// <param name="a"></param>
        /// minimum a threshold, range [0,1]
        /// <param name="b"></param>
                /// maximum b threshold, range [0,1]
        /// <param name="y"></param>
                /// image value , range [0,1]
        /// <returns></returns>

        private double V(double val, double a, double b, double c){
      double r = 0; //initialize return value
      
      if (val < a){
          r  = 0;
      }else{
          if(val>b){
              r=c;
          }else{
              r = (c/2)+ Math.Cos((((val-a)/(b-a))*Math.PI)+Math.PI)/(2/c);
          }
      }
      return r;


  }

        /// <summary>
        /// Hybrid distence between two points
        /// </summary>
        /// <param name="a_row"></param>
        /// <param name="a_col"></param>
        /// <param name="b_row"></param>
        /// <param name="b_col"></param>
        /// <param name="a"></param>
        /// <param name="b"></param>
        /// <param name="c"></param>
        /// <returns></returns>

        private double distanceH(int a_row, int a_col, int b_row, int b_col, double a, double b, double c)
        {

            double den = Math.Sqrt(255 * 255 * 3); //maximum value

            double l1 = L[a_row, a_col]/den;
            double l2 = L[b_row, b_col]/den;

            double intensity = (l1 +l2)/2;


            double v0 = V(intensity, a, b, c);

            double v1 = 1 - v0;

            return v0 * distanceCH(a_row, a_col, b_row,b_col) + v1 * distanceI(a_row, a_col,  b_row,  b_col);



        }

        /// <summary>
        /// this methos is used in fastSegmentation2. Calculate a hybrid distance
        /// </summary>
        /// <param name="a_zenith"></param>
        /// chromatic zenith of the point of study
        /// <param name="a_azimuth"></param>
        /// chromatic azimuth of the point of study
        /// <param name="a_intensity"></param>
        /// intensity of the point of study
        /// <param name="b_zenith"></param>
        /// chromatic zenith of the label neighboor point 
        /// <param name="b_azimuth"></param>
        /// chromatic azimuth of the label neighboor point 
        /// <param name="b_intensity"></param>
        /// intensity of the neighboor pixel
        /// <returns></returns>
        private double distanceH(double a_zenith, double a_azimuth, double a_intensity, double b_zenith, double b_azimuth, double b_intensity, double a, double b, double c)
        {

            double den = Math.Sqrt(255 * 255 * 3); //maximum value

            double l1 = a_intensity / den;//normalize
            double l2 = b_intensity / den;

            double intensity = (l1 + l2) / 2;


            double v0 = V(intensity, a, b, c);

            double v1 = 1 - v0;

            return v0 * distanceCH(a_zenith,a_azimuth,b_zenith, b_azimuth) + v1 * distanceI(a_intensity,b_intensity);



        }

        /// <summary>
        /// Method added by dominik
        /// </summary>
        /// <param name="imageArray"></param>
        /// <returns></returns>
        public static Image<Gray, byte> ArrayToImage(double[,] imageArray)
        {
            return ArrayToImage(imageArray, false);
        }


        /// <summary>
        /// method added by Dominik
        /// </summary>
        /// <param name="imageArray"></param>
        /// <param name="byteRangeFormat"></param>
        /// <returns></returns>
        public static Image<Gray, byte> ArrayToImage(double[,] imageArray, bool byteRangeFormat)
        {
            Image<Gray, byte> img = new Image<Gray, byte>(imageArray.GetLength(1), imageArray.GetLength(0));

            if (byteRangeFormat)
            {
                for (int i = 0; i < img.Width; i++)
                    for (int j = 0; j < img.Height; j++)
                        img.Data[j, i, 0] = (byte)(imageArray[j, i]);
            }
            else
            {
                for (int i = 0; i < img.Width; i++)
                    for (int j = 0; j < img.Height; j++)
                        img.Data[j, i, 0] = (byte)(255 * imageArray[j, i]);
            }

            return img;
        }

    }//end class
}

