/**
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

http://www.gnu.org/copyleft/lesser.html

*****************************************************************************************
*		UNIVERSIDAD DEL PAIS VASCO - EUSKAL HERRIKO UNIBERTSITATEA 		
*											
*		DEPARTAMENTO DE INGENIERIA DE SISTEMAS Y AUTOMATICA			
*											
*		GRUPO DE INVESTIGACION EN CONTROL INTELIGENTE				
*											
*****************************************************************************************
*											
*	Fuentes:		- Plantilla Comedi para la escritura de drivers			
*				  	  (comedi/drivers/skel.c).
*				    - Driver para CX-1100 (Beckhoff Driver - www.sourceforge.net)
*				       Richard Lemon.
*				    - Hojas de datos CX-1100, KL3404, KL4034.
*											
*	Descripción:	Driver hecho en Comedi para los equipos de Bekchoff: CPU CX1020, Fuente	
*				de alimentación CX1100-0002, con terminales:
*					- KL1408 (Entradas digitales, 24V, 8 ch.).
*					- KL2408 (Salidas digitales, 24V, 8 ch.).
*					- KL3404 (Entradas Analógicas, +/-10V, 4 ch.)
*					- KL4034 (Salidas Analógicas, +/-10V, 4 ch.)							
*												
*	Alcance:		Este driver en primera fase, permite el acceso a la Lectura/Escritura de las E/S
*				CX1020 (KL1408, KL2408, KL3404, y KL4034) en tiempo real para aplicaciones
*				desarrolladas en RTAI (RtaiLab).  También se puede usar en aplicaciones programadas
*				en C.
*											
***************************************************************************************
*											
*	Autor:		J. Andrés Cubillos Galvis	(2012)				
*											
***************************************************************************************
*/
											
#include <linux/reboot.h>
#include <linux/comedidev.h>
#include <linux/delay.h>
#include <rtai_sched.h>
#include <rtai_sem.h>
#include <rtai_shm.h>

/** Zona declaración variables globales y estructuras
*/

//	CX-1100 DPRAM Memory addresses
static unsigned long io_physical = 0xD0000;
static unsigned long io_length = 0x4000;

static unsigned long AUX_OFFSET = 0x1000;
static unsigned long GCB_OFFSET = 0xFF0;
static unsigned long KBCB_OFFSET = 0xFD0;
static unsigned long KBOP_OFFSET = 0x200;
static unsigned long KBIP_OFFSET = 0x000;

//	General Control Block y K-Bus Control Block (Tamaño en bytes)
static unsigned long GCB_SIZE = 0x10;	// 16 dec.
static unsigned long KBCB_SIZE = 0x20;	// 32 dec.

/**	Nota:	Los valores of KBIP_BYTES_COPY, y KBOP_BYTES_COPY se estiman de la siguiente manera:
*		
*		KPIP_BYTES_COPY = Bit width in the process image (KL1408, KL3404, KL4034 ) / 8 [bytes]
*
*		KPOP_BYTES_COPY = Bit width in the process image (KL2408, KL3404, KL4034 ) / 8 [bytes]
*
*		KL1408 = 8 bits
*		KL2408 = 8 bits
*		KL3404 = (4 x 16 ) + (4 x 8) = 96 bits
*		KL4034 = (4 x 16 ) + (4 x 8) = 96 bits
*
*		KPIP_BYTES_COPY = (8 + 96 + 96 ) / 8 = 25 [bytes] (0x19)
*		KPOP_BYTES_COPY = (8 + 96 + 96 ) / 8 = 25 [bytes] (0x19)
*
*		Para una configuración desconocida, asignar la totalidad de bytes de cada bloque (512 bytes cada uno)
*/
static unsigned long KBIP_BYTES_COPY = 0x19;
static unsigned long KBOP_BYTES_COPY = 0x19;

//	Variables uso general Driver

unsigned long io_poll = 0;
unsigned long ProcessDataError = 0;

unsigned long kbus_cycle_iobase;
unsigned char* io_new;
unsigned char* io_old;

/**	Variables vinculadas a la tarea de tiempo real
*	correspondiente al ciclo del K-Bus
*/

static RT_TASK kbus_cycle_task;
static RTIME kbus_cycle_period_ns = 100000;	

static RTIME kbus_cycle_period_count;
static RTIME timer_period_count;

static int kbus_task_cycle_on;
static SEM *sem_rd, *sem_wr;

// Variables de seguimiento programa

//static int ciclosr[4] = {0,0,0,0};
//static int ciclosw[4] = {0,0,0,0};
//static int ciclosk = 0;


/** Descripción de la configuración equipos Beckhoff
*	KL3404, KL4034, KL1408 y KL2408
*/
typedef struct {
	char *name;
	int di_chans;		// KL1408
	int do_chans;		// KL2408
	int ai_chans;		// KL3404
	int ai_bits;		// KL3404
	int ao_chans;		// KL4034
	int ao_bits;		// KL4034

} kbus_board;

/** Especificación de valores para módulos Beckhoff
*	KL3404, y KL4034
*/
static const kbus_board kbus = {
	"KBUS_Beckhoff",
	8,			// KL1408
	8,			// KL2408
	4,			// KL3404
	16,			// KL3404
	4,			// KL4034
	16,			// KL4034

};

/**	Zona declaración funciones empleadas por el driver
*/

static int kbus_attach(comedi_device *dev);
static int kbus_detach(comedi_device *dev);

static comedi_driver driver_kbus = {
	driver_name:"kbusdrv",
	module: THIS_MODULE,
	attach: kbus_attach,
	detach: kbus_detach
};

static void *align_remap(unsigned long phys_addr);
static void align_unmap(void *virt_add);

static int kbus_ai_rinsn(comedi_device * dev, comedi_subdevice * s, comedi_insn * insn, lsampl_t * data);
static int kbus_ao_winsn(comedi_device * dev, comedi_subdevice * s, comedi_insn * insn, lsampl_t * data);
static int kbus_di_rinsn(comedi_device * dev, comedi_subdevice * s, comedi_insn * insn, lsampl_t * data);
static int kbus_do_winsn(comedi_device * dev, comedi_subdevice * s, comedi_insn * insn, lsampl_t * data);

/** Zona especificación cuerpos de las funciones empleadas por el driver
*/

//	La función kbus_cycle_func es la encargada de realizar el el ciclo del Kbus

static void kbus_cycle_func(int t, comedi_device * dev)
{
	int retval;

	while(kbus_task_cycle_on)
	{
		//printk(KERN_INFO "K(%d)\n", ciclosk);
		//ciclosk++;

		//	Processdata Ready es igual a Processdata Request ?
		if ( readb(kbus_cycle_iobase + GCB_OFFSET + 0x0E) == readb(kbus_cycle_iobase + GCB_OFFSET + 0x0D))	
		{	//	Ha finalizado el ciclo del K-Bus, y se tiene acceso a la R/W de la DPRAM

			// printk(KERN_INFO "K(%d)\n", ciclosk);
			// ciclosk++;

			//	Activar wait de los semáforos para R/W
			rt_sem_wait(&sem_rd);			
			rt_sem_wait(&sem_wr);

			//	0. Preguntar por errores en Processdata
			ProcessDataError = readb(kbus_cycle_iobase + GCB_OFFSET + 0x09);
			if ( test_bit(0,&ProcessDataError) )
			{
				printk(KERN_INFO "KBdrv|KBcycle -> Kbus Processdata error! (Check sample times) %lx\n", ProcessDataError);

				//	Quitar el comentario a la siguiente línea si desea habilitar: "Emergency_restart"
				// goto forced_end;
			}
					
			//	1. Copiar el General Control Block desde la DPRAM a "io_old"
			memcpy_fromio((io_old + GCB_OFFSET), (kbus_cycle_iobase + GCB_OFFSET), GCB_SIZE);
					
			//	2. Copiar el KBus Control Block desde la DPRAM a "io_old"
			memcpy_fromio((io_old + KBCB_OFFSET), (kbus_cycle_iobase + KBCB_OFFSET), KBCB_SIZE);

			//	3. Copiar los datos de entrada Kbus desde la DPRAM a "io_old"
			memcpy_fromio((io_old + KBIP_OFFSET), (kbus_cycle_iobase + KBIP_OFFSET), KBIP_BYTES_COPY);
	
			//	4. Copiar los datos de salida Kbus desde la DPRAM a "io_old"
			memcpy_fromio((io_old + KBOP_OFFSET), (kbus_cycle_iobase + KBOP_OFFSET), KBOP_BYTES_COPY);

			//	Se han registrado nuevos datos de salida? (Según R. Lemon Dirty Buffer)
			if ( test_bit(4,&io_poll) )
			{	//	Si es así, copiar los datos desde el buffer "io_new" a la DPRAM
				memcpy_toio( (kbus_cycle_iobase + KBOP_OFFSET), (io_new + KBOP_OFFSET), KBOP_BYTES_COPY);
			}
			else
			{	//	De lo contrario, pasar los datos desde el buffer "io_old" al "io_new"
				memcpy( (io_new + KBOP_OFFSET), (io_old + KBOP_OFFSET), KBOP_BYTES_COPY);
			}

			//	Señalizar que se ha limpiado el buffer
			clear_bit(4,&io_poll);

			//	Liberar los semáforos para R/W

			rt_sem_signal(&sem_rd);			
			rt_sem_signal(&sem_wr);
	
			//	Lanzar el nuevo ciclo
			writeb( ( readb(kbus_cycle_iobase + GCB_OFFSET + 0x0D) + 1 ) , (kbus_cycle_iobase + GCB_OFFSET + 0x0E));

		}	

		rt_task_wait_period();
	
	}

forced_end:

	printk(KERN_INFO "KBdrv|KBcycle -> Launch Restart!\n");
	emergency_restart();	

	return;
}

/*	Funciones para "remapeo" de memoria
*/

static void *align_remap(unsigned long phys_addr)
{
	unsigned long offset, last_addr, size;

	last_addr = phys_addr + io_length - 1;
	offset = phys_addr & ~PAGE_MASK;

	phys_addr &= PAGE_MASK;
	size = PAGE_ALIGN(last_addr) - phys_addr;
	return ioremap(phys_addr, size) + offset;
}

static void align_unmap(void *virt_add)
{
	iounmap((void *)((unsigned long)virt_add & PAGE_MASK));
}

static int kbus_attach(comedi_device * dev)
{
	comedi_subdevice *s;
	
	int retval;
	int ret = 0;
	io_poll= 0;

	dev->board_name = kbus.name;
	//	Se verifica la disponibilidad de memoria del CX-1100
	ret = check_mem_region(io_physical, io_length);
	if (ret)
	{
		printk(KERN_INFO "Kbusdrv :: Cannot lock mem address \n");
		return ret;
	}
	//	Si la región está disponible, se aparta para el driver
	request_mem_region(io_physical, io_length, driver_kbus.driver_name);
	//	Se solicita la dirección base asignada
	dev->iobase = (unsigned long) align_remap(io_physical);
	if (dev->iobase == 0x0000)
		return -1;

	kbus_cycle_iobase = dev->iobase;

	//	Reservar memoria para los "buffers"
	io_new = rtai_kmalloc(nam2num("new"), io_length);
	io_old = rtai_kmalloc(nam2num("old"), io_length);

	if (alloc_subdevices(dev, 4) < 0)	//	El segundo parámetro de alloc_subdevices(), corresponde a la cantidad
						//	de modulos E/S instalados.  En este caso 4.
		return -ENOMEM;

	//	Especificación módulo KL3404 (AI)
	s = dev->subdevices + 0;
	s->type = COMEDI_SUBD_AI;
	s->subdev_flags = SDF_READABLE | SDF_GROUND;
	s->n_chan = kbus.ai_chans;
	s->maxdata = (1 << kbus.ai_bits) - 1;
	s->range_table = &range_bipolar10;
	s->insn_read = kbus_ai_rinsn;

	//	Especificación módulo KL4034 (AO)
	s++;
	s->type = COMEDI_SUBD_AO;
	s->subdev_flags = SDF_WRITABLE | SDF_GROUND;
	s->n_chan = kbus.ao_chans;
	s->maxdata = (1 << kbus.ao_bits) -1;
	s->range_table = &range_bipolar10;
	s->insn_write = kbus_ao_winsn;

	//	Especificación módulo KL1408 (DI)
	s++;
	s->type = COMEDI_SUBD_DI;
	s->subdev_flags = SDF_READABLE;
	s->n_chan = kbus.di_chans;
	s->maxdata = 1;
	s->range_table = &range_digital;
	s->insn_bits = kbus_di_rinsn;

	//	Especificación módulo KL2408 (DO)
	s++;
	s->type = COMEDI_SUBD_DO;
	s->subdev_flags = SDF_WRITABLE;
	s->n_chan = kbus.do_chans;
	s->maxdata = 1;
	s->range_table = &range_digital;
	s->insn_bits = kbus_do_winsn;


	//	Una vez terminada la configuración del equipo se procede con 
	//	la secuencia de arranque del mismo.
	
	/** Secuencia de inicio CX-1100 */
	
	//	Inicio CX-1100 ...

	if(test_bit(1,&io_poll)!=0)
		return -EFAULT;
	
	set_bit(0,&io_poll);	//	Driver en marcha...
	set_bit(1,&io_poll);
		
	//	Reset general CX-1100 ...
	writeb(1, (dev->iobase + AUX_OFFSET + 0x03));
	writeb(0, (dev->iobase + AUX_OFFSET + 0x03));

	//	Reset K-Bus Data 
	writeb(1, (dev->iobase + KBCB_OFFSET + 0x08));
	writeb(0, (dev->iobase + KBCB_OFFSET + 0x08));

	/*	Configuración y activación Tarea "Ciclo K-Bus (kbus_cycle_task)	*/
	kbus_task_cycle_on = 1;

	//	Inicialización semáforos:
	rt_sem_init(&sem_rd, 1);
	rt_sem_init(&sem_wr, 1);

	//	Configuración de timers
	rt_set_periodic_mode();	
	kbus_cycle_period_count = nano2count(kbus_cycle_period_ns);
	
	//	Iniciar el "timer"
	timer_period_count = start_rt_timer(kbus_cycle_period_count);

	//	Crear tarea de tiempo real:
	//	     |- rt task -| |-thread-||-data-| |-stack size-| |-Priority -| |-uses fpu -| |-signal-|
	retval = rt_task_init(&kbus_cycle_task, kbus_cycle_func, 0, 4096, 0, 0, 0);

	//	Verificar creación de la tarea
	if(retval!=0)
	{
		printk(KERN_INFO "Kbusdriver | Attach -> Failure init Kbus-cycle task.\n");
		return retval;
	}
	
	//	Ejecutar Kbus-cycle
	retval = rt_task_make_periodic(&kbus_cycle_task, rt_get_time() + kbus_cycle_period_count, kbus_cycle_period_count);

	//	Verificar lanzamiento de la tarea
	if(retval!=0)
	{
		printk(KERN_INFO "Kbusdriver | Attach -> Can't start Kbus-cycle task.\n");
		return retval;
	}

	printk(KERN_INFO "Kbusdriver | Attach -> Done\n");

	return 0;	

}

static int kbus_detach(comedi_device * dev)
{
	int retval;

	ProcessDataError = 0;		//	Clean ProceessDateError
	clear_bit(0,&io_poll);

	printk(KERN_INFO "Kbusdriver | Detach -> Ready to end...\n");

	//	Eliminar los semáforos creados:
	rt_sem_delete(&sem_rd);
	rt_sem_delete(&sem_wr);

	//	Finalizar Tarea Kbus-Cycle
	kbus_task_cycle_on = 0;
	stop_rt_timer();
	retval=rt_task_delete(&kbus_cycle_task);

	//	Verificar finalización de la tarea
	if(retval!=0)
	{
		printk(KERN_INFO "Kbusdriver | Detach -> Failure end Kbus-cycle task.\n");
		return retval;
	}

	//	Liberar zonas de memoria ocupadas por el driver
	rtai_kfree(nam2num("new"));
	rtai_kfree(nam2num("old"));

	align_unmap((void *) dev->iobase);
	release_mem_region(io_physical, io_length);

	printk(KERN_INFO "Kbusdriver | Detach -> Done\n");
}


/** Función especificada en el driver para la lectura de
*	Entradas Analógicas
*/

static int kbus_ai_rinsn(comedi_device * dev, comedi_subdevice * s, comedi_insn * insn, lsampl_t * data)
{
	int i, chan;
	unsigned long comedi_data, beck_data, CHIN_OFFSET;

	//	Se bloquea la tarea hasta que finalize el ciclo del K-Bus	
	rt_sem_wait(&sem_rd);			

	//	Seleccionar Canal
	chan = CR_CHAN(insn->chanspec);
		
	//printk(KERN_INFO "KBdrv|rd -> Ch: %d ; C_R: %d ; kbc: %d\n", chan, ciclosr[chan], ciclosk);
	
	switch (chan)
	{
		case 0:
			CHIN_OFFSET = 0x01;
			break;
		case 1:
			CHIN_OFFSET = 0x04;
			break;
		case 2:
			CHIN_OFFSET = 0x07;
			break;
		case 3:
			CHIN_OFFSET = 0x0A;
			break;
		default:
			return EINVAL;
			break;
	}

	//	Capturar el valor de entrada Beckhoff
	beck_data = readw(io_old + KBIP_OFFSET + CHIN_OFFSET);
	
	//	Pasar a formato Comedi
	if(beck_data <= 0x7FFF)
	{
		comedi_data = beck_data + 0x8000;
	}
	else
	{
		if (beck_data>8000)
		{
			comedi_data = beck_data - 0x8001;
		}
		else	
		{	// Significa que el valor de entrada (negativo) está saturado (beck_data = 0x8000)
			comedi_data = 0x0000;
		}
			
		
	}
	
	//	printk(KERN_INFO "KBdrv|rd -> Ch: %d, Come_D: %lx ; Beck_D: %lx \n", chan, comedi_data, beck_data);

	//	Enviar a Comedi
	data[0] = comedi_data;

	//	Habilita el acceso al ciclo del K-Bus
	rt_sem_signal(&sem_rd);			

	//	ciclosr[chan]=	ciclosr[chan] + 1;
	
	return i;

}


/** Función especificada en el driver para la escritura de
*	Salidas Analógicas
*/

static int kbus_ao_winsn(comedi_device * dev, comedi_subdevice * s, comedi_insn * insn, lsampl_t * data)
{
	int i, chan;
	unsigned long comedi_data, beck_data, CHOUT_OFFSET;

	//	Se bloquea la tarea hasta que finalize el ciclo del K-Bus	
	rt_sem_wait(&sem_wr);			
	
	//	Notificar la existencia de nuevos datos ( R.Lemon "Dirty Buffer")
	set_bit(4,&io_poll);

	//	Seleccionar Canal
	chan = CR_CHAN(insn->chanspec);

	//	printk(KERN_INFO "KBdrv|wr -> Ch: %d ; C_W: %d ; kbc: %d\n", chan, ciclosw[chan], ciclosk);

	switch (chan)
	{
		case 0:
			CHOUT_OFFSET = 0x0D;
			break;
		case 1:
			CHOUT_OFFSET = 0x10;
			break;
		case 2:
			CHOUT_OFFSET = 0x13;
			break;
		case 3:
			CHOUT_OFFSET = 0x16;
			break;
		default:
			return EINVAL;
			break;
	}

	//	Capturar valor desde Comedi(data)
	comedi_data = data[0];

	//	Pasar a formato Beckhoff
	if (comedi_data >= 0x8000)
	{
		beck_data = comedi_data - 0x8000;
	}
	else
	{
		beck_data = comedi_data + 0x8001;
	}

	//	printk(KERN_INFO "KBdrv|wr -> Ch: %d, Come_D: %lx ; Beck_D: %lx \n", chan, comedi_data, beck_data);

	//	Enviar al buffer "io_new"
	writew(beck_data, io_new + KBOP_OFFSET + CHOUT_OFFSET);

	//	Habilita el acceso al ciclo del K-Bus
	rt_sem_signal(&sem_wr);			

	//	ciclosw[chan]=ciclosw[chan]+1;

	return i;
}



/** Función especificada en el driver para la escritura de
*	Entradas Digitales
*/

static int kbus_di_rinsn(comedi_device * dev, comedi_subdevice * s, comedi_insn * insn, lsampl_t * data)
{
	unsigned long DI_BYTE_ADDRESS = 0x18;

	//	Se bloquea la tarea hasta que finalize el ciclo del K-Bus	
	rt_sem_wait(&sem_rd);			

	if (insn->n != 2)
	{
		return -EINVAL;
	}

	data[1] = readb(io_old + KBIP_OFFSET + DI_BYTE_ADDRESS);

	//	Habilita el acceso al ciclo del K-Bus
	rt_sem_signal(&sem_rd);			

	//	printk(KERN_INFO "KBdrv|rdi -> dato leído | data[1]: %d \n", data[1]);

	return 2 ;
}


/** Función especificada en el driver para la escritura de
*	Salidas Digitales
*/

static int kbus_do_winsn(comedi_device * dev, comedi_subdevice * s, comedi_insn * insn, lsampl_t * data)
{
	unsigned long DO_BYTE_ADDRESS = 0x18;

	//	Se bloquea la tarea hasta que finalize el ciclo del K-Bus	
	rt_sem_wait(&sem_wr);			

	//	Notificar la existencia de nuevos datos ( R.Lemon "Dirty Buffer")
	set_bit(4,&io_poll);

	if (insn->n != 2)
		return -EINVAL;

	if (data[0])
	{
		s->state &= ~data[0];
		s->state |= data[0] & data[1];
		writeb(s->state, io_new + KBOP_OFFSET + DO_BYTE_ADDRESS);
		// printk(KERN_INFO "KBdrv|wdo -> escrito: s-> state %d \n", s->state);
	}

	data[1] = s->state;

	//	Habilita el acceso al ciclo del K-Bus
	rt_sem_signal(&sem_wr);			

	return 2 ;
}


COMEDI_INITCLEANUP(driver_kbus);
