/***************************************************/
/***Archivo de codigo para la libreria de algebra***/
/***********de uso general alglin*******************/
/***************************************************/
/********Autor: Igor Fernandez de Bustos************/
/***************************************************/
/***Este archivo se distribuye bajo licencia GPL****/
/***************************************************/
/******************version 0.01*********************/
/****************NOTAS IMPORTANTES******************/
/***************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "alglin.h"

/**Escribe la matriz a texto
 @param archivo archivo donde escribirla
 @param A puntero a la matriz
 @param fA numero de elementos de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de la matriz
 @param cs número de cifras significativas*/
void matrizatexto(FILE *archivo, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	for (i=0;i<fA;i++)
	{
		for (j=0;j<cA;j++)
		{
			fprintf(archivo,format,A[i*ldA+j]);
			fprintf(archivo,"\t");
		}
		fprintf(archivo,"\b\n");
	}
}

/**Escribe la matriz (formato OCTAVE) a texto
 @param archivo archivo donde escribirla
 @param A puntero a la matriz
 @param fA numero de elementos de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de la matriz
 @param cs número de cifras significativas*/
void matrizatextooctave(FILE *archivo, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	fprintf(archivo,"[");
	for (i=0;i<fA;i++)
	{
		for (j=0;j<cA;j++)
		{
			fprintf(archivo,format,A[i*ldA+j]);
			fprintf(archivo,", ");
		}
		fprintf(archivo,"\b\b; ");
	}
	fprintf(archivo,"\b\b]\n");
}


/**Devuelve el vector u=aAv. No hay reserva de espacios ni nada por el estilo
 @param u puntero al vector resultado
 @param a escalar
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA separación entre el primer elemento de cada fila. Para poder aplicarlo a submatrices
 @param v puntero al vector*/
void vectoraAv(double *u, double a, double*A, unsigned long fA, unsigned long cA, unsigned long ldA, double *v)
{
	unsigned long i,j;
	for (i=0;i<fA;i++)
	{
		u[i]=0.;
		for (j=0;j<cA;j++)
			u[i]+=a*A[i*ldA+j]*v[j];
	}
}

/**Devuelve el vector u=u+aAv. No hay reserva de espacios ni nada por el estilo
 @param u puntero al vector resultado
 @param A puntero a la matriz
 @param a escalar
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA separación entre el primer elemento de cada fila. Para poder aplicarlo a submatrices
 @param v puntero al vector*/
void vectorsumaraAv(double *u, double a, double*A, unsigned long fA, unsigned long cA, unsigned long ldA, double *v)
{
	unsigned long i,j;
	for (i=0;i<fA;i++)
		for (j=0;j<cA;j++)
			u[i]+=a*A[i*ldA+j]*v[j];
}

/**Devuelve el vector u=aAtv. No hay reserva de espacios ni nada por el estilo
 @param u puntero al vector resultado
 @param a escalar
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA separación entre el primer elemento de cada fila. Para poder aplicarlo a submatrices
 @param v puntero al vector*/
void vectoraATv(double *u, double a, double*A, unsigned long fA, unsigned long cA, unsigned long ldA, double *v)
{
    unsigned long i,j;
    /*Obviamente, trabajamos por filas que es más rápido*/
    if (fA==0)
        return;
    for (j=0;j<cA;j++)
    {
        u[j]=a*A[j]*v[0];
    }
    for (i=1;i<fA;i++)
        for (j=0;j<cA;j++)
        {
            u[j]+=a*A[ldA*i+j]*v[i];
        }
}

/**Devuelve el vector u=aAtv. No hay reserva de espacios ni nada por el estilo
 @param u puntero al vector resultado
 @param a escalar
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA separación entre el primer elemento de cada fila. Para poder aplicarlo a submatrices
 @param v puntero al vector*/
void vectorsumaraATv(double *u, double a, double*A, unsigned long fA, unsigned long cA, unsigned long ldA, double *v)
{
    unsigned long i,j;
    /*Obviamente, trabajamos por filas que es más rápido*/
    if (fA==0)
        return;
    for (i=0;i<fA;i++)
        for (j=0;j<cA;j++)
        {
            u[j]+=a*A[ldA*i+j]*v[i];
        }
}

/**Establece en la matriz C la A multiplicada por un escalar
 @param C puntero a la matriz donde se quiere sumar
 @param ldC leading dimension de C
 @param a escalar
 @param A puntero a la matriz que se quiere sumar
 @param fA numero de filas de A
 @param cA numero de columnas de A
 @param ldA separacion entre las filas de A*/
void matrizaA(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long cA, unsigned long ldA)
{
	unsigned long i,j;
	for (i=0;i<fA;i++)
	{
		for (j=0;j<cA;j++)
		{
			C[i*ldC+j]=a*A[i*ldA+j];
		}
	}
}

/**Suma a una matriz C la A multiplicada por un escalar
 @param C puntero a la matriz donde se quiere sumar
 @param ldC leading dimension de C
 @param a escalar
 @param A puntero a la matriz que se quiere sumar
 @param fA numero de filas de A
 @param cA numero de columnas de A
 @param ldA separacion entre las filas de A*/
void matrizsumaraA(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long cA, unsigned long ldA)
{
	unsigned long i,j;
	for (i=0;i<fA;i++)
	{
		for (j=0;j<cA;j++)
		{
			C[i*ldC+j]+=a*A[i*ldA+j];
		}
	}
}

/**Establece en la matriz C la A traspuesta y multiplicada por un escalar
 @param C puntero a la matriz donde se quiere sumar
 @param ldC leading dimension de C
 @param a escalar
 @param A puntero a la matriz que se quiere sumar
 @param fA numero de filas de A
 @param cA numero de columnas de A
 @param ldA separacion entre las filas de A*/
void matrizaAT(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long cA, unsigned long ldA)
{
	unsigned long i,j;
	for (i=0;i<fA;i++)
	{
		for (j=0;j<cA;j++)
		{
			C[i*ldC+j]=a*A[j*ldA+i];
		}
	}
}

/**Suma a la matriz C la A traspuesta y multiplicada por un escalar
 @param C puntero a la matriz donde se quiere sumar
 @param ldC leading dimension de C
 @param a escalar
 @param A puntero a la matriz que se quiere sumar
 @param fA numero de filas de A
 @param cA numero de columnas de A
 @param ldA separacion entre las filas de A*/
void matrizsumaraAT(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long cA, unsigned long ldA)
{
	unsigned long i,j;
	for (i=0;i<fA;i++)
	{
		for (j=0;j<cA;j++)
		{
			C[i*ldC+j]+=a*A[j*ldA+i];
		}
	}
}


/**Multiplica la matriz A por la B y la devuelve en C, C no puede ser ni A ni B
 @param C puntero a la matriz donde se quiere almacenar AB
 @param ldC numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de c
 @param a factor de escala
 @param A puntero a la matriz A
 @param fA numero de filas de a
 @param cA numero de columnas de a
 @param ldA numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de a
 @param B puntero a la matriz B
 @param cB numero de columnas de b
 @param ldB numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de b*/
void matrizaAB(double *C, unsigned long ldC, double a, double *A,unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i;
	unsigned long j;
	unsigned long k;
	for (i=0;i<fA;i++)
		for (j=0;j<cB;j++)
		{
			C[i*ldC+j]=0.;
			for (k=0;k<cA;k++)
				C[i*ldC+j]+=a*A[i*ldA+k]*B[k*ldB+j];
		}
}

/**Multiplica la matriz A por la B y la suma a C, C no puede ser ni A ni B
 @param C puntero a la matriz donde se quiere sumar aAB
 @param ldC numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de c
 @param a factor de escala
 @param A puntero a la matriz A
 @param fA numero de filas de a
 @param cA numero de columnas de a
 @param ldA numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de a
 @param B puntero a la matriz B
 @param cB numero de columnas de b
 @param ldB numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de b*/
void matrizsumaraAB(double *C, unsigned long ldC, double a, double *A,unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i;
	unsigned long j;
	unsigned long k;
	for (i=0;i<fA;i++)
	    for (k=0;k<cA;k++)
		for (j=0;j<cB;j++)
		    C[i*ldC+j]+=a*A[i*ldA+k]*B[k*ldB+j];
}

/**Calcula la matriz A por la traspuesta de B multiplicada por a, C no puede ser ni A ni B
 @param C puntero a la matriz donde se quiere almacenar aABT
 @param ldc numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de c
 @param a escalar por el que se multiplica ABT antes de sumar a C
 @param A puntero a la matriz A
 @param fa numero de filas de a
 @param ca numero de columnas de a
 @param lda numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de a
 @param B puntero a la matriz B
 @param fb numero de filas de b
 @param ldb numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de b*/
void matrizaABT(double *C, unsigned long ldC, double a, double *A,unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long fB, unsigned long ldB)
{
    unsigned long i;
    unsigned long j;
    unsigned long k;
    for (i=0;i<fA;i++)
        for (j=0;j<fB;j++)
		{
			C[i*ldC+j]=a*A[i*ldA]*B[j*ldB];
			for (k=1;k<cA;k++)
				C[i*ldC+j]+=a*A[i*ldA+k]*B[j*ldB+k];
		}
}

/**Multiplica la matriz A por la traspuesta de B y la suma a C multiplicada por a, C no puede ser ni A ni B
 @param C puntero a la matriz donde se quiere almacenar AB
 @param ldC numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de c
 @param a escalar por el que se multiplica ABT antes de sumar a C
 @param A puntero a la matriz A
 @param fA numero de filas de a
 @param cA numero de columnas de a
 @param ldA numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de a
 @param B puntero a la matriz B
 @param fB numero de filas de b
 @param ldB numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de b*/
void matrizsumaraABT(double *C, unsigned long ldC, double a, double *A,unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long fB, unsigned long ldB)
{
    unsigned long i;
    unsigned long j;
    unsigned long k;
    for (i=0;i<fA;i++)
        for (j=0;j<fB;j++)
			for (k=0;k<cA;k++)
				C[i*ldC+j]+=a*A[i*ldA+k]*B[j*ldB+k];
}

/**Calcula la matriz A traspuesta por B multiplicada por a, C no puede ser ni A ni B
 @param C puntero a la matriz donde se quiere almacenar aABT
 @param ldC numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de C
 @param a escalar por el que se multiplica ABT antes de sumar a C
 @param A puntero a la matriz A
 @param fA numero de filas de a
 @param cA numero de columnas de a
 @param ldA numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de a
 @param B puntero a la matriz B
 @param cB numero de filas de b
 @param ldB numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de b*/
void matrizaATB(double *C, unsigned long ldC, double a, double *A,unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i;
	unsigned long j;
	unsigned long k;
	/*printf("A:\n");
	matrizatexto(stdout,A,fA,cA,ldA,8);
	printf("B:\n");
	matrizatexto(stdout,B,fA,cB,ldB,8);*/
	
	for (i=0;i<cA;i++)
	{
		for (j=0;j<cB;j++)
			{
				C[i*ldC+j]=0.;
			}
	}
	for (i=0;i<fA;i++)
	{	
		for (j=0;j<cA;j++)
			for (k=0;k<cB;k++)
				C[j*ldC+k]+=a*A[i*ldA+j]*B[i*ldB+k];
	}
	
	/*printf("A^TB:\n");
	matrizatexto(stdout,C,cA,cB,ldC,8);*/
}

/**Calcula la matriz A traspuesta por B multiplicada por a, C no puede ser ni A ni B
 @param C puntero a la matriz donde se quiere almacenar aABT
 @param ldC numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de C
 @param a escalar por el que se multiplica ABT antes de sumar a C
 @param A puntero a la matriz A
 @param fA numero de filas de a
 @param cA numero de columnas de a
 @param ldA numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de a
 @param B puntero a la matriz B
 @param cB numero de filas de b
 @param ldB numero de elementos desde el primer elemento de una fila al primer elemento de la siguiente de b*/
void matrizsumaraATB(double *C, unsigned long ldC, double a, double *A,unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i;
	unsigned long j;
	unsigned long k;
	for (i=0;i<cA;i++)
		for (j=0;j<fA;j++)
			for (k=0;k<cB;k++)
				C[i*ldC+k]+=a*A[j*ldA+i]*B[j*ldB+k];
}

/*Devuelve el producto vectorial de v por A multiplicado por a
 @param B puntero a la matriz donde se quiere almacenar avxA
 @param ldB leading dimension de B
 @param a factor de escala
 @param v vector. Obviamente de 3 elementos
 @param A puntero a la matriz
 @param ldA leading dimension de A
 @param cA numero de columnas de A (obviamente 3 filas)*/
void matrizavxA(double *B, unsigned long ldB, double a, double *v, double *A, unsigned long ldA,double cA)
{
	unsigned long k;
	for (k=0;k<cA;k++)
	{
		B[k]=v[1]*A[2*ldA+k]-v[2]*A[ldA+k];
	}
	for (k=0;k<cA;k++)
	{
		B[ldB+k]=v[2]*A[k]-v[0]*A[2*ldA+k];
	}
	for (k=0;k<cA;k++)
	{
		B[2*ldB+k]=v[0]*A[ldA+k]-v[1]*A[k];
	}
}

/*Suma a la matriz B el producto vectorial de v por A multiplicado por a
 @param B puntero a la matriz donde se quiere sumar avxA
 @param ldB leading dimension de B
 @param a factor de escala
 @param v vector. Obviamente de 3 elementos
 @param A puntero a la matriz
 @param ldA leading dimension de A
 @param cA numero de columnas de A (obviamente 3 filas)*/
void matrizsumaravxA(double *B, unsigned long ldB, double a, double *v, double *A, unsigned long ldA,double cA)
{
	unsigned long k;
	for (k=0;k<cA;k++)
	{
		B[k]+=v[1]*A[2*ldA+k]-v[2]*A[ldA+k];
	}
	for (k=0;k<cA;k++)
	{
		B[ldB+k]+=v[2]*A[k]-v[0]*A[2*ldA+k];
	}
	for (k=0;k<cA;k++)
	{
		B[2*ldB+k]+=v[0]*A[ldA+k]-v[1]*A[k];
	}
}

/*Establece en la matriz A el producto vectorial de v por I y multiplicado por a, es equivalente a generar la matriz antisimétrica equivalente al producto vectorial y sumarla a la matriz regular multiplicada por un escalar
 @param A puntero a la matriz donde se quiere sumar avxI
 @param ldA leading dimension de A
 @param a factor de escala
 @param v vector, obviamente de 3 elementos
 */
void matrizavxI(double *A, unsigned long ldA, double a, double *v)
{
	A[0]=0.;
	A[1]=-a*v[2];
	A[2]=a*v[1];
	A[ldA]=a*v[2];
	A[ldA+1]=0.;
	A[ldA+2]=-a*v[0];
	A[2*ldA]=-a*v[1];
	A[2*ldA+1]=a*v[0];
	A[2*ldA+2]=0.;
}

/*Suma a la matriz A el producto vectorial de v por I y multiplicado por a, es equivalente a generar la matriz antisimétrica equivalente al producto vectorial y sumarla a la matriz regular multiplicada por un escalar
 @param A puntero a la matriz donde se quiere sumar avxI
 @param ldA leading dimension de A
 @param a factor de escala
 @param v vector, obviamente de 3 elementos
 */
void matrizsumaravxI(double *A, unsigned long ldA, double a, double *v)
{
	A[1]-=a*v[2];
	A[2]+=a*v[1];
	A[ldA]+=a*v[2];
	A[ldA+2]-=a*v[0];
	A[2*ldA]-=a*v[1];
	A[2*ldA+1]+=a*v[0];
}

/*Suma a la matriz A el producto vectorial de v por u multiplicado vectorialmente por u I y multiplicado por a, es equivalente a generar la matriz equivalente a un producto vectorial doble
 @param A puntero a la matriz donde se quiere sumar avxuxI
 @param ldA leading dimension de A
 @param a factor de escala
 @param v vector, obviamente de 3 elementos
 @param u vector, obviamente de 3 elementos
 */
void matrizsumaravxuxI(double *A, unsigned long ldA, double a, double *v, double *u)
{
	A[0]-=a*(v[2]*u[2]+v[1]*u[1]);
	A[1]+=a*v[0]*u[1];
	A[2]+=a*v[2]*u[0];
	A[ldA]+=a*v[0]*u[1];
	A[ldA+1]-=a*(v[2]*u[2]+v[0]*u[0]);
	A[ldA+2]+=a*(v[2]*u[1]);
	A[2*ldA]+=a*(v[0]*u[2]);
	A[2*ldA+1]+=a*(v[1]*u[2]);
	A[2*ldA+2]-=a*(v[1]*u[1]+v[0]*u[0]);
}