/***************************************************************************
 *            permsuce.h
 *
 *  Tue June 16 12:02:04 2015
 *  Copyright  2015  Igor Fernández de Bustos
 *  <impfedei@ehu.es>
 ****************************************************************************/
/*
 * matricespermsuce.h
 *
 * Copyright (C) 2015 - Igor Fernández de Bustos
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/** matrizpermsuce incluye la definición de funciones para trabajar con matrices
 * de permutaciones sucesivas. Son matrices que definen una permutación que
 * afecta a la fila 0 con una posterior, despues a la 1 con una posterior, y así
 * sucesivamente. Se codifican como un array de enteros que indican con que fila
 * se permuta la actual. Se almacena de forma incremental, es decir, almacenamos
 * cuanto hay que sumar a la fila actual para encontrar aquella con la que
 * queremos permutarla, de manera que si algún registro vale cero, no hay
 * permutación. Vale para filas y columnas.
 * El sistema de definición de funciones es continuación de lo escrito en
 * vector.h y en matriz.h, con las siguientes adiciones
		*Ps denota matriz de permutacion
		*PsT denota matriz de permutación traspuesta (y por lo tanto inversa)
 */

#ifndef MATRICESPERMSUCE_H
#define MATRICESPERMSUCE_H

/**Premultiplica el vector v por la matriz de permutaciones sucesivas Ps
@param Ps puntero a la matriz de permutaciones sucesivas
@param ePs numero de cambios en la matriz de permutación (obviamente es cuadrada)
@param v vector a multiplicar*/
void vectorPsv(unsigned long int *Ps, unsigned long ePs, double *v);

/**Premultiplica el vector v por la matriz de permutaciones sucesivas Ps traspuesta
@param Ps puntero a la matriz de permutaciones sucesivas
@param ePs numero de elementos de la matriz de permutación (obviamente es cuadrada)
@param v vector a multiplicar*/
void vectorPsTv(unsigned long int *Ps, unsigned long ePs, double *v);

/**Premultiplica la matriz A por la matriz de permutaciones sucesivas
@param Ps puntero a la matriz de permutaciones sucesivas
@param ePs tamaño de la matriz de permutaciones
@param A puntero a la matriz a multiplicar
@param cA numero de columnas a girar
@param ldA leading dimension de la matriz A*/
void matrizPsA(unsigned long int *Ps, unsigned long int ePs, double *A, unsigned long int cA, unsigned long ldA);

/**Premultiplica la matriz A por la matriz de permutaciones sucesivas traspuesta
@param Ps puntero a la matriz de permutaciones sucesivas
@param ePs tamaño de la matriz de permutaciones
@param A puntero a la matriz a multiplicar
@param cA numero de columnas a girar
@param ldA leading dimension de la matriz A*/
void matrizPsTA(unsigned long int *Ps, unsigned long int ePs, double *A, unsigned long int cA, unsigned long ldA);

/**Postmultiplica la matriz A por la matriz de permutaciones sucesivas Ps
@param A puntero a la matriz a multiplicar
@param fA numero de filas de A
@param ldA leading dimension de la matriz A
@param Ps puntero a la matriz de permutaciones sucesivas
@param ePs número de cambios en Ps
*/
void matrizAPs(double *A, unsigned long int fA, unsigned long ldA, unsigned long int *Ps, unsigned long ePs);

/**Postmultiplica la matriz A por la matriz de permutaciones sucesivas Ps traspuesta
@param A puntero a la matriz a multiplicar
@param fA numero de filas de A
@param ldA leading dimension de la matriz A
@param Ps puntero a la matriz de permutaciones sucesivas
@param ePs número de cambios en Ps
*/
void matrizAPsT(double *A, unsigned long int fA, unsigned long ldA, unsigned long int *Ps, unsigned long ePs);

/**Pasa de un vector de permutaciones sucesivas a un vector de orden.
 Para ello aplica las permutaciones sucesivas sobre un vector que va desde 0 hasta N
 @param Ps puntero al vector de permutaciones sucesivas
 @param t dimensión de lo permutado
 @param orden puntero a un espacio de memoria que debe contener el orden de los elementos
 * permutados. Debe tener el espacio reservado para ePs elementos y lo que tenía se pierde
 @param ePs numero de cambios en Ps*/
void permsuceaorden(unsigned long *Ps, unsigned long t, unsigned long *orden, unsigned long ePs);

/*escribe a texto la matriz de permutaciones sucesivas
 @param archivo: archivo en el que se quiere escribir la matriz
 @param Ps: puntero al vector de permutaciones que describe la matriz de permutaciones sucesivas
 @param t: tamaño de la matriz de permutaciones
 @param ePs: numero de permutaciones realizadas*/
void permsuceatexto(FILE *archivo, unsigned long *Ps, unsigned long t, unsigned long ePs);

/*Escribe a texto (formato octave) la matriz de permutaciones sucesivas
 @param archivo: archivo en el que se quiere escribir la matriz
 @param Ps: puntero al vector de permutaciones que describe la matriz de permutaciones sucesivas
 @param t: tamaño de la matriz de permutaciones
 @param ePs: numero de permutaciones realizadas*/
void permsuceatextooctave(FILE *archivo, unsigned long *Ps, unsigned long t, unsigned long ePs);

#endif /*MATRICESPERMSUCE_H*/
