/*
 * pivotes.c
 * funciones para la búsqueda de pivotes
 * Copyright (C) 2017 - Igor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <math.h>
#include "alglin.h"

/**Busca el pivote más grande a lo bestia
 @param A puntero a la matriz de datos
 @param fA número de filas de la (sub)matriz
 @param cA número de columnas de la (sub)matriz
 @param ldA leading dimension. en este caso número de columnas de la matriz original
 @param f parámetro de vuelta. fila donde está el máximo
 @param c parámetro de vuelta. columna donde está el máximo
 @return el valor absoluto del pivote
*/
double pivotajecompleto(double *A,unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *f, unsigned long *c)
{
	double max;
	unsigned long i,j;
	max=0.;
	*f=0;
	*c=0;
	for (i=0;i<fA;i++)
	{
		for (j=0;j<cA;j++)
		{
			if (fabs(A[i*ldA+j])>max)
			{
				max=fabs(A[i*ldA+j]);
				*f=i;
				*c=j;
			}
		}
	}
	return max;
}

/**Busca el pivote más grande mediante la técnica optimizada, minimizando el número de ifs
 @param A puntero a la matriz de datos
 @param fA número de filas de la (sub)matriz
 @param cA número de columnas de la (sub)matriz
 @param ldA leading dimension. en este caso número de columnas de la matriz original
 @param f parámetro de vuelta. fila donde está el máximo
 @param c parámetro de vuelta. columna donde está el máximo
 @return el valor absoluto del pivote 
*/
double pivotajecompletorapido(double *A,unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *f, unsigned long *c)
{
	double max,cmax;
	size_t i,j;
	max=0.;
	cmax=0.;

	for (i=0;i<fA;i++)
	{
		cmax=0.;
		for (j=0;j<cA;j++)
		{
			cmax=cmax>fabs(A[i*ldA+j])?cmax:fabs(A[i*ldA+j]);
		}
		if (cmax>max)
		{
			max=cmax;
			*f=i;
		}
	}
	
	i=*f;
	for(j=0;j<cA;j++)
		if (fabs(A[i*ldA+j])==max)
		{
			*c=j;
			break;
		}
	return max;
}


/**Busca el primer pivote dominante en fila y columna
 @param A puntero a la matriz de datos
 @param fA número de filas de la (sub)matriz
 @param cA número de columnas de la (sub)matriz
 @param ldA leading dimension. en este caso número de columnas de la matriz original
 @param f parámetro de vuelta. Fila donde está el máximo
 @param c parámetro de vuelta. Columna donde está el máximo
 @param tol valor del cero numerico
 @return el valor absoluto del pivote
*/
double pivotajerook(double *A,unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *f, unsigned long *c, double tol)
{
    double max;
    unsigned long i,j,cmax,fmax,salir;
    /*primer paso, buscamos una fila que tenga un candidato a pivote*/
    max=0.;
    fmax=0;
    cmax=0;
    for (j=0;j<fA;j++)
    {
	for (i=0;i<cA;i++)
	{
	    if (fabs(A[j*ldA+i])>max)
	    {
		max=fabs(A[j*ldA+i]);
		fmax=j;
		cmax=i;
	    }
	}
	if (max>tol)
	{
	    break;
	}
    }
    if (max<=tol)
    {
	*f=fmax;
	*c=cmax;
	return max;
    }
    
    /*Ahora comprobar que se cumple rook y seguir*/
    
    do
    {	
	salir=1;
	for (i=0;i<fA;i++)
	{
	    if (fabs(A[i*ldA+cmax])>max)
	    {
		max=fabs(A[i*ldA+cmax]);
		fmax=i;
		salir=0;
	    }
	}
	if (salir) /*si hemos llegado hasta el final y no hemos encontrado nada mayor, fiesta*/
	    break;
	salir=1;
	for (i=0;i<cA;i++)
	{
	    if (fabs(A[fmax*ldA+i])>max)
	    {
		max=fabs(A[fmax*ldA+i]);
		cmax=i;
		salir=0;
	    }
	}
	if (salir)
	    break;
    }
    while (1); /*poner una medida de seguridad?*/
    *f=fmax;
    *c=cmax;
    return max;
}

/**Busca el primer pivote dominante en fila y columna
 @param A puntero a la matriz de datos
 @param fA número de filas de la (sub)matriz
 @param cA número de columnas de la (sub)matriz
 @param ldA leading dimension. en este caso número de columnas de la matriz original
 @param f parámetro de vuelta. fila donde está el máximo
 @param c parámetro de vuelta. columna donde está el máximo
 @param tol valor del cero numerico
 @return el valor absoluto del pivote
*/
/*double pivotajerookviejo(double *A,unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *f, unsigned long *c, double tol)
{
	double max,maxt;
	unsigned int ft,ct,i,j;
	max=A[0];
	maxt=0;
	ft=0;
	ct=0;
	for (j=0;j<fA;j++)//no sea que todo sea ceros en las primeras filas
	{
		for (i=0;i<cA;i++)
			if (fabs(A[ft*ldA+i])>max)
			{
				max=fabs(A[ft*ldA+i]);
				ct=i;
			}
		if (max>tol)
			break;
		if (max>maxt)
			maxt=max;
		if (ft==fA-1)
		{
			*f=0;
			*c=0;
			return maxt;
		}
		ft+=1;		
	}
	if (j==fA)
	{
		return maxt;
	}
	maxt=max;
	do
	{	
		for (i=0;i<fA;i++)
		{
			if (fabs(A[i*ldA+ct])>maxt)
			{
				maxt=fabs(A[i*ldA+ct]);
				ft=i;
			}
		}
		if (maxt<=max)
		{
			max=maxt;
			break;
		}
		max=maxt;
		for (i=0;i<cA;i++)
		{
			if (fabs(A[ft*ldA+i])>maxt)
			{
				maxt=fabs(A[ft*ldA+i]);
				ct=i;
			}
		}
		if (maxt<=max)
		{
			max=maxt;
			break;
		}
		max=maxt;
	}
	while (1); //poner una medida de seguridad?
	*f=ft;
	*c=ct;
	return max; 
}*/


#define INDICEARRAYDEFILACOLUMNA(t,f,c) (f)*(t)-((f)*((f)+1))/2+(c)

/*Busca el elemento más alto de la matriz simétrica indicada
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param i parámetro de vuelta: indice de la fila del elemento más alto
@param j parámetro de vuelta: indice de la columna del elemento más alto*/
double pivotajecompletosimmatriz(double *sA, unsigned long esA, unsigned long ldsA, unsigned long *i, unsigned long *j)
{
	unsigned long int jt,it;
	double max;
	int m,n;
	*i=0;
	*j=0;
	m=0;
	n=0;
	max=0.;
	for (it=0;it<esA;it++)
	{
		for(jt=it;jt<esA;jt++)
		{
			if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA,it,jt)])>max)
			{
				max=fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA,it,jt)]);
				m=it;
				n=jt;
			}
		}	
	}
	*i=m;
	*j=n;
	return max;
}