/***************************************************************************
 *            matrixldu.h
 *
 *  mar enero 14 12:18:38 2014
 *  Copyright  2014  Igor
 *  <user@host>
 ****************************************************************************/
/*
 * matrixldu.h
 *
 * Copyright (C) 2014 - Igor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Biblioteca de funciones para trabajar con factorizaciones LDU
 * Las funciones empiezan con matrixldu 
 * Ademas de lo definido en vector.h y matrix.h, respecto de la
 * nomenclatura de prototipos de funciones:
	* L: matriz triangular inferior dentro de la matriz factorizada ldu
	* U: matriz triangular superior dentro de la matriz factoriada ldu
	* D: matriz diagonal dentro de la matriz factorizada ldu
	* L1: submatriz de L dentro de la matriz factorizada ldu conteniendo las
		columnas de L correspondiente a columnas linealmente independientes de A
	* L11: submatriz cuadrada de L dentro de la matriz factorizada lu
		conteniendo las r primeras filas/columnas
	* U1: submatriz de U dentro de la matriz factorizada ldu conteniendo las
		filas de U correspondientes a filas linealmente independientes
	* U11:submatriz cuadrada de U dentro de la matriz factorizada lu
		conteniendo las r primeras filas/columnas
	* D1: submatriz de D correspondiente a la matriz cuadrada que contiene
		los r elementos no nulos de D
	* Pf: matriz de pivotaje de filas contenida en array de pivotes
	* Pc: matriz de pivotaje de columnas contenido en array de pivotes
 * Para indicar cual es la matriz factorizada, a continuación del factor o
 submatriz se indica el nombre de la matriz, así, por ejemplo, LA sería la
 matriz triangular inferior dentro de la matriz factorizada A.
*/

#ifndef MATRICESLDU_H
#define MATRICESLDU_H

/**Saca a texto la submatriz L11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduL11atexto(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto (formato OCTAVE) la submatriz L11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduL11atextooctave(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto la submatriz D1
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduD1atexto(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto (formato OCTAVE) la submatriz D1
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduD1atextooctave(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto la submatriz U11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduU11atexto(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto (formato OCTAVE) la submatriz U11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduU11atextooctave(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto la submatriz N1 (parte del subespacio nulo)
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada y debe tener el subespacio nulo calculado
 @param cA numero de columnas de A
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduN1atexto(FILE *archivo, double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto (formato OCTAVE) la submatriz N1 (parte del subespacio nulo)
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada y debe tener el subespacio nulo calculado
 @param cA numero de columnas de A
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduN1atextooctave(FILE *archivo, double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto la submatriz S1T (parte del subespacio nulo izquierdo)
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada y debe tener el subespacio nulo izquierdo calculado
 @param fA numero de filas de A
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduS1Tatexto(FILE *archivo, double *A, unsigned long fA, unsigned long ldA, unsigned long rnA, char cs);

/**Saca a texto (formato OCTAVE) la submatriz S1T (parte del subespacio nulo izquierdo)
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada y debe tener el subespacio nulo izquierdo calculado
 @param fA numero de filas de A
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduS1Tatextooctave(FILE *archivo, double *A, unsigned long fA, unsigned long ldA, unsigned long rnA, char cs);

/**Hace una iteración de la factorización LDU una vez estan reordenadas las
 filas/columnas. Vale para matrices en row_major, es decir, con orden normal
 @param A puntero a los elementos de la matriz (empezando en el elemento 0,0)
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimension fundamental de A
 @param it indice de la posición en la que se quiere hacer la etapa (corresponde
con el índice de etapa*/
void ldumatrizetapareduccion(double *A, unsigned long int fA ,unsigned long int cA, unsigned long int ldA);

/**Realiza la factorización ldu de la matriz A con pivotaje completo. la factorización se hace dentro de la propia matriz A
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivf puntero a los cambios de filas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param eps valor a considerar cero numérico*/
void ldumatrizfactptA(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *rnA, unsigned long *pivf, unsigned long *pivc, double eps);

/**Realiza la factorización ldu de la matriz A con pivotaje de Rook. la factorización se hace dentro de la propia matriz A
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivf puntero a los cambios de filas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param eps valor a considerar como cero numérico*/
void ldumatrizfactrookA(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *rnA, unsigned long *pivf, unsigned long *pivc, double eps);

/**Multiplica la matriz L11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduL11Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la matriz L11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param rnA rango de la matriz factorizada
 @param ldA dimension fundamental de A
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduL11AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la inversa de la matriz L11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvL11Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la inversa de la matriz L11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvL11AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la matriz L11 de la matriz factorizada A traspuesta por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduL11TAv(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la matriz L11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduL11TAB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la inversa de la matriz L11 de la matriz factorizada A traspuesta por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvL11TAv(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la matriz L11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvL11TAB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la matriz D11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduD1Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la matriz D1 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduD1AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la inversa de la matriz D1 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvD1Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la inversa de la matriz D1 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvD1AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la matriz U11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduU11Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la matriz U11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduU11AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la inversa de la matriz U11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvU11Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la inversa de la matriz U11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvU11AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la matriz U11 de la matriz factorizada A traspuesta por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduU11TAv(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la matriz U11 de la matriz factorizada A traspuesta por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduU11TAB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la inversa de la matriz U11 de la matriz factorizada A traspuesta por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvU11TAv(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v);

/**Multiplica la inversa de la matriz U11 de la matriz factorizada A traspuesta por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvU11TAB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB);

/*MANEJO DE SUBESPACIOS Y COSAS POR EL ESTILO*/

/**Calcula la parte superior del subespacio nulo de la matriz factorizada (U11^-1 U12)
 y lo almacena en la parte U_12 de la propia matriz, de manera que la matriz
|-U12|
|  I |
Es el subespacio nulo de la matriz factorizada.
@param cA número de columnas de A
@param ldA dimensión fundamental de A
@param rnA rango de la matriz factorizada*/
void ldumatrizobtenersubespacionulo(double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA);

/**Calcula la parte superior del subespacio nulo izquierdo de la matriz factorizada ( L21 L11^-1)
 y lo almacena en la parte L_21 de la propia matriz, de manera que la matriz
|-L21 || I |
Es el subespacio nulo izquierdo de la matriz factorizada.
@param fA número de filas de A
@param ldA dimensión fundamental de A
@param rnA rango de la matriz factorizada*/
void ldumatrizobtenersubespacionuloizquierdo(double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA);

/**Obtiene una solución particular del sistema factorizado. No tiene por que tener los subespacios calculados, y si los tiene no pasa nada
@param u vector solución. Debe tener el espacio reservado
@param A matriz. Debe estar factorizada
@param fA número de filas de A
@param cA número de columnas de A
@param ldA dimensión fundamental de A
@param rnA rango de la matriz factorizada
@param pivf matriz de cambio de filas de A (dimensión rnA)
@param pivC matriz de cambio de columnas de A (dimensión rnA)
@param v vector de independientes. Debe tener el espacio reservado*/
void ldumatrizsolucionparticular(double *u, double *A, unsigned long int fA, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, unsigned long int *pivf, unsigned long int *pivc, double *v);

/**Establece en el vector destino, u, el escalar a por la matriz S1 en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectoraldusubS1Av(double *u, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *v);

/**Suma al vector destino, u, el escalar a por la matriz S1 en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectorsumaraldusubS1Av(double *u, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *v);

/**Suma a la matriz destino, C la matriz S1 por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraldusubS1AB(double *C, unsigned long ldC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB);

/**Suma al vector destino, u, el escalar a por la matriz S1 traspuesta en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectorsumaraldusubS1ATv(double *u, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *v);

/**Suma a la matriz destino, C la matriz S1 traspuesta por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraldusubS1ATB(double *C, unsigned long ldC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB);

/**Establece en el vector destino, u, el escalar a por la matriz N1 en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectoraldusubN1Av(double *u, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *v);

/**Suma al vector destino, u, el escalar a por la matriz N1 en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectorsumaraldusubN1Av(double *u, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *v);

/**Establece en la matriz destino, C la matriz N1 por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere realizar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de columnas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizaldusubN1AB(double *C, unsigned long ldC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB);

/**Suma a la matriz destino, C la matriz N1 por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de columnas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraldusubN1AB(double *C, unsigned long ldC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB);

/**Establece en a la matriz destino, C, el escalar a por la matriz A por la matriz N1 de B factorizada ldu y con subespacio nulo calculado 
@param C matriz donde se quiere obtener el resultado. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz a multiplicar
@param fA número de filas de la matriz A
@param cA numero de columnas de la matriz, obviamente igual al rango de B
@param ldA dimension fundamental de A
@param B matriz cuyo subespacio nulo se quiere postmultiplicar. Debe estar
 * factorizada y con los subespacios nulos calculados
@param cB número de columnas de B
@param ldB dimension fundamental de la matriz B*/
void matrizaAldusubN1B(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long int cA, unsigned long int ldA, double *B, unsigned long cB, unsigned long ldB);

/**Suma a la matriz destino, C, el escalar a por la matriz A por la matriz N1 de B factorizada ldu y con subespacio nulo calculado 
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz a multiplicar
@param fA número de filas de la matriz A
@param cA numero de columnas de la matriz, obviamente igual al rango de B
@param ldA dimension fundamental de A
@param B matriz cuyo subespacio nulo se quiere postmultiplicar. Debe estar
 * factorizada y con los subespacios nulos calculados
@param cB número de columnas de B
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraAldusubN1B(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long int cA, unsigned long int ldA, double *B, unsigned long cB, unsigned long ldB);

/**Establece en el vector destino, u, el escalar a por la matriz N1 en la matriz A traspuesta
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectoraldusubN1ATv(double *u, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *v);

/**Establece en la matriz destino, C, el escalar a por la matriz N1 en la matriz A traspuesta
 * factorizada ldu con subespacio nulo izquierdo calculado por la matriz B
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de columnas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizaldusubN1ATB(double *C, unsigned long ldC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB);


/**Suma al vector destino, u, el escalar a por la matriz N1 en la matriz A traspuesta
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectorsumaraldusubN1ATv(double *u, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *v);

/**Suma a la matriz destino, C la matriz N1 traspuesta por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de columnas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraldusubN1ATB(double *C, unsigned long ldC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB);

/**Establece en C el producto de A por el subespacio nulo de B
 @param C matriz resultado
 @param ldC dimension fundamental de C
 @param a escalar
 @param A matriz a multiplicar
 @param fA número de filas de A
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param B matriz factorizada ldu debe terner los subespacios calculados
 @param ldB dimensión fundamental de B
 @param rnB rango de B
 @param pivcB cambios de columnas en B en la factorización
*/
void matrizaAldusubNB(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long ldB, unsigned long rnB, unsigned long *pivcB);

/**Establece en &A([rnB]) el producto de A por el subespacio nulo de B (ojo que el ldA se mantiene)
 * El resto de A queda inutilizado
 @param A matriz a reducir
 @param fA número de filas de A
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param B matriz factorizada ldu. Debe tener los subespacios calculados
 @param ldB dimensión fundamental de B
 @param rnB rango de B
 @param pivcB cambios de columnas en B en la factorización
*/
void matrizreduccionAldusubNB(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long ldB, unsigned long rnB, unsigned long *pivcB);

/*Establece en u el resultado del escalar a por el producto del subespacio nulo de A por v
 @param u puntero al vector donde se quiere establecer el resultado
 @param a escalar
 @param A matriz factorizada ldu. Debe tener el subespacio nulo calculado
 @param fA número de filas de A
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param rnA rango de A
 @param pivcA cambios de columnas en A en la factorización
 @param v vector a multiplicar
 */
void vectoraldusubNAv(double *u, double a, double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long *pivcA, double *v);

/**Establece en &v([rnA]) el producto del subespacio nulo de A traspuesto por v.
 * El resto de v queda inutilizado
 @param A matriz factorizada ldu. Debe tener el subespacio nulo calculado
 @param fA número de filas de A
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param rnA rango de A
 @param pivcA cambios de columnas en A en la factorización
 @param v vector a reducir
 */
void vectorreduccionldusubNTAv(double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long *pivcA, double *v);

/**Establece en &B([rnA*ldB]) el producto del subespacio nulo de A traspuesto por B. El ldB se mantiene, ojo.
 * El resto de B queda inutilizado
 @param A matriz factorizada ldu. Debe tener el subespacio nulo calculado
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param rnA rango de A
 @param pivcA cambios de columnas en A en la factorización
 @param B matriz a reducir
 @param cB columnas de B
 @param ldB dimensión fundamental de B
*/
void matrizreduccionldusubNTAB(double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long *pivcA, double *B, unsigned long cB, unsigned long ldB);

/**Crea en la matriz simétrica destino, C, el escalar a por la matriz S1 traspuesta por la matriz S1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubS1TS1A(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA);

/**Suma a la matriz simétrica destino, C, el escalar a por la matriz S1 traspuesta por la matriz S1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizsumaraldusubS1TS1A(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA);

/**Crea en la matriz simétrica destino, C, el escalar a por la matriz S1 traspuesta por la matriz S1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubS1S1TA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA);

/**Suma a la matriz simétrica destino, C, el escalar a por la matriz S1 traspuesta por la matriz S1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizsumaraldusubS1S1TA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA);

/**Crea en la matriz simétrica destino, C, el escalar a por la matriz N1 traspuesta por la matriz N1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubN1TN1A(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA);

/**Suma a la matriz simétrica destino, C, el escalar a por la matriz N1 traspuesta por la matriz N1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizsumaraldusubN1TN1A(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA);

/**Crea en la matriz simétrica destino, C, el escalar a por la matriz N1 traspuesta por la matriz N1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubN1N1TA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA);

/**Suma a la matriz simétrica destino, C, el escalar a por la matriz N1 traspuesta por la matriz N1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizsumaraldusubN1N1TA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA);

/**Construye en la matriz simétrica destino, C el escalar a por la matriz N traspuesta por la matriz N
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubNTNA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA);

/**Construye en la matriz simétrica destino, C el escalar a por la matriz S traspuesta por la matriz S
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubSTSA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA);

/**Reserva espacio para las matrices simétricas necesarias para calcular soluciones de mínimos cuadrados y mínima norma,
 * calcula estas matrices y las factoriza
 @param w parámetro de vuelta. En retorno, puntero al espacio reservado. Se debe librar con free cuando se haya acabado con él
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A
 @return 0 si todo correcto, 1 si no memoria*/
int ldumatrizcalcularsimetricaspinv(double **w,double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA);

/**Resuelve el sistema Au=v, dando solución de norma mínima de mínimos
 * cuadrados, a partir de una matriz ya factorizada y con subespacios
 * calculados. Requiere el cálculo de las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param u vector solución. puede ser igual a v
 * @param A matriz. Debe estar ya factorizada
 * @param fA número de filas de la matriz
 * @param cA número de columnas de la matriz
 * @param ldA dimensión fundamental de la matriz A
 * @param rnA rango de la matriz A
 * @param pivf matriz de cambio de filas de A (dimensión rnA)
 * @param pivC matriz de cambio de columnas de A (dimensión rnA)
 * @param w puntero a las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param v vector de independientes
 */
void vectorpinvlduAv(double *u, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long int *pivf, unsigned long int *pivc, double *w, double *v);

/**Resuelve el sistema AC=B, dando solución de norma mínima de mínimos
 * cuadrados, a partir de una matriz ya factorizada y con subespacios
 * calculados. Requiere el cálculo de las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param C vector solución. Puede ser igual a B
 * @param ldC leading dimension de C.
 * @param A matriz. Debe estar ya factorizada
 * @param fA número de filas de la matriz
 * @param cA número de columnas de la matriz
 * @param ldA dimensión fundamental de la matriz A
 * @param rnA rango de la matriz A
 * @param pivf matriz de cambio de filas de A (dimensión rnA)
 * @param pivC matriz de cambio de columnas de A (dimensión rnA)
 * @param w puntero a las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param B matriz de independientes
 * @param cB número de columnas de B
 * @param ldB leading dimension de B
 */
void matrizpinvlduAB(double *C,unsigned long ldC, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long int *pivf, unsigned long int *pivc, double *w, double *B, unsigned long cB, unsigned long ldB);

/**Resuelve el sistema ATu=v, dando solución de norma mínima de mínimos
 * cuadrados, a partir de una matriz ya factorizada y con subespacios
 * calculados. Requiere el cálculo de las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param u vector solución. puede ser igual a v
 * @param A matriz. Debe estar ya factorizada
 * @param fA número de filas de la matriz
 * @param cA número de columnas de la matriz
 * @param ldA dimensión fundamental de la matriz A
 * @param rnA rango de la matriz A
 * @param pivf matriz de cambio de filas de A (dimensión rnA)
 * @param pivC matriz de cambio de columnas de A (dimensión rnA)
 * @param w puntero a las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param v vector de independientes
 */
void vectorpinvlduATv(double *u, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long int *pivf, unsigned long int *pivc, double *w, double *v);

/*Calcula la pseudoinversa del subespacio nulo de A por el vector v y lo almacena
 en &(V[rnA]). Esto es equivalente a proyectar el vector v en el subespacio. La
 matriz A debe estar factorizada por ldu con los subespacios calculados y con
 las matrices simetricas para calculos de minimos cuadrados de minima norma
 calculadas.
 @param A puntero a la matriz factorizada
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio
 * reservado de al menos el rango de la matriz (en caso de no saber a priori,
 * vale con el menor de fA y fC)
 @param w: puntero a espacio de trabajo. Es para almacenar las matrices
 * simetricas. Debe tener reservado al menos dim*(dim+1)/2+dim2*(dim2+1)/2
 * siendo dim1=min(rnA,cA-rnA) y dim2=min(rnA,fA-rnA). En salida tiene las
 * matrices simétricas necesarias para la solución de mínimos cuadrados de
 * mínima norma
 @param v vector que se quiere reducir
 */
void vectorreduccionldupinvsubNAv(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long int *pivc, double *w, double *v);

/**Realiza la factorización ldu de la matriz A con pivotaje completo, cálculo de
 * subespacios y de matrices simétricas (factorizadas) para el cálculo de solución de mínimos
 * cuadrados de mínima norma. Todo lo posible se hace dentro de la matriz A.
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivf puntero a los cambios de filas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param w puntero a espacio de trabajo. Es para almacenar las matrices simetricas.
 * Debe tener reservado al menos dim*(dim+1)/2+dim2*(dim2+1)/2 siendo dim1=min(rnA,cA-rnA) y dim2=min(rnA,fA-rnA) @param eps valor a considerar cero numérico
 */
void ldumatrizpinvptA(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *rnA, unsigned long *pivf, unsigned long *pivc, double *w, double eps);

/**Realiza la factorización ldu de la matriz A con pivotaje de Rook, cálculo de
 * subespacios y de matrices simétricas (factorizadas) para el cálculo de solución de mínimos
 * cuadrados de mínima norma. Todo lo posible se hace dentro de la matriz A.
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivf puntero a los cambios de filas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param eps valor a considerar como cero numérico
 @param w puntero a espacio de trabajo. Es para almacenar las matrices simetricas.
 * Debe tener reservado al menos dim*(dim+1)/2+dim2*(dim2+1)/2 siendo dim1=min(rnA,cA-rnA) y dim2=min(rnA,fA-rnA)
 */
void ldumatrizpinvrookA(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *rnA, unsigned long *pivf, unsigned long *pivc, double *w, double eps);

#endif /*MATRICESLDU_H*/

