// simmatricesLDLBase.h
//
// Copyright (C) 2011 - Igor Fernández de Bustos & Ibai Coria
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef SIMMATRICESLDL_H
#define SIMMATRICESLDL_H

/**Hace la etapa de reducción de la factorización
 @param sA puntero a la matriz simétrica
 @param esA numero de elementos de la matriz
 @param ldsA dimensión fundamental de la matriz*/
void ldlsimmatrizetapareduccionsA(double *sA, unsigned long esA, unsigned long ldsA);

/**Hace la factorización ldl sin pivotamiento. Interesante para sistemas positivos definidos.
 @param sA puntero a la matriz simátrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de la matriz simétrica*/
void ldlsimmatrizfactsA(double *sA, unsigned long esA, unsigned long ldsA);

/**Multiplica la matriz diagonal D en la matriz sA factorizada ldl por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlDsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v);

/**Multiplica la matriz B por la matriz diagonal D en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param esA número de elementos de la matriz
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B*/
void matrizldlDsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la inversa de la matriz diagonal D en la matriz sA factorizada ldl por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlinvDsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v);

/**Multiplica la matriz B por el inverso de la matriz diagonal D en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B*/
void matrizldlinvDsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la matriz L en la matriz sA factorizada ldl por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlLsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v);

/**Multiplica la matriz B por la matriz L en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B*/
void matrizldlLsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la inversa de la matriz L en la matriz sA factorizada ldl por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlinvLsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v);

/**Multiplica la matriz B por el inverso de la matriz L en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B*/
void matrizldlinvLsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la matriz L en la matriz sA factorizada ldl traspuesta por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlLTsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v);

/**Multiplica la matriz B por la matriz L traspuesta en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlLTsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la inversa de la matriz L en la matriz sA factorizada ldl traspuesta por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlinvLTsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v);

/**Multiplica la matriz B por el inverso de la matriz L traspuesta en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B*/
void matrizldlinvLTsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB);

/**Calcula en u la solución de sA^{-1}v, a partir de la matriz A factorizada ldl
 @param u vector solución. Puede ser igual a v
 @param sA matriz simétrica factorizada
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de sA
 @param v vector de independientes.*/
void vectorldlinvAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v);

/**Calcula en C la solución de sA^{-1}B, a partir de la matriz A factorizada ldl
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlinvAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB);

/**Realiza una factorización completa de la matriz simétrica por el método de rotación (algoritmo recursivo) con pivotaje completo. Los giros solo se aplican en la submatriz. Solo para sistemas compatibles determinados.
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param rnsA en salida, rango de la matriz A
@param piv array de pivotes. Tiene por dimension la dimensión de la submatriz multiplicada por dos. Son cambios sucesivos, es decir,
la fila/columna a cambiar se permuta por la que esta piv veces a la derecha/abajo. Si estamos en la columna i y el pivote es j se
cambia la i por la j+i. Si el elemento de mayor valor absoluto está en la diagonal principal, piv[0]=i, piv[1]=0, si no, piv[0]=i,
piv[1]=j, donde i es la fila del pivote ij en la que aparece el elemento en la diagonal principal
de mayor valor absoluto.
@param tol tolerancia para definir el cero numérico que determina el fin de la factorización*/
void ldlgensimmatrizfactptsAgirosubmatriz(double *sA, unsigned long esA, unsigned long ldsA, unsigned long *rnsA, unsigned long int *piv, double tol);

/**Resolución del problema E=L^-1 E mediante pivotamiento girado completo, con E matriz (incluye giros y cambios de filas/columnas)
 @param sA puntero a la matriz simétrica factorizada con ldlgen
 @param ldsA dimensión fundamental de la matriz
 @param rsA rango de la matriz simétrica
 @param B matriz a multiplicar
 @param cB numero de columnas de B
 @param ldB Dimension pricipal de B
 @param piv puntero a los pivotes*/
void matrizldlgengirosubmatrizinvL11sAB(double *sA, unsigned long int ldsA, unsigned long rsA, double *B, unsigned long cB, unsigned long int ldB, unsigned long int *piv);

/**Resolución del problema v=L11t^-1 v
 @param sA matriz factorizada de forma LDL generalizada
 @param ldsA dimension ppal de A.
 @param rsA rango de la matriz (y tamaño del vector)
 @param v vector
 @param piv puntero a los pivotes*/
void matrizldlgengirosubmatrizinvL11TsAv(double *sA, unsigned long int ldsA, unsigned long int rsA, double *v, unsigned long int *piv);

/**Multiplica la pseudoinversa de la matriz diagonal D en la matriz sA
 * factorizada ldl generalizada por la matriz B y lo almacena en C. C puede
ser la misma matriz que B.
 @param C matriz resultado
 @param ldC dimension principal de C
 @param sA puntero a la matriz simétrica factorizada con ldlgen
 @param ldsA dimensión fundamental de la matriz
 @param rsA rango de la matriz simétrica
 @param B matriz a multiplicar
 @param cB numero de columnas de B
 @param ldB Dimension pricipal de B
*/
void matrizldlgengirosubmatrizpinvDsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rsA,double *B, unsigned long cB, unsigned long ldB);

/**Multiplica la pseudoinversa de la matriz diagonal D en la matriz sA
 * factorizada ldl generalizada por el vector v y lo almacena en el
 * vector u. u puede ser el mismo que v
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param rsA rango de la matriz simétrica
 @param v puntero al vector v*/
void vectorldlgengirosubmatrizpinvDsAv(double *u,  double *sA, unsigned long esA, unsigned long ldsA, unsigned long rsA, double *v);


/*****IBAI*****/

/**Realiza una factorización completa de la matriz simétrica por el método de rotación (algoritmo no recursivo) con pivotaje completo. El giro se aplica en toda la matriz
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param rnsA en salida, rango de la matriz A
@param piv array de pivotes. Tiene por dimension la dimensión de la submatriz multiplicada por dos. Son cambios sucesivos, es decir,
la fila/columna a cambiar se permuta por la que esta piv veces a la derecha/abajo. Si estamos en la columna i y el pivote es j se
cambia la i por la j+i. Si el elemento de mayor valor absoluto está en la diagonal principal, piv[0]=i, piv[1]=0, si no, piv[0]=i,
piv[1]=j, donde i es la fila del pivote ij en la que aparece el elemento en la diagonal principal
de mayor valor absoluto.
@param giros puntero al vector de giros
@param tol tolerancia para definir el cero numérico que determina el fin de la factorización
 */
void ldlgensimmatrizfactptsA(double *sA, unsigned long esA, unsigned long ldsA, unsigned long *rnsA, unsigned long int *piv, double *giros, double tol);

/**Realiza una factorización completa de la matriz simétrica por el método de rotación (algoritmo no recursivo) con pivotaje de rook. El giro se aplica en toda la matriz
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param rnsA en salida, rango de la matriz A
@param piv array de pivotes. Tiene por dimension la dimensión de la submatriz multiplicada por dos. Son cambios sucesivos, es decir,
la fila/columna a cambiar se permuta por la que esta piv veces a la derecha/abajo. Si estamos en la columna i y el pivote es j se
cambia la i por la j+i. Si el elemento de mayor valor absoluto está en la diagonal principal, piv[0]=i, piv[1]=0, si no, piv[0]=i,
piv[1]=j, donde i es la fila del pivote ij en la que aparece el elemento en la diagonal principal
de mayor valor absoluto.
@param giros puntero al vector de giros
@param tol tolerancia para definir el cero numérico que determina el fin de la factorización
 */
void ldlgensimmatrizfactrooksA(double *sA, unsigned long esA, unsigned long ldsA, unsigned long *rnsA, unsigned long int *piv, double *giros, double tol);

/**Calcula la parte superior del subespacio nulo de la matriz factorizada (L'11^-1 L'21)
 y lo almacena en la parte L_21 de la propia matriz, de manera que una vez actualizada la matriz
|-L'21| 
|  I |
Es el subespacio nulo de la matriz.
@param sA puntero a la matriz simétrica factorizada 
@param esA orden de la matriz A
@param ldsA dimensión fundamental de A
@param rnsA rango de la matriz factorizada*/
void ldlgensimmatrizobtenersubespacionulosA(double *sA, unsigned long int esA, unsigned long int ldsA, unsigned long int rnsA);

/**Premultiplica el vector v por n matrices de pivotaje y de giro traspuestas en orden ascendente
@param giros puntero al vector de permutaciones sucesivas
@param piv puntero al vector de permutaciones sucesivas
@param numero de permutaciones y giros aplicadas al vector v
@param v vector a multiplicar*/
void vectorGtPtv(double *giros, unsigned long int *piv, unsigned long ncambios, double *v);

/**Premultiplica el vector v por n matrices de giro y de pivotaje en orden descendiente
@param giros puntero al vector de permutaciones sucesivas
@param piv puntero al vector de permutaciones sucesivas
@param numero de permutaciones y giros aplicadas al vector v
@param v vector a multiplicar*/
void vectorPGv(double *giros, unsigned long int *piv, unsigned long ncambios, double *v);

/**Suma al vector destino, u, el escalar a por la matriz -N1 en la matriz sA
 * (factorizada ldl con subespacio nulo calculado) por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param sA matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param esA numero de filas de la matriz
@param ldsA dimension fundamental de A
@param rnsA rango de la matriz
@param v vector*/
void vectorsumaraldlgenN1sAv(double *u, double a, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rnsA, double *v);

/**Suma al vector destino, u, el escalar a por la matriz -N1' en la matriz sA
 * (factorizada ldl con subespacio nulo calculado) por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param sA matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param esA numero de filas de la matriz
@param ldsA dimension fundamental de A
@param rnsA rango de la matriz
@param v vector*/
void vectorsumaraldlgenN1tsAv(double *u, double a, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rnsA, double *v);

/**Guarda en el vector destino, u, el escalar a por la matriz -N1' en la matriz sA
 * (factorizada ldl con subespacio nulo calculado) por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param sA matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param esA numero de filas de la matriz
@param ldsA dimension fundamental de A
@param rnsA rango de la matriz
@param v vector*/
void vectoraldlgenN1tsAv(double *u, double a, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rnsA, double *v);

/**Resuelve el sistema Au=v, dando solución de norma mínima de mínimos
 * cuadrados, a partir de una matriz simetrica ya factorizada y con subespacio nulo
 * calculado. 
 * @param u vector solución. puede ser igual a v
 * @param sA matriz. Debe estar ya factorizada con la funcion ldlgenfactrookcandidata y ldlmatrizobtenersubespacionulo
 * @param esA número de filas de la matriz
 * @param ldsA dimensión fundamental de la matriz A
 * @param rnsA rango de la matriz A
 * @param piv vector de pivotaje (dimensión rnsA*2 si r<m, y 2*(esA-1) si r=m)
 * @param giros puntero al vector de giros (dimensión rnsA si r<m, y (esA-1) si r=m)
 * @param v vector de independientes
 * @param simmatriz puntero a la matriz simétrica
 */ 
void vectorpginvldlgensAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rnsA, unsigned long int *piv, double *giros, double *v, double*simmatriz);




#endif /*SIMMATRICESLDL_H*/

