/*
 * matrixldu.c
 * Copyright (C) 2014 - Igor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "alglin.h"


/*para matrices simetricas*/
#define INDICEFILACOLUMNASIMETRICA(t,f,c) (f)*(t)-((f)*((f)+1))/2+(c)

/*para indexar dentro de matriz factorizada f: numero de filas. c: numero de columnas. r: rango, if: indice fila, ic indice columna*/
#define INDICEFILACOLUMNAL11(f,c,r,if,ic) (if)*(c)+(ic)
#define INDICEFILACOLUMNAL21(f,c,r,if,ic) ((if)+(r))*(c)+(ic)
#define INDICEFILACOLUMNAU11(c,if,ic) (if)*(c)+(ic)
#define INDICEFILACOLUMNAU12(c,r,if,ic) (if)*(c)+(ic)+(r)

/**Saca a texto la submatriz L11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduL11atexto(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	for (i=0;i<rnA;i++)
	{
		for (j=0;j<i;j++)
		{
			fprintf(archivo,format,A[i*ldA+j]);
			fprintf(archivo,"\t");
		}
			fprintf(archivo,format,1.);
			fprintf(archivo,"\t");
		j++;
		for(;j<rnA;j++)
		{
			fprintf(archivo,format,0.);
			fprintf(archivo,"\t");
		}
		fprintf(archivo,"\b\n");
	}
}

/**Saca a texto (formato OCTAVE) la submatriz L11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduL11atextooctave(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	fprintf(archivo,"[");
	for (i=0;i<rnA;i++)
	{
		for (j=0;j<i;j++)
		{
			fprintf(archivo,format,A[i*ldA+j]);
			fprintf(archivo,", ");
		}
			fprintf(archivo,format,1.);
			fprintf(archivo,", ");
		j++;
		for(;j<rnA;j++)
		{
			fprintf(archivo,format,0.);
			fprintf(archivo,", ");
		}
		fprintf(archivo,"\b\b; ");
	}
	fprintf(archivo,"\b\b]\n");
}

/**Saca a texto la submatriz D1
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduD1atexto(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	for (i=0;i<rnA;i++)
	{
		for (j=0;j<i;j++)
		{
			fprintf(archivo,format,0.);
			fprintf(archivo,"\t");
		}
			fprintf(archivo,format,A[i*ldA+j]);
			fprintf(archivo,"\t");
		j++;
		for(;j<rnA;j++)
		{
			fprintf(archivo,format,0.);
			fprintf(archivo,"\t");
		}
		fprintf(archivo,"\b\n");
	}
}

/**Saca a texto (formato OCTAVE) la submatriz L11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduD1atextooctave(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	fprintf(archivo,"[");
	for (i=0;i<rnA;i++)
	{
		for (j=0;j<i;j++)
		{
			fprintf(archivo,format,0.);
			fprintf(archivo,", ");
		}
			fprintf(archivo,format,A[i*ldA+j]);
			fprintf(archivo,", ");
		j++;
		for(;j<rnA;j++)
		{
			fprintf(archivo,format,0.);
			fprintf(archivo,", ");
		}
		fprintf(archivo,"\b\b; ");
	}
	fprintf(archivo,"\b\b]\n");
}

/**Saca a texto la submatriz U11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduU11atexto(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	for (i=0;i<rnA;i++)
	{
		for (j=0;j<i;j++)
		{
			fprintf(archivo,format,0.);
			fprintf(archivo,"\t");
		}
			fprintf(archivo,format,1.);
			fprintf(archivo,"\t");
		j++;
		for(;j<rnA;j++)
		{
			fprintf(archivo,format,A[i*ldA+j]);
			fprintf(archivo,"\t");
		}
		fprintf(archivo,"\b\n");
	}
}

/**Saca a texto (formato OCTAVE) la submatriz U11
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduU11atextooctave(FILE *archivo, double *A, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	fprintf(archivo,"[");
	for (i=0;i<rnA;i++)
	{
		for (j=0;j<i;j++)
		{
			fprintf(archivo,format,0.);
			fprintf(archivo,", ");
		}
			fprintf(archivo,format,1.);
			fprintf(archivo,", ");
		j++;
		for(;j<rnA;j++)
		{
			fprintf(archivo,format,A[i*ldA+j]);
			fprintf(archivo,", ");
		}
		fprintf(archivo,"\b\b; ");
	}
	fprintf(archivo,"\b\b]\n");
}

/**Saca a texto la submatriz N1 (parte del subespacio nulo)
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada y debe tener el subespacio nulo calculado
 @param cA numero de columnas de A
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduN1atexto(FILE *archivo, double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	for (i=0;i<rnA;i++)
	{
		for (j=rnA;j<cA;j++)
		{
			fprintf(archivo,format,-A[i*ldA+j]);
			fprintf(archivo,"\t");
		}
		fprintf(archivo,"\b\n");
	}
}

/**Saca a texto (formato OCTAVE) la submatriz N1 (parte del subespacio nulo)
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada y debe tener el subespacio nulo calculado
 @param cA numero de columnas de A
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduN1atextooctave(FILE *archivo, double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	fprintf(archivo,"[");
	
	for (i=0;i<rnA;i++)
	{
		for (j=rnA;j<cA;j++)
		{
			fprintf(archivo,format,-A[i*ldA+j]);
			fprintf(archivo,", ");
		}
		fprintf(archivo,"\b\b; ");
	}
	fprintf(archivo,"\b\b]\n");
}

/**Saca a texto la submatriz S1T (parte del subespacio nulo izquierdo)
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada y debe tener el subespacio nulo izquierdo calculado
 @param fA numero de filas de A
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduS1Tatexto(FILE *archivo, double *A, unsigned long fA, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	for (i=rnA;i<fA;i++)
	{
		for (j=0;j<rnA;j++)
		{
			fprintf(archivo,format,-A[i*ldA+j]);
			fprintf(archivo,"\t");
		}
		fprintf(archivo,"\b\n");
	}
}

/**Saca a texto (formato OCTAVE) la submatriz S1T (parte del subespacio nulo izquierdo)
 @param archivo puntero al archivo donde se quiere volcar
 @param A matriz. Debe estar factorizada y debe tener el subespacio nulo izquierdo calculado
 @param fA numero de filas de A
 @param ldA leading dimension de A
 @param rnA rango de la matriz A
 @param cs numero de cifras significativas a escribir*/
void matrizlduS1Tatextooctave(FILE *archivo, double *A, unsigned long fA, unsigned long ldA, unsigned long rnA, char cs)
{
	unsigned long i,j;
	char format[10];
	snprintf(&(format[1]),9,".%dE", (int)cs);
	format[0]='%';
	fprintf(archivo,"[");
	
	for (i=rnA;i<fA;i++)
	{
		for (j=0;j<rnA;j++)
		{
			fprintf(archivo,format,-A[i*ldA+j]);
			fprintf(archivo,", ");
		}
		fprintf(archivo,"\b\b; ");
	}
	fprintf(archivo,"\b\b]\n");
}

/*Permuta dos filas de la matriz A
 @param A puntero a la matriz
 @param cA número de columnas de la matriz
 @param i indice de la primera fila a permutar
 @param j indice de la segunda fila a permutar*/
void matrizcambiarfilas(double *A,unsigned long cA, unsigned long ldA, unsigned long i, unsigned long j)
{
	double a;
	unsigned long m;
	for(m=0; m<cA; m++)
	{
		a=A[i*ldA+m];
		A[i*ldA+m]=A[j*ldA+m];
		A[j*ldA+m]=a;
	}
}

/*Permuta dos columnas de la matriz A
 @param A puntero a la matriz
 @param fA número de filas de la matriz
 @param i indice de la primera columna a permutar
 @param j indice de la segunda columna a permutar*/
void matrizcambiarcolumnas(double *A, unsigned long fA, unsigned long ldA, unsigned long i, unsigned long j)
{
	double a;
	unsigned long m;
	for(m=0; m<fA; m++)
	{
		a=A[m*ldA+i];
		A[m*ldA+i]=A[m*ldA+j];
		A[m*ldA+j]=a;
	}
}

/**Hace una iteración de la factorización LDU una vez estan reordenadas las
 filas/columnas. Vale para matrices en row_major, es decir, con orden normal
 @param A puntero a los elementos de la matriz (empezando en el elemento 0,0)
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimension fundamental de A
 */
void ldumatrizetapareduccion(double *A, unsigned long int fA ,unsigned long int cA, unsigned long int ldA)
{
	unsigned long int i,j;
	double dm,a;
	/*dividir fila it por m(it,it)*/
	dm=1./A[0]; /*inverso del pivote*/
	for (i=1;i<cA;i++) /*dividimos fila por pivote*/
		A[i]*=dm;
	/*reducción - a pelo*/
	for (j=1;j<fA;j++) /*por filas es mas eficaz*/
	{
		a=A[j*ldA];
		for (i=1;i<cA;i++)
			/*FMA*/
			/*m[j*c+i]=fma(-a,m[it*c+i],m[j*c+i]);*/
			/*NO FMA*/
			A[j*ldA+i]-=a*A[i];
	}
	/*reduccion - blas OJO, AL CAMBIAR SISTEMA HABRÍA QUE REVISAR LLAMADA*/
	/*cblas_dger(CblasRowMajor,f-it-1, c-it-1,-1.0,&(m[(it+1)*c+it]),c,&(m[it*c+it+1]),1,&(m[(it+1)*c+it+1]),c);*/

	/*dividir columna it por m(it,it)*/

	/*DEBUG*/
	/*printf("valor de m(it,it):%g\n",m[it*c+it]);*/
	for (j=1;j<fA;j++)
		A[j*ldA]*=dm;
}


/*FACTORIZACIONES*/

/**Realiza la factorización ldu de la matriz A con pivotaje completo. la factorización se hace dentro de la propia matriz A
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivf puntero a los cambios de filas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param eps valor a considerar como cero numérico*/
void ldumatrizfactptA(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *rnA, unsigned long *pivf, unsigned long *pivc, double eps)
{
	unsigned long int i,min;
	double piv;
	if (fA<cA)
		min=fA;
	else
		min=cA;
	for (i=0;i<min;i++)
	{
		piv=pivotajecompleto(&(A[ldA*i+i]),fA-i,cA-i,ldA,&(pivf[i]),&(pivc[i]));
		if (piv<=eps)/*TODO: se pueden implementar otros métodos de calcular el cero numérico*/
		{
			pivf[i]=0;
			pivc[i]=0;/*medida de seguridad.*/
			*rnA=i;
			return;
		}
		matrizcambiarfilas(A,cA, ldA,i,pivf[i]+i);
		matrizcambiarcolumnas(A,fA, ldA,i,pivc[i]+i);
		ldumatrizetapareduccion(&(A[i*ldA+i]),fA-i,cA-i,ldA);
	}
	*rnA=i;
}

/**Realiza la factorización ldu de la matriz A con pivotaje de Rook. la factorización se hace dentro de la propia matriz A
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivf puntero a los cambios de filas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param eps valor a considerar como cero numérico*/
void ldumatrizfactrookA(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *rnA, unsigned long *pivf, unsigned long *pivc, double eps)
{
	unsigned long int i,min;
	double piv;
	if (fA<cA)
		min=fA;
	else
		min=cA;
	for (i=0;i<min;i++)
	{
		piv=pivotajerook(&(A[ldA*i+i]),fA-i,cA-i,ldA,&(pivf[i]),&(pivc[i]),eps);
		if (piv<=eps)/*TODO: se pueden implementar otros métodos de calcular el cero numérico*/
		{
			pivf[i]=0;
			pivc[i]=0;/*medida de seguridad.*/
			/*printf("piv:%g<eps:%g\n", piv,eps);*/
			*rnA=i;
			return;
		}
		matrizcambiarfilas(A,cA, ldA,i,pivf[i]+i);
		matrizcambiarcolumnas(A,fA, ldA,i,pivc[i]+i);
		ldumatrizetapareduccion(&(A[i*ldA+i]),fA-i,cA-i,ldA);
	}
	
	/*printf("Salimos de factorización con i=%lu\n", i);*/
	
	*rnA=i;
}

/**Multiplica la matriz L11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduL11Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long j,k;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	if (rnA==0)
		return;
	for (k=rnA;k>0;k--)
	{
		u[k-1]=v[k-1];
		for (j=0;j<k-1;j++)
			u[k-1]+=v[j]*A[ldA*(k-1)+j];
	}
}

/**Multiplica la matriz L11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduL11AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i,j,k;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	if (rnA*cB==0)
		return;
	for (k=rnA;k>0;k--)
	{
		for (i=0;i<cB;i++)
			C[ldC*(k-1)+i]=B[ldC*(k-1)+i];
		for (j=0;j<k-1;j++)
			for (i=0;i<cB;i++)
				C[ldC*(k-1)+i]+=B[ldC*j+i]*A[ldA*(k-1)+j];
	}
}

/**Multiplica la inversa de la matriz L11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvL11Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long i,j;
	for (i=0;i<rnA;i++)
	{
		u[i]=v[i];
		for (j=0;j<i;j++)
			u[i]-=u[j]*A[ldA*i+j];
	}
}

/**Multiplica la inversa de la matriz L11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvL11AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	if (rnA*cB==0)
		return;
	unsigned long i,j,k;
	for (i=0;i<rnA;i++)
	{
		for (k=0;k<cB;k++)
			C[ldC*i+k]=B[ldB*i+k];
		for (j=0;j<i;j++)
			for (k=0;k<cB;k++)
				C[ldC*i+k]-=C[ldC*j+k]*A[ldA*i+j];
	}
}

/**Multiplica la matriz L11 de la matriz factorizada A traspuesta por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduL11TAv(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long i,j;
	u[0]=v[0];
	for (i=1;i<rnA;i++)
	{
		for (j=0;j<i;j++)
			u[j]+=v[i]*A[ldA*i+j];
		u[i]=v[i];
	}
}

/**Multiplica la matriz L11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduL11TAB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i,j,k;
	for (k=0;k<cB;k++)
		C[k]=B[k];
	for (i=1;i<rnA;i++)
	{
		for (j=0;j<i;j++)
			for (k=0;k<cB;k++)
				C[ldC*j+k]+=B[ldB*i+k]*A[ldA*i+j];
		for (k=0;k<cB;k++)
			C[ldC*i+k]=B[ldB*i+k];
	}
}

/**Multiplica la inversa de la matriz L11 de la matriz factorizada A traspuesta por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvL11TAv(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long i,j;
	if (rnA==0)
		return;
	u[rnA-1]=v[rnA-1];
	for (j=0;j<rnA-1;j++)
		u[j]=v[j]-u[rnA-1]*A[ldA*(rnA-1)+j];
	for (i=rnA-1;i>0;i--)
	{
		for (j=0;j<i-1;j++)
			u[j]-=u[i-1]*A[ldA*(i-1)+j];
	}
}

/**Multiplica la matriz L11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvL11TAB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i,j,k;
	if (rnA*cB==0)
		return;
	for (k=0;k<cB;k++)
		C[ldC*(rnA-1)+k]=B[ldB*(rnA-1)+k];
	for (j=0;j<rnA-1;j++)
		for (k=0;k<cB;k++)
			C[ldC*j+k]=C[ldC*j+k]-B[ldB*(rnA-1)+k]*A[ldA*(rnA-1)+j];
	for (i=rnA-1;i>0;i--)
	{
		for (j=0;j<i-1;j++)
			for (k=0;k<cB;k++)
				C[ldC*j+k]-=C[ldC*(i-1)+k]*A[ldA*(i-1)+j];
	}	
}

/**Multiplica la matriz D1 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduD1Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long k;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	for (k=0;k<rnA;k++)
	{
		u[k]=v[k]*A[ldA*k+k];
	}
}

/**Multiplica la matriz D1 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduD1AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long k,j;
	if (rnA*cB==0)
		return;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	for (k=0;k<rnA;k++)
	{
		for (j=0;j<cB;j++)
			C[ldC*k+j]=B[ldB*k+j]*A[ldA*k+k];
	}
}

/**Multiplica la inversa de la matriz D1 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvD1Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long k;
	for (k=0;k<rnA;k++)
	{
		u[k]=v[k]/A[ldA*k+k];
	}
}

/**Multiplica la inversa de la matriz D1 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvD1AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long k,j;
	if (rnA*cB==0)
		return;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	for (k=0;k<rnA;k++)
	{
		for (j=0;j<cB;j++)
			C[ldC*k+j]=B[ldB*k+j]/A[ldA*k+k];
	}
}

/**Multiplica la matriz U11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduU11Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long j,k;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	for (k=0;k<rnA;k++)
	{
		u[k]=v[k];
		for (j=k+1;j<rnA;j++)
			u[k]+=v[j]*A[ldA*k+j];
	}
}

/**Multiplica la matriz U11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduU11AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long j,k,i;
	if (rnA*cB==0)
		return;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	for (k=0;k<rnA;k++)
	{
		for (i=0;i<cB;i++)
			C[ldC*k+i]=B[ldB*k+i];
		for (j=k+1;j<rnA;j++)
			for (i=0;i<cB;i++)
				C[ldC*k+i]+=B[ldB*j+i]*A[ldA*k+j];
	}
}

/**Multiplica la inversa de la matriz U11 de la matriz factorizada A por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvU11Av(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long j,k;
	if (rnA==0)
		return;
	for (k=rnA;k>0;k--)
	{
		u[k-1]=v[k-1];
		for (j=k;j<rnA;j++)
			u[k-1]-=u[j]*A[ldA*(k-1)+j];
	}
}

/**Multiplica la inversa de la matriz U11 de la matriz factorizada A por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvU11AB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long j,k,i;
	if (rnA*cB==0)
		return;
	for (k=rnA;k>0;k--)
	{
		for (i=0;i<cB;i++)
			C[ldC*(k-1)+i]=B[ldB*(k-1)+i];
		for (j=k;j<rnA;j++)
			for (i=0;i<cB;i++)
				C[ldC*(k-1)+i]-=C[ldC*j+i]*A[ldA*(k-1)+j];
	}
}

/**Multiplica la matriz U11 de la matriz factorizada A traspuesta por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduU11TAv(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long i,j;
	if (rnA==0)
		return;
	for (i=rnA;i>0;i--)
	{
		for (j=i;j<rnA;j++)
			u[j]+=v[i-1]*A[ldA*(i-1)+j];
		u[i-1]=v[i-1];
	}
	
}

/**Multiplica la matriz U11 de la matriz factorizada A traspuesta por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduU11TAB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i,j,k;
	if (rnA*cB==0)
		return;
	for (i=rnA;i>0;i--)
	{
		for (j=i;j<rnA;j++)
			for (k=0;k<cB;k++)
				C[ldC*j+k]+=B[ldC*(i-1)+k]*A[ldA*(i-1)+j];
		for (k=0;k<cB;k++)
			C[ldC*(i-1)+k]=B[ldB*(i-1)+k];
	}
}

/**Multiplica la inversa de la matriz U11 de la matriz factorizada A traspuesta por el vector v
 @param u puntero al vector resultado. Puede coincidir con el origen
 @param A puntero a la matriz factorizada LDU
 @param ldA dimensión fundamental de la matriz factorizada
 @param rnA rango de la matriz
 @param v puntero al vector*/
void vectorlduinvU11TAv(double *u, double *A, unsigned long ldA, unsigned long rnA, double*v)
{
	unsigned long i,j;
	if (rnA==0)
		return;
	memcpy(u,v,sizeof(double)*rnA);
	for (i=0;i<rnA;i++)
	{
		for (j=i+1;j<rnA;j++)
			u[j]-=u[i]*A[ldA*i+j];
	}
}

/**Multiplica la inversa de la matriz U11 de la matriz factorizada A traspuesta por la matriz B
 @param C puntero a la matriz resultado. Puede ser igual a B
 @param ldC dimensión fundamental de C
 @param A puntero a los elementos de la matriz factorizada
 @param ldA dimension fundamental de A
 @param rnA rango de la matriz factorizada
 @param B puntero a los elementos de la matriz a multiplicar y en salida resultado
 @param cB número de columnas de B
 @param ldB dimension fundamental de B
 */
void matrizlduinvU11TAB(double *C, unsigned long ldC, double *A, unsigned long int ldA, unsigned long int rnA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i,j,k;
	if (rnA*cB==0)
		return;
	for (i=rnA;i>0;i--)
	{
		for (k=0;k<cB;k++)
			C[ldC*(i-1)+k]=B[ldB*(i-1)+k];
		for (j=i;j<rnA;j++)
			for (k=0;k<cB;k++)
				C[ldC*j+k]-=C[ldC*(i-1)+k]*A[ldA*(i-1)+j];
		
	}
}


/*MANEJO DE SUBESPACIOS Y COSAS POR EL ESTILO*/

/**Calcula la parte superior del subespacio nulo de la matriz factorizada (U11^-1 U12)
 y lo almacena en la parte U_12 de la propia matriz, de manera que la matriz
|-U12|
|  I |
Es el subespacio nulo de la matriz factorizada.
@param cA número de columnas de A
@param ldA dimensión fundamental de A
@param rnA rango de la matriz factorizada*/
void ldumatrizobtenersubespacionulo(double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA)
{
	unsigned long int i,j,k;
	if (rnA==0)
		return;
	for (k=rnA-1;k>0;k--)
		for (j=0;j<k;j++)
			for (i=0;i<cA-rnA;i++)
				A[INDICEFILACOLUMNAU12(ldA,rnA,j,i)]-=A[INDICEFILACOLUMNAU11(ldA,j,k)]*A[INDICEFILACOLUMNAU12(ldA,rnA,k,i)];
}

/**Calcula la parte superior del subespacio nulo izquierdo de la matriz factorizada ( L21 L11^-1)
 y lo almacena en la parte L_21 de la propia matriz, de manera que la matriz
|-L21 || I |
Es el subespacio nulo izquierdo de la matriz factorizada.
@param fA número de filas de A
@param ldA dimensión fundamental de A
@param rnA rango de la matriz factorizada*/
void ldumatrizobtenersubespacionuloizquierdo(double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA)
{
	unsigned long int i,j,k;
	if (rnA==0)
		return;
	for (j=rnA-1;j>0;j--)/*para el tamaño de L11, que es rxr, menos el primero (por 1) y el último (por unsigned)*/
	{
		for (i=0;i<fA-rnA;i++)/*para todas las filas de L12*/
		{
			for (k=0;k<j;k++) /*columnas*/
			{
				A[INDICEFILACOLUMNAL21(fA,ldA,rnA,i,k)]-=A[INDICEFILACOLUMNAL21(fA,ldA,rnA,i,j)]*A[INDICEFILACOLUMNAL11(fA,ldA,rnA,j,k)];
			}
		}
	}
}

/**Obtiene una solución particular del sistema factorizado. No tiene por que tener los subespacios calculados, y si los tiene no pasa nada
@param u vector solución. Debe tener el espacio reservado
@param A matriz. Debe estar factorizada
@param fA número de filas de A
@param cA número de columnas de A
@param ldA dimensión fundamental de A
@param rnA rango de la matriz factorizada
@param pivf matriz de cambio de filas de A (dimensión rnA)
@param pivC matriz de cambio de columnas de A (dimensión rnA)
@param v vector de independientes. Debe tener el espacio reservado*/
void ldumatrizsolucionparticular(double *u, double *A, unsigned long int fA, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, unsigned long int *pivf, unsigned long int *pivc, double *v)
{
	unsigned long i;
	if (fA>cA)
	{
		vectorPsv(pivf, rnA, v);
		memcpy(u,v,sizeof(double)*cA);
		vectorPsTv(pivf,rnA,v);
		for (i=rnA;i<cA;i++)
			u[i]=0.;
	}
	else
	{
		memcpy(u,v,sizeof(double)*fA);
		vectorPsv(pivf, rnA, u);
		for (i=rnA;i<fA;i++)
			u[i]=0.;
	}
	vectorlduinvL11Av(u,A,ldA,rnA,u);
	/*multiplicamos por D11m1->z1*/
	vectorlduinvD1Av(u,A,ldA,rnA,u);
	/*Tenemos z*/
	vectorlduinvU11Av(u,A,ldA,rnA,u);
	for (i=rnA;i<cA;i++)
			u[i]=0.;
	vectorPsTv(pivc,rnA,u);
}

/**Establece en el vector destino, u, el escalar a por la matriz S1 en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectoraldusubS1Av(double *u, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *v)
{
	vectoraATv(u,-a,&(A[rnA*ldA]),fA-rnA,rnA,ldA,v);
}

/**Suma al vector destino, u, el escalar a por la matriz S1 en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectorsumaraldusubS1Av(double *u, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *v)
{
	vectorsumaraATv(u,-a,&(A[rnA*ldA]),fA-rnA,rnA,ldA,v);
}

/**Suma a la matriz destino, C la matriz S1 por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraldusubS1AB(double *C, unsigned long ldC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB)
{
	matrizsumaraATB(C,ldC,-a,&(A[rnA*ldA]),fA-rnA,rnA,ldA,B,cB,ldB);
}

/**Suma al vector destino, u, el escalar a por la matriz S1 traspuesta en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectorsumaraldusubS1ATv(double *u, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *v)
{
	vectorsumaraAv(u,-a,&(A[rnA*ldA]),fA-rnA,rnA,ldA,v);
}

/**Suma a la matriz destino, C la matriz S1 traspuesta por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param fA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraldusubS1ATB(double *C, unsigned long ldC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB)
{
	matrizsumaraAB(C,ldC,-a,&(A[rnA*ldA]),fA-rnA, rnA,ldA,B,cB,ldB);
}

/**Establece en el vector destino, u, el escalar a por la matriz N1 en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectoraldusubN1Av(double *u, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *v)
{
	vectoraAv(u,-a,&(A[rnA]),rnA,cA-rnA,ldA,v);
}

/**Suma al vector destino, u, el escalar a por la matriz N1 en la matriz A
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectorsumaraldusubN1Av(double *u, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *v)
{
	vectorsumaraAv(u,-a,&(A[rnA]),rnA,cA-rnA,ldA,v);
}

/**Establece en la matriz destino, C la matriz N1 por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere realizar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de columnas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizaldusubN1AB(double *C, unsigned long ldC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB)
{
	matrizaAB(C,ldC,-a,&(A[rnA]),rnA,cA-rnA,ldA,B,cB,ldB);
}


/**Suma a la matriz destino, C la matriz N1 por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de columnas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraldusubN1AB(double *C, unsigned long ldC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB)
{
	matrizsumaraAB(C,ldC,-a,&(A[rnA]),rnA,cA-rnA,ldA,B,cB,ldB);
}

/**Establece en a la matriz destino, C, el escalar a por la matriz A por la matriz N1 de B factorizada ldu y con subespacio nulo calculado 
@param C matriz donde se quiere obtener el resultado. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz a multiplicar
@param fA número de filas de la matriz A
@param cA numero de columnas de la matriz, obviamente igual al rango de B
@param ldA dimension fundamental de A
@param B matriz cuyo subespacio nulo se quiere postmultiplicar. Debe estar
 * factorizada y con los subespacios nulos calculados
@param cB número de columnas de B
@param ldB dimension fundamental de la matriz B*/
void matrizaAldusubN1B(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long int cA, unsigned long int ldA, double *B, unsigned long cB, unsigned long ldB)
{
	matrizaAB(C,ldC,-a,A,fA,cA,ldA,&(B[cA]),cB-cA,ldB);
}

/**Suma a la matriz destino, C, el escalar a por la matriz A por la matriz N1 de B factorizada ldu y con subespacio nulo calculado 
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz a multiplicar
@param fA número de filas de la matriz A
@param cA numero de columnas de la matriz, obviamente igual al rango de B
@param ldA dimension fundamental de A
@param B matriz cuyo subespacio nulo se quiere postmultiplicar. Debe estar
 * factorizada y con los subespacios nulos calculados
@param cB número de columnas de B
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraAldusubN1B(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long int cA, unsigned long int ldA, double *B, unsigned long cB, unsigned long ldB)
{	
	matrizsumaraAB(C,ldC,-a,A,fA,cA,ldA,&(B[cA]),cB-cA,ldB);
}

/**Establece en el vector destino, u, el escalar a por la matriz N1 en la matriz A traspuesta
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectoraldusubN1ATv(double *u, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *v)
{
	vectoraATv(u,-a,&(A[rnA]),rnA,cA-rnA,ldA,v);
}

/**Establece en la matriz destino, C, el escalar a por la matriz N1 en la matriz A traspuesta
 * factorizada ldu con subespacio nulo izquierdo calculado por la matriz B
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de columnas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizaldusubN1ATB(double *C, unsigned long ldC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB)
{
	matrizaATB(C,ldC,-a,&(A[rnA]),rnA,cA-rnA,ldA,B,cB,ldB);
}


/**Suma al vector destino, u, el escalar a por la matriz N1 en la matriz A traspuesta
 * factorizada ldu con subespacio nulo izquierdo calculado por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de filas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param v vector*/
void vectorsumaraldusubN1ATv(double *u, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *v)
{
	vectorsumaraATv(u,-a,&(A[rnA]),rnA,cA-rnA,ldA,v);
}

/**Suma a la matriz destino, C la matriz N1 traspuesta por la matriz B y todo ello multiplicado por el escalar a
@param C matriz donde se quiere sumar la operación. Debe tener espacio reservado
@param ldC dimension fundamental de C
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param A matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param cA numero de columnas de la matriz
@param ldA dimension fundamental de A
@param rnA rango de la matriz
@param B matriz a multiplicar
@param cB numero de columnas de la matriz
@param ldB dimension fundamental de la matriz B*/
void matrizsumaraldusubN1ATB(double *C, unsigned long ldC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA, double *B, double cB, unsigned long ldB)
{
	matrizsumaraATB(C,ldC,-a,&(A[rnA]),rnA,cA-rnA,ldA,B,cB,ldB);
}

/**Establece en C el producto de A por el subespacio nulo de B
 @param C matriz resultado
 @param ldC dimension fundamental de C
 @param a escalar
 @param A matriz a multiplicar
 @param fA número de filas de A
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param B matriz factorizada ldu. Debe tener los subespacios calculados
 @param ldB dimensión fundamental de B
 @param rnB rango de B
 @param pivcB cambios de columnas en B en la factorización
*/
void matrizaAldusubNB(double *C, unsigned long ldC, double a, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long ldB, unsigned long rnB, unsigned long *pivcB)
{
	matrizAPsT(A,fA,ldA,pivcB,rnB);
	/*printf("APsT:\n");
	matrizatexto(stdout,A,fA,cA,ldA,5);*/
	
	matrizaA(C,ldC,a,&(A[rnB]),fA,cA-rnB,ldA);
	
	/*printf("PivcB[0]:%ld\n",pivcB[0]);
	printf("Medio AN\n");
	matrizatexto(stdout,C,fA,cA-rnB,ldC,5);*/
	
	matrizsumaraAldusubN1B(C, ldC, a, A, fA, rnB, ldA, B, cA, ldB);
	matrizAPs(A,fA,ldA,pivcB,rnB);
}

/**Establece en &A([rnB]) el producto de A por el subespacio nulo de B (ojo que el ldA se mantiene)
 * El resto de A queda inutilizado
 @param A matriz a reducir
 @param fA número de filas de A
 @param cA número de columnas de A. Igual al de B (ambas se pueden postmultiplicar por el subespacio nulo de B)
 @param ldA dimensión fundamental de A
 @param B matriz factorizada ldu. Debe tener los subespacios calculados
 @param ldB dimensión fundamental de B
 @param rnB rango de B
 @param pivcB cambios de columnas en B en la factorización
*/
void matrizreduccionAldusubNB(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, double *B, unsigned long ldB, unsigned long rnB, unsigned long *pivcB)
{
	matrizAPsT(A,fA,ldA,pivcB,rnB);
	matrizsumaraAldusubN1B(&(A[rnB]), ldA, 1., A, fA, rnB, ldA, B, cA, ldB);
}

/*Establece en u el resultado del escalar a por el producto del subespacio nulo de A por v
 @param u puntero al vector donde se quiere establecer el resultado
 @param a escalar
 @param A matriz factorizada ldu. Debe tener el subespacio nulo calculado
 @param fA número de filas de A
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param rnA rango de A
 @param pivcA cambios de columnas en A en la factorización
 @param v vector a multiplicar
 */
void vectoraldusubNAv(double *u, double a, double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long *pivcA, double *v)
{
	vectoraldusubN1Av(u,a,A,cA,ldA,rnA,v);
	vectorav(&(u[rnA]),a,v,cA-rnA);
	vectorPsTv(pivcA,rnA,u);
}

/**Establece en &v([rnA]) el producto del subespacio nulo de A traspuesto por v.
 * El resto de v queda inutilizado
 @param A matriz factorizada ldu. Debe tener el subespacio nulo calculado
 @param fA número de filas de A
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param rnA rango de A
 @param pivcA cambios de columnas en A en la factorización
 @param v vector a reducir
 */
void vectorreduccionldusubNTAv(double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long *pivcA, double *v)
{
	vectorPsv(pivcA,rnA,v);
	vectorsumaraldusubN1ATv(&(v[rnA]),1.,A ,cA,ldA,rnA,v);
}

/**Establece en &B([rnA*ldB]) el producto del subespacio nulo de A traspuesto por B. El ldB se mantiene, ojo.
 * El resto de B queda inutilizado
 @param A matriz factorizada ldu. Debe tener el subespacio nulo calculado
 @param cA número de columnas de A
 @param ldA dimensión fundamental de A
 @param rnA rango de A
 @param pivcA cambios de columnas en A en la factorización
 @param B matriz a reducir
 @param cB columnas de B
 @param ldB dimensión fundamental de B
*/
void matrizreduccionldusubNTAB(double *A, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long *pivcA, double *B, unsigned long cB, unsigned long ldB)
{
	matrizPsA(pivcA,rnA,B,cB,ldB);
	matrizsumaraldusubN1ATB(&(B[rnA*ldB]), ldB ,1.,A ,cA,ldA,rnA, B,cB,ldB);
}

/**Crea en la matriz simétrica destino, C, el escalar a por la matriz S1 traspuesta por la matriz S1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubS1TS1A(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizaAAT(sC,ldsC,a,&(A[rnA*ldA]),fA-rnA,rnA,ldA);
}

/**Suma a la matriz simétrica destino, C, el escalar a por la matriz S1 traspuesta por la matriz S1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizsumaraldusubS1TS1A(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizsumaraAAT(sC,ldsC,a,&(A[rnA*ldA]),fA-rnA,rnA,ldA);
}

/**Crea en la matriz simétrica destino, C, el escalar a por la matriz S1 traspuesta por la matriz S1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubS1S1TA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizaATA(sC,ldsC,a,&(A[rnA*ldA]),fA-rnA,rnA,ldA);
}

/**Suma a la matriz simétrica destino, C, el escalar a por la matriz S1 traspuesta por la matriz S1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizsumaraldusubS1S1TA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizsumaraATA(sC,ldsC,a,&(A[rnA*ldA]),fA-rnA,rnA,ldA);
}

/**Crea en la matriz simétrica destino, C, el escalar a por la matriz N1 traspuesta por la matriz N1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubN1TN1A(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizaATA(sC,ldsC,a,&(A[rnA]),rnA,cA-rnA,ldA);
}

/**Suma a la matriz simétrica destino, C, el escalar a por la matriz N1 traspuesta por la matriz N1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizsumaraldusubN1TN1A(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizsumaraATA(sC,ldsC,a,&(A[rnA]),rnA,cA-rnA,ldA);
}

/**Crea en la matriz simétrica destino, C, el escalar a por la matriz N1 traspuesta por la matriz N1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubN1N1TA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizaAAT(sC,ldsC,a,&(A[rnA]),rnA,cA-rnA,ldA);
}

/**Suma a la matriz simétrica destino, C, el escalar a por la matriz N1 traspuesta por la matriz N1
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizsumaraldusubN1N1TA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizsumaraAAT(sC,ldsC,a,&(A[rnA]),rnA,cA-rnA,ldA);
}

/**Construye en la matriz simétrica destino, C el escalar a por la matriz N traspuesta por la matriz N
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param cA número de columnas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubNTNA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int cA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizaldusubN1TN1A(sC,ldsC,a,A,cA,ldA,rnA);
	simmatrizsumaraI(sC,cA-rnA,ldsC,a);
}

/**Construye en la matriz simétrica destino, C el escalar a por la matriz S traspuesta por la matriz S
 @param sC puntero a la matriz simétrica
 @param ldsC dimensión fundamental de la matriz simétrica
 @param a escalar
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A*/
void simmatrizaldusubSTSA(double *sC, unsigned long ldsC, double a, double *A, unsigned long int fA, unsigned long int ldA, unsigned long int rnA)
{
	simmatrizaldusubS1TS1A(sC,ldsC,a,A,fA,ldA,rnA);
	simmatrizsumaraI(sC,fA-rnA,ldsC,a);
}

/**Reserva espacio para las matrices simétricas necesarias para calcular soluciones de mínimos cuadrados y mínima norma,
 * calcula estas matrices y las factoriza
 @param w parámetro de vuelta. En retorno, puntero al espacio reservado. Se debe librar con free cuando se haya acabado con él
 @param A puntero a la matriz factorizada ldu con el subespacio nulo calculado
 @param fA número de filas de la matriz A
 @param ldA dimensión fundamental de la matriz A
 @param rnA rango de la matriz A
 @return 0 si todo correcto, 1 si no memoria*/
int ldumatrizcalcularsimetricaspinv(double **w,double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA)
{
	unsigned long dim, tam, dim2;
	double *w2;
	dim=rnA<(fA-rnA)?rnA:(fA-rnA);
	tam=dim*(dim+1)/2;
	dim2=tam;
	dim=rnA<(cA-rnA)?rnA:(cA-rnA);
	tam+=dim*(dim+1)/2;
	if (tam==0)
		return 0;
	*w=malloc(sizeof(double)*tam);
	if (w==NULL)
		return 1;
	w2=&((*w)[dim2]);
	if (rnA<fA-rnA)
	{
		simmatrizaldusubS1S1TA(*w,rnA,1.,A,fA,ldA,rnA);
		simmatrizsumaraI(*w,rnA,rnA,1.);
		ldlsimmatrizfactsA(*w,rnA,rnA);
	}
	else
	{
		simmatrizaldusubSTSA(*w,fA-rnA,1.,A,fA,ldA,rnA);
		ldlsimmatrizfactsA(*w,fA-rnA,fA-rnA);
	}
	if (rnA<cA-rnA)
	{
		simmatrizaldusubN1N1TA(w2,rnA,1.,A,cA,ldA,rnA);
		simmatrizsumaraI(w2,rnA,rnA,1.);
		ldlsimmatrizfactsA(w2,rnA,rnA);
	}
	else
	{
		simmatrizaldusubNTNA(w2,cA-rnA,1.,A,cA,ldA,rnA);
		ldlsimmatrizfactsA(w2,cA-rnA,cA-rnA);
	}
	return 0;
}

/**Resuelve el sistema Au=v, dando solución de norma mínima de mínimos
 * cuadrados, a partir de una matriz ya factorizada y con subespacios
 * calculados. Requiere el cálculo de las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param u vector solución. puede ser igual a v
 * @param A matriz. Debe estar ya factorizada
 * @param fA número de filas de la matriz
 * @param cA número de columnas de la matriz
 * @param ldA dimensión fundamental de la matriz A
 * @param rnA rango de la matriz A
 * @param pivf matriz de cambio de filas de A (dimensión rnA)
 * @param pivC matriz de cambio de columnas de A (dimensión rnA)
 * @param w puntero a las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param v vector de independientes
 */
void vectorpinvlduAv(double *u, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long int *pivf, unsigned long int *pivc, double *w, double *v)
{
	double *w2;
	unsigned long int i;
	/*reordenamos*/
	if (u!=v)
	    memcpy(u,v, sizeof(double)*fA);
	
	vectorPsv(pivf, rnA, u);
	/*CASO r<f-r*/
	if (rnA<fA-rnA)
	{
		w2=&(w[(rnA*(rnA+1))/2]);
		/*obtenemos b_1-S_1b_2*/
		vectorsumaraldusubS1Av(u, -1.,A, fA, ldA, rnA, &(u[rnA]));
		vectorldlinvAv(u,w,rnA,rnA,u);
	}
	else
	{
		w2=&(w[((fA-rnA)*(fA-rnA+1))/2]);
		vectorsumaraldusubS1ATv(&(u[rnA]), 1.0, A, fA, ldA, rnA, u);
				
		/*cambiamos signo*/
		for (i=rnA;i<fA;i++)
			u[i]*=-1.;
		i=fA-rnA;
		if (i!=0)
		{
			vectorldlinvAv(&u[rnA],w,fA-rnA,fA-rnA,&u[rnA]);
			vectorsumaraldusubS1Av(u,1.,A,fA,ldA,rnA,&u[rnA]);
		}	
	}
	/*las premultiplicamos por L11m1*/
	vectorlduinvL11Av(u,A,ldA,rnA,u);
	/*multiplicamos por D11m1->z1*/
	vectorlduinvD1Av(u,A,ldA,rnA,u);
	/*Tenemos z*/
	vectorlduinvU11Av(u,A,ldA,rnA,u);
	/*CASO r<c-r. Nótese que es lo mismo que r<c/2*/
	if (rnA<cA-rnA)
	{
		vectorldlinvAv(u,w2,rnA,rnA,u);
		vectoraldusubN1ATv(&(u[rnA]), -1.,A,cA,ldA,rnA,u);
	}
	else
	{
		vectoraldusubN1ATv(&(u[rnA]), -1.,A,cA,ldA,rnA,u);	
		i=cA-rnA;
		if (i!=0)
		{
			vectorldlinvAv(&u[rnA],w2,i,i,&u[rnA]);		
			vectorsumaraldusubN1Av(u,1.,A,cA,ldA,rnA,&u[rnA]);
		}
	}	
	/*Cambio de columnas*/
	vectorPsTv(pivc,rnA,u);
	
}

/**Resuelve el sistema AC=B, dando solución de norma mínima de mínimos
 * cuadrados, a partir de una matriz ya factorizada y con subespacios
 * calculados. Requiere el cálculo de las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param C vector solución. Puede ser igual a B
 * @param ldC leading dimension de C.
 * @param A matriz. Debe estar ya factorizada
 * @param fA número de filas de la matriz
 * @param cA número de columnas de la matriz
 * @param ldA dimensión fundamental de la matriz A
 * @param rnA rango de la matriz A
 * @param pivf matriz de cambio de filas de A (dimensión rnA)
 * @param pivC matriz de cambio de columnas de A (dimensión rnA)
 * @param w puntero a las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param B matriz de independientes
 * @param cB número de columnas de B
 * @param ldB leading dimension de B
 */
void matrizpinvlduAB(double *C,unsigned long ldC, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long int *pivf, unsigned long int *pivc, double *w, double *B, unsigned long cB, unsigned long ldB)
{
	double *w2;
	unsigned long int i,j;
	/*reordenamos*/
	if (B!=C)
	{
		for (j=0;j<fA;j++)
			memcpy(&(C[ldC*j]),&(B[ldB*j]), sizeof(double)*cB);
	}
	matrizPsA(pivf, rnA, C, cB, ldC);
	/*CASO r<f-r*/
	if (rnA<fA-rnA)
	{
		w2=&(w[(rnA*(rnA+1))/2]);
		/*obtenemos b_1-S_1b_2*/
		matrizsumaraldusubS1AB(C, ldC, -1., A, fA, ldA, rnA, &(C[ldC*rnA]), cB, ldC);
		matrizldlinvAB(C, ldC, w, rnA, rnA, C, cB, ldC);
	}
	else
	{
		w2=&(w[((fA-rnA)*(fA-rnA+1))/2]);
		matrizsumaraldusubS1ATB(&(C[ldC*rnA]), ldC, 1., A, fA, ldA, rnA, C, cB, ldC);
		/*cambiamos signo*/
		matrizaA(&(C[rnA*ldC]),ldC,-1,&(C[rnA*ldC]),fA-rnA,cB,ldC);
		i=fA-rnA;
		if (i!=0)
		{
			matrizldlinvAB(&(C[rnA*ldC]), ldC, w, i, i, &(C[rnA*ldC]), cB, ldC);
			matrizsumaraldusubS1AB(C, ldC, 1., A, fA, ldA, rnA, &(C[ldC*rnA]), cB, ldC);
		}
	}
	/*las premultiplicamos por L11m1*/
	matrizlduinvL11AB(C, ldC, A, ldA, rnA, C, cB, ldC);
	/*multiplicamos por D11m1->z1*/
	matrizlduinvD1AB(C, ldC, A, ldA, rnA, C, cB, ldC);
	/*Tenemos z*/
	matrizlduinvU11AB(C, ldC, A, ldA, rnA, C, cB, ldC);
	/*CASO r<c-r. Nótese que es lo mismo que r<c/2*/
	if (rnA<cA-rnA)
	{		
		matrizldlinvAB(C, ldC, w2, rnA, rnA, C, cB, ldC);
		matrizaldusubN1ATB(&(C[rnA*ldC]), ldC, -1., A, cA, ldA, rnA, C, cB, ldC);
	}
	else
	{
		
		matrizaldusubN1ATB(&(C[rnA*ldC]), ldC, -1., A, cA, ldA, rnA, C, cB, ldC);
		i=cA-rnA;
		if (i!=0)
		{
	
			matrizldlinvAB(&(C[rnA*ldC]), ldC, w2, i, i, &(C[rnA*ldC]), cB, ldC);
			matrizsumaraldusubN1AB(C, ldC, 1., A, cA, ldA, rnA, &(C[rnA*ldC]), cB, ldC);
		}
	}
	/*Cambio de columnas*/
	matrizPsTA(pivc, rnA, C, cB, ldC);
}

/**Resuelve el sistema ATu=v, dando solución de norma mínima de mínimos
 * cuadrados, a partir de una matriz ya factorizada y con subespacios
 * calculados. Requiere el cálculo de las matrices simétricas realizado con
 * ldumatrizcalcularsimetricaspinv
 * @param u vector solución. puede ser igual a v
 * @param A matriz. Debe estar ya factorizada
 * @param fA número de filas de la matriz
 * @param cA número de columnas de la matriz
 * @param ldA dimensión fundamental de la matriz A
 * @param rnA rango de la matriz A
 * @param pivf matriz de cambio de filas de A (dimensión rnA)
 * @param pivC matriz de cambio de columnas de A (dimensión rnA)
 * @param w puntero a las matrices simétricasrealizado con
 * ldumatrizcalcularsimetricaspinv
 * @param v vector de independientes
 */
void vectorpinvlduATv(double *u, double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long int *pivf, unsigned long int *pivc, double *w, double *v)
{
	double *w2;
	unsigned long int i, dim;
	if (u!=v)
		memcpy(u,v,sizeof(double)*cA);/*Cambiado*/
	/*reordenamos*/
	vectorPsv(pivc, rnA, u);/*cambiado*/
	
	/*printf("P_c u\n");
	vectoratexto(stdout,u,cA,7);*/
	
	dim=rnA<(fA-rnA)?rnA:(fA-rnA);
	w2=&(w[dim*(dim+1)/2]);
	/*CASO r<c-r*/
	if (rnA<cA-rnA)
	{
		
		vectorsumaraldusubN1Av(u,-1.,A,cA,ldA,rnA,&u[rnA]);
		vectorldlinvAv(u,w2,rnA,rnA,u);
	}
	else
	{
		vectorsumaraldusubN1ATv(&(u[rnA]), 1.,A,cA,ldA,rnA,u);
		
		/*printf("(Pc*u)(rnA+1:cA)+N1T*(Pcu*u(1:rnA)\n");
		vectoratexto(stdout,(&u[rnA]),cA-rnA,7);*/
		
		/*cambiamos signo*/
		for (i=rnA;i<cA;i++)
			u[i]*=-1.;
		i=cA-rnA;
		if (i!=0)
		{
			vectorldlinvAv(&u[rnA],w2,i,i,&u[rnA]);
			vectorsumaraldusubN1Av(u,1.,A,cA,ldA,rnA,&u[rnA]);
		}	
	}
	
	/*printf("Solucion intermedia antes de U11\n");
	vectoratextooctave(stdout,u,rnA,7);*/
	
	vectorlduinvU11TAv(u,A,ldA,rnA,u);
	
	/*printf("Solucion intermedia despues de U11\n");
	vectoratextooctave(stdout,u,rnA,7);*/
	
	vectorlduinvD1Av(u,A,ldA,rnA,u);
	
	/*printf("Solucion intermedia antes de L11\n");
	vectoratextooctave(stdout,u,rnA,7);*/
	
	vectorlduinvL11TAv(u,A,ldA,rnA,u);
	
	/*printf("Solucion intermedia despues de L11\n");
	vectoratextooctave(stdout,u,rnA,7);*/
	
	
	/*CASO r<c-r. Nótese que es lo mismo que r<c/2*/
	if (rnA<fA-rnA)
	{
		vectorldlinvAv(u,w,rnA,rnA,u);
		/*ponemos a cero lo que hay detras de b*/
		for (i=rnA;i<fA;i++)
			u[i]=0.;
		vectorsumaraldusubS1ATv(&(u[rnA]), -1.,A, fA, ldA, rnA, u);
	}
	else
	{
		/*ponemos a cero lo que hay detras de b*/
		for (i=rnA;i<fA;i++)
			u[i]=0.;
		vectorsumaraldusubS1ATv(&(u[rnA]), -1.,A, fA, ldA, rnA, u);
		
		/*printf("-S1T*st1\n");
		vectoratexto(stdout,&u[rnA],fA-rnA,8);*/
		
		/*Reservamos espacio para NTN*/
		i=fA-rnA;
		if (i!=0)
		{
			vectorldlinvAv(&u[rnA],w,i,i,&u[rnA]);
			
			/*printf("(S1T*S1+I)\(-S1T*st1)\n");
			vectoratexto(stdout,&u[rnA],fA-rnA,8);*/
			
			vectorsumaraldusubS1Av(u, 1.,A, fA, ldA, rnA, &u[rnA]);
			
			/*printf("u+S1*((S1T*S1+I)\(-S1T*st1))\n");
			vectoratexto(stdout,u,fA,8);*/
		}

	}
	/*Cambio de columnas*/
	vectorPsTv(pivf,rnA,u);/*Cambiado*/
}

/*Calcula la pseudoinversa del subespacio nulo de A por el vector v y lo almacena
 en &(V[rnA]). Esto es equivalente a proyectar el vector v en el subespacio. La
 matriz A debe estar factorizada por ldu con los subespacios calculados y con
 las matrices simetricas para calculos de minimos cuadrados de minima norma
 calculadas.
 @param A puntero a la matriz factorizada
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio
 * reservado de al menos el rango de la matriz (en caso de no saber a priori,
 * vale con el menor de fA y fC)
 @param w: puntero a espacio de trabajo. Es para almacenar las matrices
 * simetricas. Debe tener reservado al menos dim*(dim+1)/2+dim2*(dim2+1)/2
 * siendo dim1=min(rnA,cA-rnA) y dim2=min(rnA,fA-rnA). En salida tiene las
 * matrices simétricas necesarias para la solución de mínimos cuadrados de
 * mínima norma
 @param v vector que se quiere reducir
 */
void vectorreduccionldupinvsubNAv(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long rnA, unsigned long int *pivc, double *w, double *v)
{
	unsigned long dim;
	double *w2;
	dim=rnA<(fA-rnA)?rnA:(fA-rnA);
	w2=&(w[dim*(dim+1)/2]);	
	if (2*rnA>=cA)
	{
		vectorreduccionldusubNTAv(A, cA, ldA, rnA, pivc, v);
		vectorldlinvAv(&(v[rnA]),w2,cA-rnA,cA-rnA,&v[rnA]);
	}
	else
	{
		vectorPsv(pivc, rnA, v);
		vectorsumaraldusubN1Av(v, -1., A, cA, ldA, rnA, &(v[rnA]));
		vectorldlinvAv(v,w2,rnA,rnA,v);
		vectorsumaraldusubN1ATv( &(v[rnA]), 1., A, cA, ldA, rnA, v);
	}
}


/**Realiza la factorización ldu de la matriz A con pivotaje completo, cálculo de
 * subespacios y de matrices simétricas (factorizadas) para el cálculo de solución de mínimos
 * cuadrados de mínima norma. Todo lo posible se hace dentro de la matriz A.
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivf puntero a los cambios de filas. Debe apuntar a un espacio reservado
 * de al menos el rango de la matriz (en caso de no saber a priori, vale con el
 * menor de fA y fC)
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio
 * reservado de al menos el rango de la matriz (en caso de no saber a priori,
 * vale con el menor de fA y fC)
 @param w: puntero a espacio de trabajo. Es para almacenar las matrices
 * simetricas. Debe tener reservado al menos dim*(dim+1)/2+dim2*(dim2+1)/2
 * siendo dim1=min(rnA,cA-rnA) y dim2=min(rnA,fA-rnA). En salida tiene las
 * matrices simétricas necesarias para la solución de mínimos cuadrados de
 * mínima norma
 @param eps valor a considerar cero numérico
*/
void ldumatrizpinvptA(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *rnA, unsigned long *pivf, unsigned long *pivc, double *w, double eps)
{
	double *w2;
	ldumatrizfactptA(A,fA,cA,ldA,rnA,pivf,pivc,eps);
	ldumatrizobtenersubespacionulo(A,cA,ldA,*rnA);
	ldumatrizobtenersubespacionuloizquierdo(A,fA,ldA,*rnA);

	if (*rnA<fA-*rnA)
	{
		simmatrizaldusubS1S1TA(w,*rnA,1.,A,fA,ldA,*rnA);
		simmatrizsumaraI(w,*rnA,*rnA,1.);
		ldlsimmatrizfactsA(w,*rnA,*rnA);
		w2=&(w[*rnA*(*rnA+1)/2]);
	}
	else
	{
		simmatrizaldusubSTSA(w,fA-*rnA,1.,A,fA,ldA,*rnA);
		ldlsimmatrizfactsA(w,fA-*rnA,fA-*rnA);
		w2=&(w[(fA-*rnA)*(fA-*rnA+1)/2]);
	}
	if (*rnA<cA-*rnA)
	{
		simmatrizaldusubN1N1TA(w2,*rnA,1.,A,cA,ldA,*rnA);
		simmatrizsumaraI(w2,*rnA,*rnA,1.);
		ldlsimmatrizfactsA(w2,*rnA,*rnA);
	}
	else
	{
		simmatrizaldusubNTNA(w2,cA-*rnA,1.,A,cA,ldA,*rnA);
		ldlsimmatrizfactsA(w2,cA-*rnA,cA-*rnA);
	}

}

/**Realiza la factorización ldu de la matriz A con pivotaje de Rook, cálculo de
 * subespacios y de matrices simétricas (factorizadas) para el cálculo de solución de mínimos
 * cuadrados de mínima norma. Todo lo posible se hace dentro de la matriz A.
 @param A puntero a la matriz
 @param fA numero de filas de la matriz
 @param cA numero de columnas de la matriz
 @param ldA dimensión fundamental de A
 @param rnA en salida, rango de la matriz
 @param pivf puntero a los cambios de filas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param pivc puntero a los cambios de columnas. Debe apuntar a un espacio reservado de al menos el rango de la matriz (en caso de no saber a priori, vale con el menor de fA y fC)
 @param w puntero a espacio de trabajo. Es para almacenar las matrices simetricas.
 * Debe tener reservado al menos dim*(dim+1)/2+dim2*(dim2+1)/2 siendo dim1=min(rnA,cA-rnA) y dim2=min(rnA,fA-rnA)
 @param eps valor a considerar como cero numérico
 */
void ldumatrizpinvrookA(double *A, unsigned long fA, unsigned long cA, unsigned long ldA, unsigned long *rnA, unsigned long *pivf, unsigned long *pivc, double *w, double eps)
{
	double *w2;
	ldumatrizfactrookA(A,fA,cA,ldA,rnA,pivf,pivc,eps);
	/*printf("rango de A:%ld\n",*rnA);*/
	ldumatrizobtenersubespacionulo(A,cA,ldA,*rnA);
	/*printf("rango de A:%ld\n",*rnA);*/
	ldumatrizobtenersubespacionuloizquierdo(A,fA,ldA,*rnA);
	/*printf("rango de A:%ld\n",*rnA);*/
	if (*rnA<fA-*rnA)
	{
		simmatrizaldusubS1S1TA(w,*rnA,1.,A,fA,ldA,*rnA);
		simmatrizsumaraI(w,*rnA,*rnA,1.);
		ldlsimmatrizfactsA(w,*rnA,*rnA);
		w2=&(w[*rnA*(*rnA+1)/2]);
	}
	else
	{
		simmatrizaldusubSTSA(w,fA-*rnA,1.,A,fA,ldA,*rnA);		
		ldlsimmatrizfactsA(w,fA-*rnA,fA-*rnA);
		w2=&(w[(fA-*rnA)*(fA-*rnA+1)/2]);
	}
	if (*rnA<cA-*rnA)
	{
		simmatrizaldusubN1N1TA(w2,*rnA,1.,A,cA,ldA,*rnA);
		simmatrizsumaraI(w2,*rnA,*rnA,1.);
		ldlsimmatrizfactsA(w2,*rnA,*rnA);
	}
	else
	{
		simmatrizaldusubNTNA(w2,cA-*rnA,1.,A,cA,ldA,*rnA);
		ldlsimmatrizfactsA(w2,cA-*rnA,cA-*rnA);
	}
}
