// simmatricesLDLBase.c
//
// Copyright (C) 2011 - Igor Fernández de Bustos & Ibai Coria
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#include "alglin.h"
#include "math.h"

/*Macro para obtener el indice en el array del elemento fila,columna*/
#define INDICEARRAYDEFILACOLUMNA(t,f,c) (f)*(t)-((f)*((f)+1))/2+(c)

/*Intercambia las filas/columnas i y j
 @param sA puntero a matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de la matriz
 @param i indice de la primera fila a rotar
 @param j indice de la segunda fila a rotar*/
void simmmatrizintercambiarfilacolummna(double *sA, unsigned long esA, unsigned long ldsA, unsigned long i, unsigned long j)
{
	unsigned long k;
	double sw;
	if (i==j)
		return;
	if (i<j)
	{
		for(k=0; k<i; k++)
		{
			sw=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)]=sw;
		}
		sw=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, j)];
		sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, j)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];
		sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)]=sw;
		k++;
		for(; k<j; k++)
		{
			sw=sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, k)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, k)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)]=sw;
		}
		k++;
	}
	else
	{
		for(k=0; k<j; k++)
		{
			sw=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)]=sw;
		}
		sw=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, j)];
		sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, j)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];
		sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)]=sw;
		k++;
		for(; k<i; k++)
		{
			sw=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)]=sw;
		}
		k++;
	}
	for(; k<esA; k++)
	{
		sw=sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, k)];
		sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, k)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)];
		sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)]=sw;
	}
}

/*Hace una rotación a las filas/columnas i, j, con ángulo igual a 2 arctan(t)
 @param sA puntero a matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de la matriz
 @param i indice de la primera fila a rotar
 @param j indice de la segunda fila a rotar
 @param t tangente de la mitad del ángulo de rotación.*/
void simmatrizrotacion(double *sA, unsigned long esA, unsigned long ldsA, unsigned long i, unsigned long j, double t)
{
	unsigned long int k;
	double e0, e1, s, r;
	if (t==0.)
		return;
	e0=sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];
	e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, j)];
	s=sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, j)];
	r=1./(t*t+1);
	
	/*Ponemos directamente el cero del elemento i,j*/
	sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, j)]=0.;
	/*Si queremos calcular el elemento i,j, incluir la siguiente ecuacion*/
	/*sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, j)]=r*((1-t*t)*s+t*(e0-e1));*/
	sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)]=r*(e0+t*(t*e1-2.*s));
	sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, j)]=r*(e1+t*(t*e0+2.*s));


	if (t>1.)
	{
		e0=sqrt(r);
		r=1./(1./t+e0);
		s=1./e0;
	}
	else
	{
		e0=sqrt(t*t+1.0); /*t<=1, por lo que esto es seguro*/
		r=t/(e0+1.0);
		s=t/e0;
	}



	if (i<j)
	{
		for(k=0; k<i; k++)
		{
			e0=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)];
			e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)]-=s*(e1+r*e0);
			e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)]+=s*(e0-r*e1);
		}
		k++;
		for(; k<j; k++)
		{
			e0=sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, k)];
			e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, k)]-=s*(e1+r*e0);
			e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)]+=s*(e0-r*e1);
		}
		k++;
	}
	else
	{
		for(k=0; k<j; k++)
		{
			e0=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)];
			e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)]-=s*(e1+r*e0);
			e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, j)]+=s*(e0-r*e1);
		}
		k++;
		for(; k<i; k++)
		{
			e0=sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)];
			e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)];
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, k, i)]-=s*(e1+r*e0);
			e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)]+=s*(e0-r*e1);
		}
		k++;
	}
	for(; k<esA; k++)
	{
		e0=sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, k)];
		e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)];
		sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, k)]-=s*(e1+r*e0);
		e1=sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)]+=s*(e0-r*e1);
	}
	k++;
}

/**Hace la etapa de reducción de la factorización
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de la matriz*/
void ldlsimmatrizetapareduccionsA(double *sA, unsigned long esA, unsigned long ldsA)
{
	double invd, d;
	unsigned long int j, k;
	invd=1.0/sA[0];
	for(j=1; j<esA; j++)
	{
		/*TODO:if d!=0 y probar eficacia*/
		d=sA[j]*invd;
		sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, j)]-=sA[j]*d;
		sA[j]=d;
		for(k=j+1; k<esA; k++)
		{
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, k)]-=sA[k]*d;
		}
	}
}

/**Hace la factorización ldl sin pivotamiento. Interesante para sistemas positivos definidos.
 @param sA puntero a la matriz simátrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de la matriz simétrica*/
void ldlsimmatrizfactsA(double *sA, unsigned long esA, unsigned long ldsA)
{
	double *matrizactual;
	unsigned long int tamanoactual, ldactual;
	if (esA==0)
		return;
	matrizactual=sA;
	ldactual=ldsA;
	tamanoactual=esA;
	do
	{
		ldlsimmatrizetapareduccionsA(matrizactual, tamanoactual, ldactual);
		matrizactual+=ldactual;
		tamanoactual--;
		ldactual--;
	}
	while (tamanoactual>0);
}

/**Multiplica la matriz diagonal D en la matriz sA factorizada ldl por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlDsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v)
{
	unsigned long int i;
	for(i=0; i<esA; i++)
	{
		u[i]=v[i]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];
	}
}

/**Multiplica la matriz B por la matriz diagonal D en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlDsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB)
{
	if (cB*esA==0)
		return;
	unsigned long int i, k;
	for(i=0; i<esA; i++)
	{
		for(k=0; k<cB; k++)
			C[i*ldC+k]=B[i*ldB+k]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];

	}
}

/**Multiplica la inversa de la matriz diagonal D en la matriz sA factorizada ldl por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlinvDsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v)
{
	unsigned long int i;
	for(i=0; i<esA; i++)
	{
		u[i]=v[i]/sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];
	}
}

/**Multiplica la matriz B por el inverso de la matriz diagonal D en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlinvDsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long int i, k;
	if (cB*esA==0)
		return;
	for(i=0; i<esA; i++)
	{
		for(k=0; k<cB; k++)
			C[i*ldC+k]=B[i*ldB+k]/sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];
	}
}

/**Multiplica la matriz L en la matriz sA factorizada ldl por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA número de elementos de la matriz
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlLsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v)
{
	unsigned long j, k;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	if (esA==0)
		return;
	for(k=esA; k>0; k--)
	{
		u[k-1]=v[k-1];
		for(j=k; j<esA; j++)
			u[j]+=u[k-1]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, k-1, j)];
	}
}

/**Multiplica la matriz B por la matriz L en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlLsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i, j, k;
	/*para poder hacerlo en el mismo vector sin problemas, lo hacemos de abajo a arriba*/
	if (esA*cB==0)
		return;
	for(k=esA; k>0; k--)
	{
		for(i=0; i<cB; i++)
			C[ldC*(k-1)+i]=B[ldB*(k-1)+i];
		for(j=k; j<esA; j++)
			for(i=0; i<cB; i++)
				C[ldC*j+i]+=C[ldC*(k-1)+i]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, k-1, j)];
	}
}

/**Multiplica la inversa de la matriz L en la matriz sA factorizada ldl por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlinvLsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v)
{
	unsigned long int i, j;
	if (esA==0)
		return;
	u[0]=v[0];
	for(i=1; i<esA; i++)
		u[i]=v[i]-u[0]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, 0, i)];
	for(j=1; j<esA; j++)
	{
		for(i=j+1; i<esA; i++)
			u[i]-=u[j]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, i)];
	}
}

/**Multiplica la matriz B por el inverso de la matriz L en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlinvLsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long int i, j, k;
	if (esA*cB==0)
		return;
	for(k=0; k<cB; k++)
		C[k]=B[k];
	for(i=1; i<esA; i++)
		for(k=0; k<cB; k++)
			C[ldC*i+k]=B[ldB*i+k]-C[k]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, 0, i)];
	for(j=1; j<esA; j++)
	{
		for(i=j+1; i<esA; i++)
			for(k=0; k<cB; k++)
				C[ldC*i+k]-=C[ldC*j+k]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, j, i)];
	}
}

/**Multiplica la matriz L en la matriz sA factorizada ldl traspuesta por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlLTsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v)
{
	unsigned long i, j;
	for(i=0; i<esA; i++)
	{
		u[i]=v[i];
		for(j=i+1; j<esA; j++)
			u[i]+=v[j]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, j)];
	}
}

/**Multiplica la matriz B por la matriz L traspuesta en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlLTsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long i, j, k;
	if (esA*cB==0)
		return;
	for(i=0; i<esA; i++)
	{
		for(k=0; k<cB; k++)
			C[ldC*i+k]=B[ldB*i+k];
		for(j=i+1; j<esA; j++)
			for(k=0; k<cB; k++)
				C[ldC*i+k]+=B[ldB*j+k]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, j)];
	}
}

/**Multiplica la inversa de la matriz L en la matriz sA factorizada ldl traspuesta por el vector v y lo almacena en u.
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param v puntero al vector v*/
void vectorldlinvLTsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v)
{
	unsigned long int i, j;
	if (esA==0)
		return;
	for(j=esA; j>0; j--)
	{
		u[j-1]=v[j-1];
		for(i=j; i<esA; i++)
			u[j-1]-=v[i]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, j-1, i)];
	}
}

/**Multiplica la matriz B por el inverso de la matriz L traspuesta en la matriz sA factorizada ldl y lo almacena en C.
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlinvLTsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long int i, j, k;
	if (esA*cB==0)
		return;
	for(j=esA; j>0; j--)
	{
		for(k=0; k<cB; k++)
			C[ldC*(j-1)+k]=B[ldB*(j-1)+k];
		for(i=j; i<esA; i++)
			for(k=0; k<cB; k++)
				C[ldC*(j-1)+k]-=B[ldB*i+k]*sA[INDICEARRAYDEFILACOLUMNA(ldsA, j-1, i)];
	}
}

/**Calcula en u la solución de sA^{-1}v, a partir de la matriz A factorizada ldl
 @param u vector solución. puede ser igual a v
 @param sA matriz simétrica factorizada
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de sA
 @param v vector de independientes.*/
void vectorldlinvAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, double *v)
{
	vectorldlinvLsAv(u, sA, esA, ldsA, v);
	vectorldlinvDsAv(u, sA, esA, ldsA, v);
	vectorldlinvLTsAv(u, sA, esA, ldsA, v);
}

/**Calcula en C la solución de sA^{-1}B, a partir de la matriz A factorizada ldl
 @param C puntero a la matriz resultado. puede ser igual a B
 @param ldC dimensión fundamental de la matriz C
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param B puntero a la matriz B
 @param cB número de columnas de B
 @param ldB dimension fundamental de B*/
void matrizldlinvAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, double *B, unsigned long cB, unsigned long ldB)
{
	matrizldlinvLsAB(C, ldC, sA, esA, ldsA, B, cB, ldB);
	matrizldlinvDsAB(C, ldC, sA, esA, ldsA, B, cB, ldB);
	matrizldlinvLTsAB(C, ldC, sA, esA, ldsA, B, cB, ldB);
}

/**Hace la etapa de reducción de la factorización para la factorización ldl
 * generalizada. Es decir, el elemento 0,1 es 0 pero no está almacenado como tal
 * se utiliza tanto en la factorizacion que aplica los giros en la submatriz como la que aplica los giros en la matriz completa
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de la matriz*/
void ldlgensimmatrizetapareduccionsA(double *sA, unsigned long esA, unsigned long ldsA)
{
	unsigned long int i, j;
	double invd, d;
	invd=1./sA[0];
	for(i=2; i<esA; i++)
	{
		d=sA[i]*invd;
		for(j=i; j<esA; j++)
			sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, j)]-=d*sA[j];
		sA[i]=d;
	}
}

/*Realiza una iteracion (recursiva) de la factorizacion LDL con pivotación generalizada. Para ello busca el elemento de mayor valor absoluto
y a partir de el realiza un intercambio/rotación entre las filas/columnas de la submatriz de manera que si el elemento es el ii, se intercambia
la fila/columna i por la 0, se rota con la 0 y se reduce. Si el elemento es ij, se hace una rotación en la que la fila de mayor valor absoluto en la diagonal
principal de entre i y j va a la 0 y la otra a la 1, tras girarse para que el elemento 01 sea igual a cero. Realmente en ese elemento se almacena
la tangente del giro realizado.
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param piv array de pivotes. Tiene por dimension la dimensión de la submatriz multiplicada por dos. Si el elemento de mayor valor absoluto está en
la diagonal principal, piv[0]=i, piv[1]=1, si no, piv[0]=i, piv[1]=j, donde i es la fila del pivote ij en la que aparece el elemento en la diagonal principal
de mayor valor absoluto.
@param tol tolerancia para definir el cero numérico que determina el fin de la factorización
@return 0 si ok, 1 si se ha acabado (es todo ceros)
 */
int ldlgensimmatriziteracionsAgirosubmatriz(double *sA, unsigned long esA, unsigned long ldsA, unsigned long int *piv, double tol)
{
	double t, aki, akj, aij, temp;
	simmmatrizintercambiarfilacolummna(sA, esA, ldsA, 0, piv[0]);
	simmmatrizintercambiarfilacolummna(sA, esA, ldsA, 1, piv[1]+1);

	/*calculo de  t!!*/
	aki=sA[INDICEARRAYDEFILACOLUMNA(ldsA, 0, 0)]; /*aki=aii*/
	akj=sA[INDICEARRAYDEFILACOLUMNA(ldsA, 1, 1)]; /*akj=ajj*/
	aij=sA[INDICEARRAYDEFILACOLUMNA(ldsA, 0, 1)];

	/*Es posible que akj=aki y aij=0, en estas condiciones el giro queda sin definir, y, además, no tiene sentido*/
	temp=0.5*(akj-aki)/aij;
	if (fabs(temp)<1.)
	{
		if (temp>0.)
			t=1./(temp+sqrt(temp*temp+1.));
		else
		{
			if (temp<0.)
				t= -1./(sqrt(temp*temp+1.)-temp);
			else /*parece tonteria, pero el caso de temp=0 es peligroso, porque el signo tiene que venir determinado por que el valor gordo tiene que quedar en el primer elemento*/
			{
				if (aki<0.)
					t=1.;
				else
					t= -1.;
			}
		}
	}
	else
	{
		temp=2.*aij/(akj-aki);
		if (temp>0.)
			t=temp/(1.+sqrt(temp*temp+1.));
		else
		{
			if (temp<0.)
			{
				t=temp/(1.+sqrt(temp*temp+1.));
			}
			else
			{
				t=0.;
			}
		}
	}
	simmatrizrotacion(sA, esA, ldsA, 0, 1, t);
	/*printf("Matriz girada:\n");
	simmatrizatexto(stdout,sA,esA,ldsA,5);*/
	ldlgensimmatrizetapareduccionsA(sA, esA, ldsA);
	sA[1]=t;
	/*printf("Matriz reducida:\n");
	simmatrizatexto(stdout,sA,esA,ldsA,5);*/

	return 0;
}

/**Realiza una factorización completa de la matriz simétrica por el método de rotación (algoritmo recursivo) con pivotaje completo
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param rnsA en salida, rango de la matriz A
@param piv array de pivotes. Tiene por dimension la dimensión de la submatriz multiplicada por dos. Son cambios sucesivos, es decir,
la fila/columna a cambiar se permuta por la que esta piv veces a la derecha/abajo. Si estamos en la columna i y el pivote es j se
cambia la i por la j+i. Si el elemento de mayor valor absoluto está en la diagonal principal, piv[0]=i, piv[1]=0, si no, piv[0]=i,
piv[1]=j, donde i es la fila del pivote ij en la que aparece el elemento en la diagonal principal
de mayor valor absoluto.
@param tol tolerancia para definir el cero numérico que determina el fin de la factorización
 */
void ldlgensimmatrizfactptsAgirosubmatriz(double *sA, unsigned long esA, unsigned long ldsA, unsigned long *rnsA, unsigned long int *piv, double tol)
{
	unsigned long int i, f, c;
	double pivote;
	if (esA<=1)
	{
		*rnsA=0;
		return;
	}
	for(i=0; i<esA-1; i++)
	{
		pivote=pivotajecompletosimmatriz(&sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)], esA-i, ldsA-i, &f, &c);
		if (pivote<tol)
		{
			*rnsA=i;
			return;
		}
		/*Buscamos que orden nos va mejor*/
		if (f==c)
		{
			piv[2*i]=f;
			piv[2*i+1]=0; /*si el pivote esta en
					 * la diagonal ppal, la
					 * segunda fila/columna no se mueve*/
		}
		else
		{
			if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, f+i, f+i)])>=fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, c+i, c+i)]))
			{
				piv[2*i]=f;
				/*esta mierda de abajo es por los cambios sucesivos. Si el segundo cambio se hace sobre la primera,
				 * como la hemos cambiado antes, es cagadita.*/
				if (c==0)
					piv[2*i+1]=f-1;
				else
					piv[2*i+1]=c-1;
			}
			else
			{
				piv[2*i]=c;
				if (f==0)
					piv[2*i+1]=c-1;
				else
					piv[2*i+1]=f-1;
			}
		}
		ldlgensimmatriziteracionsAgirosubmatriz(&(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)]), esA-i, ldsA-i, &(piv[2*i]), tol);
	}
	if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)])>tol)
		i++;
	*rnsA=i;
	return;
}

/**Etapa de resolución del problema C=L^-1 G P A, solo procesa una fila, de manera que el problema vuelve a ser el mismo luego, pero de distinto tamaño
 @param sA puntero a la submatriz simétrica factorizada con ldlgen correspondiente a la fila a procesar
 @param esA tamaño de la matriz
 @param A matriz a multiplicar por L-1
 @param cA numero de columnas de la matriz a multiplicar
 @param piv puntero a los pivotes*/
void matrizldlgengirosubmatrizetaparesolucioninvLB(double *sA, unsigned long esA, double *B, unsigned long int cB, unsigned long ldB, unsigned long int *piv)
{
	unsigned long int i, j;
	double t, c, s;
	/*Intercambio en v*/
	matrizPsA(piv, 2, B, cB, ldB);

	/*giro en e*/
	t=sA[1];
	/*siempre es menor que 1, por lo que no hay que complicarse*/
	c=1./sqrt(t*t+1.);
	s=c*t;
	c=s/(1.+c); /*c=s/1+c*/

	for(i=0; i<cB; i++)
	{
		t=B[i];
		B[i]=t-s*(B[ldB+i]+c*t);
		B[ldB+i]=B[ldB+i]+s*(t-c*B[ldB+i]);
	}

	/*resolucion*/
	for(j=0; j<cB; j++)
	{
		t=B[j];
		for(i=2; i<esA; i++) /*el 1 es cero en la matriz, por lo que no modifica nada*/
			B[i*ldB+j]-=t*sA[i];
	}
}

/**Resolución del problema E=L^-1 E mediante pivotamiento girado completo, con E matriz (incluye giros y cambios de filas/columnas)
 @param sA puntero a la matriz simétrica factorizada con ldlgen
 @param ldsA dimensión fundamental de la matriz
 @param rsA rango de la matriz simétrica
 @param B matriz a multiplicar
 @param cB numero de columnas de B
 @param ldB Dimension pricipal de B
 @param piv puntero a los pivotes*/
void matrizldlgengirosubmatrizinvL11sAB(double *sA, unsigned long int ldsA, unsigned long rsA, double *B, unsigned long cB, unsigned long int ldB, unsigned long int *piv)
{
	unsigned long int i;
	for(i=0; i<rsA-1; i++)
		matrizldlgengirosubmatrizetaparesolucioninvLB(&(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)]), rsA-i, &(B[ldB*i]), cB, ldB, &piv[2*i]);
}

/**Etapa de resolución del problema b=Lt^-1 b, solo procesa una fila, de manera que el problema vuelve a ser el mismo luego
 @param matriz matriz factorizada de forma LDL generalizada
 @param tam tamano de la matriz (y del vector)
 @param v vector a procesar
 @param piv puntero a los pivotes
 @param n etapa que se quiere realizar. Va de tam-2 hasta 0 para hacer la resolucion completa.*/
void matrizldlgengirosubmatrizetaparesolucioninvLTv(double *sA, unsigned long int esA, unsigned long ldsA, double *v, unsigned long int *piv, unsigned long int n)
{
	unsigned long int i, e;
	double t, c, s;
	e=INDICEARRAYDEFILACOLUMNA(ldsA, n, n+2);
	for(i=n+2; i<esA; i++)/*el elemento n+1 es cero patatero*/
	{
		v[n]-=v[i]*sA[e];
		e+=1;
	}
	/*giro en v---x*/
	t=sA[INDICEARRAYDEFILACOLUMNA(ldsA, n, n+1)];
	/*t siempre es menor que 1, no hay que complicarse*/
	c=1./sqrt(t*t+1.);
	s=c*t;
	t=v[n];
	/*vector[n]=t*c+s*vector[n+1];
	vector[n+1]=-t*s+c*vector[n+1];*/
	c=s/(1.+c); /*c=s/1+c*/
	v[n]=t+s*(v[n+1]-c*t);
	v[n+1]=v[n+1]-s*(t+c*v[n+1]);

	/*intercambio elementos en v---x*/
	vectorPsTv(&(piv[2*n]), 2, &(v[n]));
}

/**Resolución del problema v=L11t^-1 v
 @param sA matriz factorizada de forma LDL
 @param ldsA dimension ppal de A.
 @param rsA rango de la matriz (y tamaño del vector)
 @param v vector
 @param piv puntero a los pivotes*/
void matrizldlgengirosubmatrizinvL11TsAv(double *sA, unsigned long int ldsA, unsigned long int rsA, double *v, unsigned long int *piv)
{
	unsigned long int i;
	if (rsA<2)
		return;
	for(i=rsA-2; i>0; i--)
	{
		matrizldlgengirosubmatrizetaparesolucioninvLTv(sA, rsA, ldsA, v, piv, i);
	}
	matrizldlgengirosubmatrizetaparesolucioninvLTv(sA, rsA, ldsA, v, piv, i);
}

/**Multiplica la pseudoinversa de la matriz diagonal D en la matriz sA
 * factorizada ldl generalizada por la matriz B y lo almacena en C. C puede
ser la misma matriz que B.
 @param C matriz resultado
 @param ldC dimension principal de C
 @param sA puntero a la matriz simétrica factorizada con ldlgen
 @param ldsA dimensión fundamental de la matriz
 @param rsA rango de la matriz simétrica
 @param B matriz a multiplicar
 @param cB numero de columnas de B
 @param ldB Dimension pricipal de B
 */
void matrizldlgengirosubmatrizpinvDsAB(double *C, unsigned long ldC, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rsA, double *B, unsigned long cB, unsigned long ldB)
{
	unsigned long int i, j;
	for(i=0; i<rsA; i++)
	{
		for(j=0; j<cB; j++)
			C[i*ldC+j]=B[i*ldC+j]/sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];
	}
	for(; i<esA; i++)
	{
		for(j=0; j<cB; j++)
			C[i*ldC+j]=0.;
	}
}

/**Multiplica la pseudoinversa de la matriz diagonal D en la matriz sA
 * factorizada ldl generalizada por el vector v y lo almacena en el
 * vector u. u puede ser el mismo que v
 @param u puntero al vector resultado. puede ser igual a v
 @param sA puntero a la matriz simétrica
 @param esA dimensión de la matriz (numero de filas/columnas)
 @param ldsA dimensión fundamental de A
 @param rsA rango de la matriz simétrica
 @param v puntero al vector v*/
void vectorldlgengirosubmatrizpinvDsAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rsA, double *v)
{
	unsigned long int i;
	for(i=0; i<rsA; i++)
	{
		u[i]=v[i]/sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)];
	}
	for(; i<esA; i++)
	{
		u[i]=0.;
	}
}


/**********IBAI*****/
/*Todas las funciones que hay de aqui en adelante son para resolver sistema con 
la factorizacion ldlgen*/

#define INDICEFILACOLUMNAL11(t,f,c) (f)*(t)-((f)*((f)+1))/2+(c)
#define INDICEFILACOLUMNAL12(t,r,f,c) (f)*(t)-((f)*((f)+1))/2+(r)+(c)

/**Busca el primer pivote dominante en fila y columna
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
 @param f parámetro de vuelta. Fila donde está el máximo
 @param c parámetro de vuelta. Columna donde está el máximo
 @param tol valor del cero numerico
 @return el valor absoluto del pivote
 */
double pivotajerooksimmatriz(double *sA, unsigned long esA, unsigned long ldsA, unsigned long *f, unsigned long *c,unsigned long *f2, double tol)
{
	double max;
	unsigned long i, j, indice,anterior, cmax, fmax, salir;
	/*primer paso, buscamos una fila que tenga un candidato a pivote*/
	max=0.;
	fmax=0;
	cmax=0;
	anterior=0;
	for(i=0; i<esA; i++) /*Vamos a buscar un elemento (siguiendo el procedimiento del PC) que tenga un valor por encima de la tolerancia establecida*/
	{
		for(j=i; j<esA; j++)
		{
			if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, j)])>max)
			{
				max=fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, j)]);
				fmax=i;
				cmax=j;
			}
		}
		if (max>tol) /*Si el elemento mas grande de la fila esta por encima de la tolerancia terminaremos el ciclo*/
		{
			break;
		}
	}
	if (max<=tol)
	{
		*f=fmax;
		*c=cmax;
		*f2=anterior;  /*Da igual que valor devolver, no se va a utilizar*/
		return max;
	}
		
	/*Guardamos el indice anterior.
	 Este parametro solo se utiliza si el pivote queda en la diagonal principal.
	 Nos da una fila con todos los valores menores que el pivote*/
	if (fmax==0) 
	{
		if (fmax==cmax) /*En caso de que hayamos obtenido como pivote el elemento (0,0), estudiamos la segunda fila*/
		{
			for(j=1; j<esA; j++)
			{
				if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, 1, j)])>max)
				{
					max=fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, 1, j)]);
					fmax=1;
					cmax=j;
				}
			}
			if (fmax==0) /*Si el pivote sigue siendo el elemento (0,0), nos quedamos con ese*/
			{
				*f=fmax;
				*c=cmax;
				*f2=1;
				return max;
			}
		}
	}
	else
		anterior=fmax-1; /*Por si acaso fmax=cmax*/
	
	/*Ahora comprobar que se cumple rook y seguir*/
	indice=fmax;
	do
	{
		if (fmax==cmax)
			break;
		salir=1;
		if (cmax==indice)
		{
			indice=fmax;
			anterior=cmax;/*Lo guardamos por por si acaso, ya que quiza encontremos el pivote en la diagonal*/
		}
		else
		{
			indice=cmax;
			anterior=fmax;
		}

		for(i=0; i<indice; i++)
		{
			if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, indice)])>max)
			{
				max=fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, indice)]);
				fmax=i;
				cmax=indice;
				salir=0;
			}
		}
		for(i=indice; i<esA; i++)
		{
			if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, indice, i)])>max)
			{
				max=fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, indice, i)]);
				fmax=indice;
				cmax=i;
				salir=0;
			}
		}
		if (salir) /*si hemos llegado hasta el final y no hemos encontrado nada mayor, fiesta*/
		{
			anterior=indice;/*Esto nos dara la ultima fila estudiada, aunque no lo usamos para nada*/
			break;
		}
	}
	while (1); /*poner una medida de seguridad?*/
	*f=fmax;
	*c=cmax;
	*f2=anterior; 

	return max;
}

/*Realiza una iteracion (esta ya no es recursiva) de la factorizacion LDL con pivotación generalizada. Para ello busca el elemento de mayor valor absoluto
y a partir de el realiza un intercambio/rotación entre las filas/columnas de la matriz de manera que si el elemento es el ii, se intercambia
la fila/columna i por la et +1, se rota con la et y se reduce. Si el elemento es ij, se hace una rotación en la que la fila de mayor valor absoluto en la diagonal
principal de entre i y j va a la et y la otra a la et+1, tras girarse para que el elemento et,et+1 sea igual a cero. Los giros se almacenan en un vector de giros.
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param piv array de pivotes. Tiene por dimension la dimensión de la submatriz multiplicada por dos. Si el elemento de mayor valor absoluto está en
la diagonal principal, piv[et]=i, piv[et+1]=1, si no, piv[et]=i, piv[et+1]=j, donde i es la fila del pivote ij en la que aparece el elemento en la diagonal principal
de mayor valor absoluto.
@param giros puntero al vector de giros
@param tol tolerancia para definir el cero numérico que determina el fin de la factorización
@param et Etapa de la factorización en la que se está.
 */
void ldlgensimmatriziteracionsA(double *sA, unsigned long esA, unsigned long ldsA, unsigned long int *piv, double *giros, unsigned long et)
{
	double t, aki, akj, aij, temp;
	simmmatrizintercambiarfilacolummna(sA, esA, ldsA, et, piv[0]+et);
	simmmatrizintercambiarfilacolummna(sA, esA, ldsA, et+1, piv[1]+et+1);
	/*printf("\nA tras cambio fila columna:\n");
	simmatrizatexto(stdout,sA,esA,ldsA,7);*/

	/*calculo de  t!!*/
	aki=sA[INDICEARRAYDEFILACOLUMNA(ldsA, et, et)]; /*aki=aii*/
	akj=sA[INDICEARRAYDEFILACOLUMNA(ldsA, et+1, et+1)]; /*akj=ajj*/
	aij=sA[INDICEARRAYDEFILACOLUMNA(ldsA, et, et+1)];

	/*Es posible que akj=aki y aij=0, en estas condiciones el giro queda sin definir, y, además, no tiene sentido*/
	temp=0.5*(akj-aki)/aij;
	if (fabs(temp)<1.)
	{
		if (temp>0.)
			t=1./(temp+sqrt(temp*temp+1.));
		else
		{
			if (temp<0.)
				t= -1./(sqrt(temp*temp+1.)-temp);
			else /*parece tonteria, pero el caso de temp=0 es peligroso, porque el signo tiene que venir determinado por que el valor gordo tiene que quedar en el primer elemento*/
			{
				if (aki<0.)
					t=1.;
				else
					t= -1.;
			}
		}
	}
	else
	{
		temp=2.*aij/(akj-aki);
		if (temp>0.)
			t=temp/(1.+sqrt(temp*temp+1.));
		else
		{
			if (temp<0.)
			{
				t=temp/(1.+sqrt(temp*temp+1.));
			}
			else
			{
				t=0.;
			}
		}
	}
	simmatrizrotacion(sA, esA, ldsA, et, et+1, t);

	/*printf("Giro:%g, et:%d\n",t,et);*/

	/*printf("Matriz girada:\n");
	simmatrizatexto(stdout,sA,esA,ldsA,5);*/

	ldlgensimmatrizetapareduccionsA(&(sA[INDICEARRAYDEFILACOLUMNA(ldsA, et, et)]), esA-et, ldsA-et);
	giros[0]=t;

	/*printf("Matriz reducida:\n");
	simmatrizatexto(stdout,sA,esA,ldsA,5);*/
}

/**Realiza una factorización completa de la matriz simétrica por el método de rotación (algoritmo no recursivo) con pivotaje completo
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param rnsA en salida, rango de la matriz A
@param piv array de pivotes. Tiene por dimension la dimensión de la submatriz multiplicada por dos. Son cambios sucesivos, es decir,
la fila/columna a cambiar se permuta por la que esta piv veces a la derecha/abajo. Si estamos en la columna i y el pivote es j se
cambia la i por la j+i. Si el elemento de mayor valor absoluto está en la diagonal principal, piv[0]=i, piv[1]=0, si no, piv[0]=i,
piv[1]=j, donde i es la fila del pivote ij en la que aparece el elemento en la diagonal principal
de mayor valor absoluto.
@param giros puntero al vector de giros
@param tol tolerancia para definir el cero numérico que determina el fin de la factorización
 */
void ldlgensimmatrizfactptsA(double *sA, unsigned long esA, unsigned long ldsA, unsigned long *rnsA, unsigned long int *piv, double *giros, double tol)
{
	unsigned long int i, f, c;
	double pivote;
	if (esA<=1)
	{
		*rnsA=0;
		return;
	}
	/*printf("Matriz inicial:\n");
	simmatrizatexto(stdout,sA,esA,ldsA,5);*/
	for(i=0; i<esA-1; i++)
	{
		pivote=pivotajecompletosimmatriz(&sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)], esA-i, ldsA-i, &f, &c);
		if (pivote<tol)
		{
			*rnsA=i;
			return;
		}
		/*Buscamos que orden nos va mejor*/
		if (f==c) /*Ojo, esto con Rook es distinto*/
		{
			piv[2*i]=f;
			piv[2*i+1]=0; /*si el pivote esta en
					 * la diagonal ppal, la
					 * segunda fila/columna no se mueve*/
		}
		else
		{
			if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, f+i, f+i)])>=fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, c+i, c+i)]))
			{
				piv[2*i]=f;
				/*esta mierda de abajo es por los cambios sucesivos. Si el segundo cambio se hace sobre la primera,
				 * como la hemos cambiado antes, es cagadita.*/
				if (c==0)
					piv[2*i+1]=f-1;
				else
					piv[2*i+1]=c-1;
			}
			else
			{
				piv[2*i]=c;
				if (f==0)
					piv[2*i+1]=c-1;
				else
					piv[2*i+1]=f-1;
			}
		}
		ldlgensimmatriziteracionsA(sA, esA, ldsA, &(piv[2*i]), &(giros[i]), i);
	}
	if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)])>tol)
		i++;
	*rnsA=i;
	return;
}

/**Realiza una factorización completa de la matriz simétrica por el método de rotación (algoritmo no recursivo) con pivotaje de rook
@param sA puntero a la matriz simétrica
@param esA dimensión de la matriz (numero de filas/columnas)
@param ldsA dimensión fundamental de la matriz
@param rnsA en salida, rango de la matriz A
@param piv array de pivotes. Tiene por dimension la dimensión de la submatriz multiplicada por dos. Son cambios sucesivos, es decir,
la fila/columna a cambiar se permuta por la que esta piv veces a la derecha/abajo. Si estamos en la columna i y el pivote es j se
cambia la i por la j+i. Si el elemento de mayor valor absoluto está en la diagonal principal, piv[0]=i, piv[1]=0, si no, piv[0]=i,
piv[1]=j, donde i es la fila del pivote ij en la que aparece el elemento en la diagonal principal
de mayor valor absoluto.
@param giros puntero al vector de giros
@param tol tolerancia para definir el cero numérico que determina el fin de la factorización
 */
void ldlgensimmatrizfactrooksA(double *sA, unsigned long esA, unsigned long ldsA, unsigned long *rnsA, unsigned long int *piv, double *giros, double tol)
{
	unsigned long int i, f, c,f2;
	double pivote;
	if (esA<=1)
	{
		*rnsA=0;
		return;
	}
	for(i=0; i<esA-1; i++)
	{
		pivote=pivotajerooksimmatriz(&sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)], esA-i, ldsA-i, &f, &c,&f2, tol);

		/*printf("\nA en paso intermedio vale:\n");
		simmatrizatexto(stdout,sA,esA,ldsA,7);*/

		if (pivote<tol)
		{
			*rnsA=i;
			return;
		}
		/*Buscamos que orden nos va mejor*/
		if (f==c) /*Si son iguales, utilizar la segunda fila guardada*/
		{
			piv[2*i]=f;
			if (f2==0)
				piv[2*i+1]=f-1;
			else
				piv[2*i+1]=f2-1;
		}
		else
		{
			if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, f+i, f+i)])>=fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, c+i, c+i)]))
			{
				piv[2*i]=f;
				/*esta mierda de abajo es por los cambios sucesivos. Si el segundo cambio se hace sobre la primera,
				 * como la hemos cambiado antes, es cagadita.*/
				if (c==0)
					piv[2*i+1]=f-1;
				else
					piv[2*i+1]=c-1;
			}
			else
			{
				piv[2*i]=c;
				if (f==0)
					piv[2*i+1]=c-1;
				else
					piv[2*i+1]=f-1;
			}
		}
		ldlgensimmatriziteracionsA(sA, esA, ldsA, &(piv[2*i]), &(giros[i]), i);
		
		/*printf("pivotes empleados rook2:%f,%ld,%ld,%ld\n",fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)]),f,c,f2);*/
		
	}
	if (fabs(sA[INDICEARRAYDEFILACOLUMNA(ldsA, i, i)])>tol)
		i++;
	*rnsA=i;
	
	/*printf("\nA al final:\n");
	simmatrizatexto(stdout,sA,esA,ldsA,7);*/
	
	return;
}

/**Calcula la parte superior del subespacio nulo de la matriz factorizada (L'11^-1 L'21)
 y lo almacena en la parte L_21 de la propia matriz, de manera que una vez actualizada la matriz
|-L'21| 
|  I |
Es el subespacio nulo de la matriz.
@param sA puntero a la matriz simétrica factorizada
@param esA orden de la matriz A
@param ldsA dimensión fundamental de A
@param rnsA rango de la matriz factorizada*/
void ldlgensimmatrizobtenersubespacionulosA(double *sA, unsigned long int esA, unsigned long int ldsA, unsigned long int rnsA)
{
	unsigned long int i, j, k;
	if (rnsA==0||rnsA==esA)
		return;
	for(k=rnsA-1; k>0; k--)
		for(j=0; j<k; j++)
			for(i=0; i<esA-rnsA; i++)
			{
				sA[INDICEFILACOLUMNAL12(ldsA, rnsA, j, i)]-=sA[INDICEFILACOLUMNAL11(ldsA, j, k)]*sA[INDICEFILACOLUMNAL12(ldsA, rnsA, k, i)];
			}
}

/**Premultiplica el vector v por n matrices de pivotaje y de giro traspuestas en orden ascendente
@param giros puntero al vector de permutaciones sucesivas
@param piv puntero al vector de permutaciones sucesivas
@param numero de permutaciones y giros aplicadas al vector v
@param v vector a multiplicar*/
void vectorGtPtv(double *giros, unsigned long int *piv, unsigned long ncambios, double *v)
{
	unsigned long int i;
	double a,b,giro;
	for (i=0;i<ncambios;i++)
	{
            /*Primera permutacion*/
            a=v[i];
            v[i]=v[piv[2*i]+i];
            v[piv[2*i]+i]=a;
            /*Segunda permutacion*/
            a=v[i+1];
            v[i+1]=v[piv[2*i+1]+i+1];
            v[piv[2*i+1]+i+1]=a;
            /*giro*/
            giro=giros[i];
            b=1./(sqrt(giro*giro+1.));
            a=v[i];
            v[i]=(v[i]-v[i+1]*giro)*b;
            v[i+1]=(a*giro+v[i+1])*b;
	}
}

/**Premultiplica el vector v por n matrices de giro y de pivotaje en orden descendiente
@param giros puntero al vector de permutaciones sucesivas
@param piv puntero al vector de permutaciones sucesivas
@param numero de permutaciones y giros aplicadas al vector v
@param v vector a multiplicar*/
void vectorPGv(double *giros, unsigned long int *piv, unsigned long ncambios, double *v)
{
	unsigned long int i;
	double a,b,giro;
	for (i=ncambios;i>0;i--)
	{
            /*giro*/
			giro=giros[i-1];
			b=1/(sqrt(giro*giro+1.));
            a=v[i-1];
            v[i-1]=(v[i-1]+v[i]*giro)*b;
            v[i]=(-a*giro+v[i])*b;
            /*Segunda permutacion*/
            a=v[i];
            v[i]=v[piv[2*(i-1)+1]+i];
            v[piv[2*(i-1)+1]+i]=a;
            /*Primera permutacion*/
            a=v[i-1];
            v[i-1]=v[piv[2*(i-1)]+i-1];
            v[piv[2*(i-1)]+i-1]=a;
	}
}

/**Suma al vector destino, u, el escalar a por la matriz -N1 en la matriz sA
 * (factorizada ldl con subespacio nulo calculado) por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param sA matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param esA numero de filas de la matriz
@param ldsA dimension fundamental de A
@param rnsA rango de la matriz
@param v vector*/
void vectorsumaraldlgenN1sAv(double *u, double a, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rnsA, double *v)
{
    unsigned long i,j;
    for (i=0;i<rnsA;i++)
            for (j=0;j<(esA-rnsA);j++)
                    u[i]+=a*sA[INDICEARRAYDEFILACOLUMNA(ldsA,i,rnsA+j)]*v[j];
}

/**Suma al vector destino, u, el escalar a por la matriz -N1' en la matriz sA
 * (factorizada ldl con subespacio nulo calculado) por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param sA matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param esA numero de filas de la matriz
@param ldsA dimension fundamental de A
@param rnsA rango de la matriz
@param v vector*/
void vectorsumaraldlgenN1tsAv(double *u, double a, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rnsA, double *v)
{
    unsigned long i,j;
    for (i=0;i<rnsA;i++)
            for (j=0;j<(esA-rnsA);j++)
                    u[j]+=a*sA[INDICEARRAYDEFILACOLUMNA(ldsA,i,rnsA+j)]*v[i];
}

/**Guarda en el vector destino, u, el escalar a por la matriz -N1' en la matriz sA
 * (factorizada ldl con subespacio nulo calculado) por el vector v
@param u vector donde se quiere obtener el resultado.
@param a escalar por el que se quiere multiplicar la operacion al sumar
@param sA matriz factorizada ldu y con la parte del subespacio nulo izquierdo calculado
@param esA numero de filas de la matriz
@param ldsA dimension fundamental de A
@param rnsA rango de la matriz
@param v vector*/
void vectoraldlgenN1tsAv(double *u, double a, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rnsA, double *v)
{
    unsigned long i,j;
    for (j=0;j<(esA-rnsA);j++)
        u[j]=a*sA[INDICEARRAYDEFILACOLUMNA(ldsA,0,rnsA+j)]*v[0]; 
    for (i=1;i<rnsA;i++)
            for (j=0;j<(esA-rnsA);j++)
                u[j]+=a*sA[INDICEARRAYDEFILACOLUMNA(ldsA,i,rnsA+j)]*v[i];
}

/*simmatriz(rxr)=(I+N1(rx(esa-r)*N1'((esa-r)xr)). Guarda en simmatriz la matriz (I+N1*N1t)
 * @param sA matriz. Debe estar ya factorizada con la funcion ldlgenfactrookcandidata
 * @param rnsA rango de la matriz A
 * @param esA número de filas de la matriz
 * @param ldsA dimensión fundamental de la matriz A
 * @param simmatriz puntero a la matriz (I+N1*N1')
 */
void simmatrizldlgenImasN1N1tsA(double *sA, unsigned long rnsA, unsigned long esA, unsigned long ldsA, double *simmatriz)
{
    unsigned long int i,j,k;
    /*Calculamos la matriz simetrica*/
    for (i=0;i<rnsA;i++)
        for (j=i;j<rnsA;j++)
        {
            if (i==j)
                simmatriz[INDICEARRAYDEFILACOLUMNA(rnsA,i,j)]=1+sA[INDICEARRAYDEFILACOLUMNA(ldsA,i,rnsA)]*sA[INDICEARRAYDEFILACOLUMNA(ldsA,j,rnsA)];
            else
                simmatriz[INDICEARRAYDEFILACOLUMNA(rnsA,i,j)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA,i,rnsA)]*sA[INDICEARRAYDEFILACOLUMNA(ldsA,j,rnsA)];
            for (k=1;k<(esA-rnsA);k++)
                simmatriz[INDICEARRAYDEFILACOLUMNA(rnsA,i,j)]+=sA[INDICEARRAYDEFILACOLUMNA(ldsA,i,rnsA+k)]*sA[INDICEARRAYDEFILACOLUMNA(ldsA,j,rnsA+k)];
        } 
}

/*simmatriz((esa-r)x(esa-r))=(N1'((esa-r)xr)*N1(rx(esa-r)+I). Guarda en simmatriz la matriz (N1'N1+I)
 * @param sA matriz. Debe estar ya factorizada con la funcion ldlgenfactrookcandidata
 * @param rnsA rango de la matriz sA
 * @param esA número de filas de la matriz
 * @param ldsA dimensión fundamental de la matriz sA
 * @param simmatriz puntero a la matriz (N1'N1+I)
 */
void simmatrizldlgenN1tN1masIsA(double *sA, unsigned long rnsA, unsigned long esA, unsigned long ldsA, double *simmatriz)
{
    unsigned long int i,j,k;
    /*Calculamos la matriz simetrica*/
    for (j=0;j<(esA-rnsA);j++)
    {
        simmatriz[INDICEARRAYDEFILACOLUMNA(esA-rnsA,j,j)]=1+sA[INDICEARRAYDEFILACOLUMNA(ldsA,0,rnsA+j)]*sA[INDICEARRAYDEFILACOLUMNA(ldsA,0,rnsA+j)];
        for (k=(j+1);k<(esA-rnsA);k++)
            simmatriz[INDICEARRAYDEFILACOLUMNA(esA-rnsA,j,k)]=sA[INDICEARRAYDEFILACOLUMNA(ldsA,0,rnsA+j)]*sA[INDICEARRAYDEFILACOLUMNA(ldsA,0,rnsA+k)];      
    }
    for (i=1;i<rnsA;i++)
        for (j=0;j<(esA-rnsA);j++)
            for (k=j;k<(esA-rnsA);k++)
                simmatriz[INDICEARRAYDEFILACOLUMNA(esA-rnsA,j,k)]+=sA[INDICEARRAYDEFILACOLUMNA(ldsA,i,rnsA+j)]*sA[INDICEARRAYDEFILACOLUMNA(ldsA,i,rnsA+k)];
}

/**Resuelve el sistema Au=v, dando solución de norma mínima de mínimos
 * cuadrados, a partir de una matriz simetrica ya factorizada y con subespacio nulo
 * calculado. 
 * @param u vector solución. puede ser igual a v
 * @param sA matriz. Debe estar ya factorizada con la funcion ldlgenfactrookcandidata y ldlmatrizobtenersubespacionulo
 * @param esA número de filas de la matriz
 * @param ldsA dimensión fundamental de la matriz A
 * @param rnsA rango de la matriz A
 * @param piv vector de pivotaje (dimensión rnsA*2 si r<m, y 2*(esA-1) si r=m)
 * @param giros puntero al vector de giros (dimensión rnsA si r<m, y (esA-1) si r=m)
 * @param v vector de independientes
 * @param simmatriz puntero a la matriz simétrica
 */ 
void vectorpginvldlgensAv(double *u, double *sA, unsigned long esA, unsigned long ldsA, unsigned long rnsA, unsigned long int *piv, double *giros, double *v, double*simmatriz)
{
	unsigned long int i;
            
        if (rnsA==esA)
        {
            vectorGtPtv(giros, piv, rnsA-1, v);
            vectorldlinvAv(v, sA, esA, ldsA, v);
            vectorPGv(giros, piv, rnsA-1, v);
            return;
        }

        vectorGtPtv(giros, piv, rnsA, v);
	/*CASO r<esA-r*/
	if (rnsA<(esA-rnsA))
	{         
            vectorsumaraldlgenN1sAv(v, 1., sA, esA, ldsA, rnsA, &v[rnsA]);          
            simmatrizldlgenImasN1N1tsA(sA, rnsA, esA, ldsA, simmatriz);
            /*Factorizacion con Cholesky de simmatriz*/
            ldlsimmatrizfactsA(simmatriz, rnsA, rnsA); 
            /*Resolvemos el problema*/
            vectorldlinvAv(v, simmatriz, rnsA, rnsA, v);
	}
        /*CASO r>=esA-r*/
	else
	{
            vectorsumaraldlgenN1tsAv(&v[rnsA], -1, sA, esA, ldsA, rnsA, v);
            /*cambiamos signo*/
            for (i=rnsA;i<esA;i++)
                    u[i]*=-1.;
            simmatrizldlgenN1tN1masIsA(sA, rnsA, esA, ldsA, simmatriz);
            /*Factorizacion con Cholesky de simmatriz*/
            ldlsimmatrizfactsA(simmatriz, (esA-rnsA), (esA-rnsA)); 
            /*Resolvemos el problema*/
            vectorldlinvAv(&v[rnsA], simmatriz, (esA-rnsA), (esA-rnsA), &v[rnsA]);
            //v1masN1v2(sA, rnsA, esA, ldsA, v);
            vectorsumaraldlgenN1sAv(v, -1., sA, esA, ldsA, rnsA, &v[rnsA]);
	}
	/*premultiplicamos v1 con la inversa de L11. Tenemos z*/
	vectorldlinvLsAv(v,sA,rnsA,ldsA,v);
        /*premultiplicamos v1 por la inversa de D11*/
	vectorldlinvDsAv(v,sA,rnsA,ldsA,v);
        /*premultiplicamos v1 por la inversa de L11 traspuesta*/
        vectorldlinvLTsAv(v,sA,rnsA,ldsA,v);
        
        if (rnsA<(esA-rnsA))
	{  
            /*Obtenemos w1. v1=invsimmatriz*v1. Resolvemos con Cholesky el problema*/
            vectorldlinvAv(v, simmatriz, rnsA, rnsA, v);
            /*Obtenemos w2*/
            vectoraldlgenN1tsAv(&v[rnsA], 1, sA, esA, ldsA, rnsA, v);
        }
        else
        {
            vectoraldlgenN1tsAv(&v[rnsA], 1, sA, esA, ldsA, rnsA, v);
            vectorldlinvAv(&v[rnsA], simmatriz, (esA-rnsA), (esA-rnsA), &v[rnsA]);
            vectorsumaraldlgenN1sAv(v, -1., sA, esA, ldsA, rnsA, &v[rnsA]);
        }	
	/*Cambio de columnas para obtener el vector solucion*/
        vectorPGv(giros, piv, rnsA, v);	
}






